package agent

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"

	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/module/modagent"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/tool/errz"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/tool/ioz"
	"gitlab.com/gitlab-org/cluster-integration/gitlab-agent/v16/internal/tool/logz"
	"go.uber.org/zap"
)

type Reporter struct {
	log *zap.Logger
	api modagent.API
}

type uuidResponse struct {
	UUID string `json:"uuid"`
}

type reqBody struct {
	UUIDs []string `json:"uuids"`
}

func NewReporter(log *zap.Logger, api modagent.API) *Reporter {
	return &Reporter{
		log: log,
		api: api,
	}
}

func (r *Reporter) Transmit(ctx context.Context, payloads []*Payload) ([]string, error) {
	uuids := make([]string, 0, len(payloads))

	// If at least one of the detected vulnerabilities fails to create, don't
	// resolve vulnerabilities. Potentially all detected vulnerabilities might
	// fail to create, and in turn all previously detected vulnerabilities would
	// get resolved otherwise.
	var errs []error

	r.log.Info("Creating vulnerabilities in GitLab", logz.VulnerabilitiesCount(len(payloads)))
	for _, payload := range payloads {
		uuid, err := r.createVulnerability(ctx, payload)
		if err != nil {
			errs = append(errs, err)
			continue
		}
		uuids = append(uuids, uuid)
	}

	if len(errs) > 0 {
		r.log.Warn("Some vulnerabilities failed to create, skipping vulnerability resolution")
		return nil, errors.Join(errs...)
	}
	if len(uuids) == 0 {
		r.log.Debug("No UUIDs collected, nothing to resolve")
		return nil, nil
	}

	return uuids, nil
}

func (r *Reporter) createVulnerability(ctx context.Context, payload *Payload) (uuid string, retError error) {
	resp, err := r.api.MakeGitLabRequest(ctx, "/",
		modagent.WithRequestMethod(http.MethodPut),
		modagent.WithJSONRequestBody(payload),
	)
	if err != nil {
		return "", fmt.Errorf("error making api request: %w", err)
	}
	defer errz.SafeClose(resp.Body, &retError)
	if resp.StatusCode != http.StatusOK {
		_ = ioz.DiscardData(resp.Body)
		return "", fmt.Errorf("unexpected status code: %d", resp.StatusCode)
	}
	var uuidResp uuidResponse
	body, err := io.ReadAll(resp.Body)
	if err != nil {
		return "", fmt.Errorf("error reading response body: %w", err)
	}
	err = json.Unmarshal(body, &uuidResp)
	if err != nil {
		return "", fmt.Errorf("error parsing response body: %w", err)
	}
	return uuidResp.UUID, nil
}

func (r *Reporter) ResolveVulnerabilities(ctx context.Context, uuids []string) (retError error) {
	body := reqBody{
		UUIDs: uuids,
	}
	resp, err := r.api.MakeGitLabRequest(ctx, "/scan_result",
		modagent.WithRequestMethod(http.MethodPost),
		modagent.WithJSONRequestBody(body),
	)
	if err != nil {
		return fmt.Errorf("error making api request: %w", err)
	}
	defer errz.DiscardAndClose(resp.Body, &retError)
	if resp.StatusCode != http.StatusOK {
		return fmt.Errorf("unexpected status code: %d", resp.StatusCode)
	}
	return nil
}
