# -*- coding: utf-8 -*-
# Copyright (c) 2022, Felix Fontein (@felixfontein) <felix@fontein.de>
# GNU General Public License v3.0+ (see LICENSES/GPL-3.0-or-later.txt or https://www.gnu.org/licenses/gpl-3.0.txt)
# SPDX-License-Identifier: GPL-3.0-or-later

# The data inside here is private to this collection. If you use this from outside the collection,
# you are on your own. There can be random changes to its format even in bugfix releases!

from __future__ import absolute_import, division, print_function
__metaclass__ = type

from ansible_collections.community.routeros.plugins.module_utils.version import LooseVersion


def _compare(a, b, comparator):
    if comparator == '==':
        return a == b
    if comparator == '!=':
        return a != b
    if comparator == '<':
        return a < b
    if comparator == '<=':
        return a <= b
    if comparator == '>':
        return a > b
    if comparator == '>=':
        return a >= b
    raise ValueError('Unknown comparator "{comparator}"'.format(comparator=comparator))


class APIData(object):
    def __init__(self,
                 unversioned=None,
                 versioned=None):
        if (unversioned is None) == (versioned is None):
            raise ValueError('either unversioned or versioned must be provided')
        self.unversioned = unversioned
        self.versioned = versioned
        if self.unversioned is not None:
            self.needs_version = self.unversioned.needs_version
            self.fully_understood = self.unversioned.fully_understood
        else:
            self.needs_version = self.versioned is not None
            # Mark as 'fully understood' if it is for at least one version
            self.fully_understood = False
            for dummy, dummy, unversioned in self.versioned:
                if unversioned and not isinstance(unversioned, str) and unversioned.fully_understood:
                    self.fully_understood = True
                    break
        self._current = None if self.needs_version else self.unversioned

    def _select(self, data, api_version):
        if data is None:
            self._current = None
            return False, None
        if isinstance(data, str):
            self._current = None
            return False, data
        self._current = data.specialize_for_version(api_version)
        return self._current.fully_understood, None

    def provide_version(self, version):
        if not self.needs_version:
            return self.unversioned.fully_understood, None
        api_version = LooseVersion(version)
        if self.unversioned is not None:
            self._current = self.unversioned.specialize_for_version(api_version)
            return self._current.fully_understood, None
        for other_version, comparator, data in self.versioned:
            if other_version == '*' and comparator == '*':
                return self._select(data, api_version)
            other_api_version = LooseVersion(other_version)
            if _compare(api_version, other_api_version, comparator):
                return self._select(data, api_version)
        self._current = None
        return False, None

    def get_data(self):
        if self._current is None:
            raise ValueError('either provide_version() was not called or it returned False')
        return self._current


class VersionedAPIData(object):
    def __init__(self,
                 primary_keys=None,
                 stratify_keys=None,
                 required_one_of=None,
                 mutually_exclusive=None,
                 has_identifier=False,
                 single_value=False,
                 unknown_mechanism=False,
                 fully_understood=False,
                 fixed_entries=False,
                 fields=None,
                 versioned_fields=None):
        if sum([primary_keys is not None, stratify_keys is not None, has_identifier, single_value, unknown_mechanism]) > 1:
            raise ValueError('primary_keys, stratify_keys, has_identifier, single_value, and unknown_mechanism are mutually exclusive')
        if unknown_mechanism and fully_understood:
            raise ValueError('unknown_mechanism and fully_understood cannot be combined')
        self.primary_keys = primary_keys
        self.stratify_keys = stratify_keys
        self.required_one_of = required_one_of or []
        self.mutually_exclusive = mutually_exclusive or []
        self.has_identifier = has_identifier
        self.single_value = single_value
        self.unknown_mechanism = unknown_mechanism
        self.fully_understood = fully_understood
        self.fixed_entries = fixed_entries
        if fixed_entries and primary_keys is None:
            raise ValueError('fixed_entries can only be used with primary_keys')
        if fields is None:
            raise ValueError('fields must be provided')
        self.fields = fields
        if versioned_fields is not None:
            if not isinstance(versioned_fields, list):
                raise ValueError('unversioned_fields must be a list')
            for conditions, name, field in versioned_fields:
                if not isinstance(conditions, (tuple, list)):
                    raise ValueError('conditions must be a list or tuple')
                if not isinstance(field, KeyInfo):
                    raise ValueError('field must be a KeyInfo object')
                if name in fields:
                    raise ValueError('"{name}" appears both in fields and versioned_fields'.format(name=name))
        self.versioned_fields = versioned_fields or []
        if primary_keys:
            for pk in primary_keys:
                if pk not in fields:
                    raise ValueError('Primary key {pk} must be in fields!'.format(pk=pk))
        if stratify_keys:
            for sk in stratify_keys:
                if sk not in fields:
                    raise ValueError('Stratify key {sk} must be in fields!'.format(sk=sk))
        if required_one_of:
            for index, require_list in enumerate(required_one_of):
                if not isinstance(require_list, list):
                    raise ValueError('Require one of element at index #{index} must be a list!'.format(index=index + 1))
                for rk in require_list:
                    if rk not in fields:
                        raise ValueError('Require one of key {rk} must be in fields!'.format(rk=rk))
        if mutually_exclusive:
            for index, exclusive_list in enumerate(mutually_exclusive):
                if not isinstance(exclusive_list, list):
                    raise ValueError('Mutually exclusive element at index #{index} must be a list!'.format(index=index + 1))
                for ek in exclusive_list:
                    if ek not in fields:
                        raise ValueError('Mutually exclusive key {ek} must be in fields!'.format(ek=ek))
        self.needs_version = len(self.versioned_fields) > 0

    def specialize_for_version(self, api_version):
        fields = self.fields.copy()
        for conditions, name, field in self.versioned_fields:
            matching = True
            for other_version, comparator in conditions:
                other_api_version = LooseVersion(other_version)
                if not _compare(api_version, other_api_version, comparator):
                    matching = False
                    break
            if matching:
                if name in fields:
                    raise ValueError(
                        'Internal error: field "{field}" already exists for {version}'.format(field=name, version=api_version)
                    )
                fields[name] = field
        return VersionedAPIData(
            primary_keys=self.primary_keys,
            stratify_keys=self.stratify_keys,
            required_one_of=self.required_one_of,
            mutually_exclusive=self.mutually_exclusive,
            has_identifier=self.has_identifier,
            single_value=self.single_value,
            unknown_mechanism=self.unknown_mechanism,
            fully_understood=self.fully_understood,
            fixed_entries=self.fixed_entries,
            fields=fields,
        )


class KeyInfo(object):
    def __init__(self,
                 _dummy=None,
                 can_disable=False,
                 remove_value=None,
                 absent_value=None,
                 default=None,
                 required=False,
                 automatically_computed_from=None,
                 read_only=False,
                 write_only=False):
        if _dummy is not None:
            raise ValueError('KeyInfo() does not have positional arguments')
        if sum([required, default is not None or can_disable, automatically_computed_from is not None]) > 1:
            raise ValueError(
                'required, default, automatically_computed_from, and can_disable are mutually exclusive '
                'besides default and can_disable which can be set together')
        if not can_disable and remove_value is not None:
            raise ValueError('remove_value can only be specified if can_disable=True')
        if absent_value is not None and any([default is not None, automatically_computed_from is not None, can_disable]):
            raise ValueError('absent_value can not be combined with default, automatically_computed_from, can_disable=True, or absent_value')
        if read_only and write_only:
            raise ValueError('read_only and write_only cannot be used at the same time')
        if read_only and any([can_disable, remove_value is not None, absent_value is not None, default is not None, required]):
            raise ValueError('read_only can not be combined with can_disable, remove_value, absent_value, default, or required')
        self.can_disable = can_disable
        self.remove_value = remove_value
        self.automatically_computed_from = automatically_computed_from
        self.default = default
        self.required = required
        self.absent_value = absent_value
        self.read_only = read_only
        self.write_only = write_only


def split_path(path):
    return path.split()


def join_path(path):
    return ' '.join(path)


# How to obtain this information:
# 1. Run `/export verbose` in the CLI;
# 2. All attributes listed there go into the `fields` list;
#    attributes which can have a `!` ahead should have `canDisable=True`
# 3. All bold attributes go into the `primary_keys` list -- this is not always true!

PATHS = {
    ('interface', '6to4'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'clamp-tcp-mss': KeyInfo(default=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'dont-fragment': KeyInfo(default=False),
                'dscp': KeyInfo(default='inherit'),
                'ipsec-secret': KeyInfo(can_disable=True),
                'keepalive': KeyInfo(default='10s,10', can_disable=True),
                'local-address': KeyInfo(default='0.0.0.0'),
                'mtu': KeyInfo(default='auto'),
                'name': KeyInfo(),
                'remote-address': KeyInfo(required=True),
            }
        ),
    ),
    ('interface', 'bonding'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'arp': KeyInfo(default='enabled'),
                'arp-interval': KeyInfo(default='100ms'),
                'arp-ip-targets': KeyInfo(default=''),
                'arp-timeout': KeyInfo(default='auto'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'down-delay': KeyInfo(default='0ms'),
                'forced-mac-address': KeyInfo(can_disable=True),
                'lacp-rate': KeyInfo(default='30secs'),
                'lacp-user-key': KeyInfo(can_disable=True, remove_value=0),
                'link-monitoring': KeyInfo(default='mii'),
                'mii-interval': KeyInfo(default='100ms'),
                'min-links': KeyInfo(default=0),
                'mlag-id': KeyInfo(can_disable=True, remove_value=0),
                'mode': KeyInfo(default='balance-rr'),
                'mtu': KeyInfo(default=1500),
                'name': KeyInfo(),
                'primary': KeyInfo(default='none'),
                'slaves': KeyInfo(required=True),
                'transmit-hash-policy': KeyInfo(default='layer-2'),
                'up-delay': KeyInfo(default='0ms'),
            }
        ),
    ),
    ('interface', 'bridge'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            versioned_fields=[
                ([('7.0', '<')], 'ingress-filtering', KeyInfo(default=False)),
                ([('7.0', '>=')], 'ingress-filtering', KeyInfo(default=True)),
                ([('7.13', '>=')], 'port-cost-mode', KeyInfo(default='long')),
                ([('7.16', '>=')], 'forward-reserved-addresses', KeyInfo(default=False)),
                ([('7.16', '>=')], 'max-learned-entries', KeyInfo(default='auto')),
            ],
            fields={
                'admin-mac': KeyInfo(default=''),
                'ageing-time': KeyInfo(default='5m'),
                'arp': KeyInfo(default='enabled'),
                'arp-timeout': KeyInfo(default='auto'),
                'auto-mac': KeyInfo(default=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'dhcp-snooping': KeyInfo(default=False),
                'disabled': KeyInfo(default=False),
                'ether-type': KeyInfo(default='0x8100'),
                'fast-forward': KeyInfo(default=True),
                'forward-delay': KeyInfo(default='15s'),
                'frame-types': KeyInfo(default='admit-all'),
                'igmp-snooping': KeyInfo(default=False),
                'max-message-age': KeyInfo(default='20s'),
                'mld-version': KeyInfo(default=1),
                'mtu': KeyInfo(default='auto'),
                'multicast-querier': KeyInfo(default=False),
                'name': KeyInfo(),
                'priority': KeyInfo(default='0x8000'),
                'protocol-mode': KeyInfo(default='rstp'),
                'pvid': KeyInfo(default=1),
                'transmit-hold-count': KeyInfo(default=6),
                'vlan-filtering': KeyInfo(default=False),
            },
        ),
    ),
    ('interface', 'eoip'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name',),
            fields={
                'allow-fast-path': KeyInfo(default=True),
                'arp': KeyInfo(default='enabled'),
                'arp-timeout': KeyInfo(default='auto'),
                'clamp-tcp-mss': KeyInfo(default=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'dont-fragment': KeyInfo(default=False),
                'dscp': KeyInfo(default='inherit'),
                'ipsec-secret': KeyInfo(can_disable=True),
                'keepalive': KeyInfo(default='10s,10', can_disable=True),
                'local-address': KeyInfo(default='0.0.0.0'),
                'loop-protect': KeyInfo(default='default'),
                'loop-protect-disable-time': KeyInfo(default='5m'),
                'loop-protect-send-interval': KeyInfo(default='5s'),
                'mac-address': KeyInfo(),
                'mtu': KeyInfo(default='auto'),
                'name': KeyInfo(),
                'remote-address': KeyInfo(required=True),
                'tunnel-id': KeyInfo(required=True),
            },
        ),
    ),
    ('interface', 'ethernet'): APIData(
        unversioned=VersionedAPIData(
            fixed_entries=True,
            fully_understood=True,
            primary_keys=('default-name', ),
            fields={
                'default-name': KeyInfo(),
                'advertise': KeyInfo(),
                'arp': KeyInfo(default='enabled'),
                'arp-timeout': KeyInfo(default='auto'),
                'auto-negotiation': KeyInfo(default=True),
                'bandwidth': KeyInfo(default='unlimited/unlimited'),
                'combo-mode': KeyInfo(can_disable=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'fec-mode': KeyInfo(can_disable=True, remove_value='auto'),
                'full-duplex': KeyInfo(default=True),
                'l2mtu': KeyInfo(),
                'loop-protect': KeyInfo(default='default'),
                'loop-protect-disable-time': KeyInfo(default='5m'),
                'loop-protect-send-interval': KeyInfo(default='5s'),
                'mac-address': KeyInfo(),
                'mdix-enable': KeyInfo(),
                'mtu': KeyInfo(default=1500),
                'name': KeyInfo(),
                'orig-mac-address': KeyInfo(),
                'poe-out': KeyInfo(can_disable=True, remove_value='auto-on'),
                'poe-priority': KeyInfo(can_disable=True, remove_value=10),
                'poe-voltage': KeyInfo(can_disable=True),
                'power-cycle-interval': KeyInfo(),
                'power-cycle-ping-address': KeyInfo(can_disable=True),
                'power-cycle-ping-enabled': KeyInfo(),
                'power-cycle-ping-timeout': KeyInfo(can_disable=True),
                'rx-flow-control': KeyInfo(default='off'),
                'sfp-rate-select': KeyInfo(default='high'),
                'sfp-shutdown-temperature': KeyInfo(default=95),
                'speed': KeyInfo(),
                'tx-flow-control': KeyInfo(default='off'),
            },
        ),
    ),
    ('interface', 'ethernet', 'poe'): APIData(
        unversioned=VersionedAPIData(
            fixed_entries=True,
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'name': KeyInfo(),
                'poe-out': KeyInfo(default='auto-on'),
                'poe-priority': KeyInfo(default=10),
                'poe-voltage': KeyInfo(default='auto'),
                'power-cycle-interval': KeyInfo(default='none'),
                'power-cycle-ping-address': KeyInfo(can_disable=True),
                'power-cycle-ping-enabled': KeyInfo(default=False),
                'power-cycle-ping-timeout': KeyInfo(can_disable=True),
            }
        ),
    ),
    ('interface', 'gre'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'allow-fast-path': KeyInfo(default=True),
                'clamp-tcp-mss': KeyInfo(default=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'dont-fragment': KeyInfo(default=False),
                'dscp': KeyInfo(default='inherit'),
                'ipsec-secret': KeyInfo(can_disable=True),
                'keepalive': KeyInfo(default='10s,10', can_disable=True),
                'local-address': KeyInfo(default='0.0.0.0'),
                'mtu': KeyInfo(default='auto'),
                'name': KeyInfo(),
                'remote-address': KeyInfo(required=True),
            },
        ),
    ),
    ('interface', 'gre6'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name',),
            fields={
                'clamp-tcp-mss': KeyInfo(default=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'dscp': KeyInfo(default='inherit'),
                'ipsec-secret': KeyInfo(can_disable=True),
                'keepalive': KeyInfo(default='10s,10', can_disable=True),
                'local-address': KeyInfo(default='::'),
                'mtu': KeyInfo(default='auto'),
                'name': KeyInfo(),
                'remote-address': KeyInfo(required=True),
            },
        ),
    ),
    ('interface', 'list'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'exclude': KeyInfo(),
                'include': KeyInfo(),
                'name': KeyInfo(),
            },
        ),
    ),
    ('interface', 'list', 'member'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('list', 'interface', ),
            fully_understood=True,
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'interface': KeyInfo(),
                'list': KeyInfo(),
                'disabled': KeyInfo(default=False),
            },
        ),
    ),
    ('interface', 'lte', 'apn'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'add-default-route': KeyInfo(),
                'apn': KeyInfo(),
                'default-route-distance': KeyInfo(),
                'name': KeyInfo(),
                'use-peer-dns': KeyInfo(),
            },
        ),
    ),
    ('interface', 'ppp-client'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'add-default-route': KeyInfo(default=True),
                'allow': KeyInfo(default='pap,chap,mschap1,mschap2'),
                'apn': KeyInfo(default='internet'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'data-channel': KeyInfo(default=0),
                'default-route-distance': KeyInfo(default=1),
                'dial-command': KeyInfo(default="ATDT"),
                'dial-on-demand': KeyInfo(default=True),
                'disabled': KeyInfo(default=True),
                'info-channel': KeyInfo(default=0),
                'keepalive-timeout': KeyInfo(default=30),
                'max-mru': KeyInfo(default=1500),
                'max-mtu': KeyInfo(default=1500),
                'modem-init': KeyInfo(default=''),
                'mrru': KeyInfo(default='disabled'),
                'name': KeyInfo(),
                'null-modem': KeyInfo(default=False),
                'password': KeyInfo(default=''),
                'phone': KeyInfo(default=''),
                'pin': KeyInfo(default=''),
                'port': KeyInfo(),
                'profile': KeyInfo(default='default'),
                'use-peer-dns': KeyInfo(default=True),
                'user': KeyInfo(default=''),
            },
        ),
    ),
    ('interface', 'pppoe-client'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'ac-name': KeyInfo(default=''),
                'add-default-route': KeyInfo(default=False),
                'allow': KeyInfo(default='pap,chap,mschap1,mschap2'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'default-route-distance': KeyInfo(default=1),
                'dial-on-demand': KeyInfo(default=False),
                'disabled': KeyInfo(default=True),
                'host-uniq': KeyInfo(can_disable=True),
                'interface': KeyInfo(required=True),
                'keepalive-timeout': KeyInfo(default=10),
                'max-mru': KeyInfo(default='auto'),
                'max-mtu': KeyInfo(default='auto'),
                'mrru': KeyInfo(default='disabled'),
                'name': KeyInfo(),
                'password': KeyInfo(default=''),
                'profile': KeyInfo(default='default'),
                'service-name': KeyInfo(default=''),
                'use-peer-dns': KeyInfo(default=False),
                'user': KeyInfo(default=''),
            },
        ),
    ),
    ('interface', 'vlan'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'arp': KeyInfo(default='enabled'),
                'arp-timeout': KeyInfo(default='auto'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'interface': KeyInfo(required=True),
                'loop-protect': KeyInfo(default='default'),
                'loop-protect-disable-time': KeyInfo(default='5m'),
                'loop-protect-send-interval': KeyInfo(default='5s'),
                'mtu': KeyInfo(default=1500),
                'name': KeyInfo(),
                'use-service-tag': KeyInfo(default=False),
                'vlan-id': KeyInfo(required=True),
            },
        ),
    ),
    ('interface', 'vrrp'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'arp': KeyInfo(default='enabled'),
                'arp-timeout': KeyInfo(default='auto'),
                'authentication': KeyInfo(default='none'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'group-master': KeyInfo(default=''),
                'interface': KeyInfo(required=True),
                'interval': KeyInfo(default='1s'),
                'mtu': KeyInfo(default=1500),
                'name': KeyInfo(),
                'on-backup': KeyInfo(default=''),
                'on-fail': KeyInfo(default=''),
                'on-master': KeyInfo(default=''),
                'password': KeyInfo(default=''),
                'preemption-mode': KeyInfo(default=True),
                'priority': KeyInfo(default=100),
                'remote-address': KeyInfo(),
                'sync-connection-tracking': KeyInfo(default=False),
                'v3-protocol': KeyInfo(default='ipv4'),
                'version': KeyInfo(default=3),
                'vrid': KeyInfo(default=1),
            },
        ),
    ),
    ('ip', 'hotspot', 'profile'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'dns-name': KeyInfo(),
                'hotspot-address': KeyInfo(),
                'html-directory': KeyInfo(),
                'html-directory-override': KeyInfo(),
                'http-cookie-lifetime': KeyInfo(),
                'http-proxy': KeyInfo(),
                'login-by': KeyInfo(),
                'name': KeyInfo(),
                'rate-limit': KeyInfo(),
                'smtp-server': KeyInfo(),
                'split-user-domain': KeyInfo(),
                'use-radius': KeyInfo(),
            },
        ),
    ),
    ('ip', 'hotspot', 'user', 'profile'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'add-mac-cookie': KeyInfo(),
                'address-list': KeyInfo(),
                'idle-timeout': KeyInfo(),
                'insert-queue-before': KeyInfo(can_disable=True),
                'keepalive-timeout': KeyInfo(),
                'mac-cookie-timeout': KeyInfo(),
                'name': KeyInfo(),
                'parent-queue': KeyInfo(can_disable=True),
                'queue-type': KeyInfo(can_disable=True),
                'shared-users': KeyInfo(),
                'status-autorefresh': KeyInfo(),
                'transparent-proxy': KeyInfo(),
            },
        ),
    ),
    ('ip', 'ipsec', 'identity'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('peer', ),
            fields={
                'auth-method': KeyInfo(default='pre-shared-key'),
                'certificate': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'eap-methods': KeyInfo(default='eap-tls'),
                'generate-policy': KeyInfo(default=False),
                'key': KeyInfo(),
                'match-by': KeyInfo(can_disable=True, remove_value='remote-id'),
                'mode-config': KeyInfo(can_disable=True, remove_value='none'),
                'my-id': KeyInfo(can_disable=True, remove_value='auto'),
                'notrack-chain': KeyInfo(can_disable=True, remove_value=''),
                'password': KeyInfo(),
                'peer': KeyInfo(),
                'policy-template-group': KeyInfo(can_disable=True, remove_value='default'),
                'remote-certificate': KeyInfo(),
                'remote-id': KeyInfo(can_disable=True, remove_value='auto'),
                'remote-key': KeyInfo(),
                'secret': KeyInfo(default=''),
                'username': KeyInfo(),
            },
        ),
    ),
    ('ip', 'ipsec', 'mode-config'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'name': KeyInfo(),
                'responder': KeyInfo(),
                'use-responder-dns': KeyInfo(),
            },
        ),
    ),
    ('ip', 'ipsec', 'peer'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'address': KeyInfo(can_disable=True, remove_value=''),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'exchange-mode': KeyInfo(default='main'),
                'local-address': KeyInfo(can_disable=True, remove_value='0.0.0.0'),
                'name': KeyInfo(),
                'passive': KeyInfo(can_disable=True, remove_value=False),
                'port': KeyInfo(can_disable=True, remove_value=500),
                'profile': KeyInfo(default='default'),
                'send-initial-contact': KeyInfo(default=True),
            },
        ),
    ),
    ('ip', 'ipsec', 'policy', 'group'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'name': KeyInfo(),
            },
        ),
    ),
    ('ip', 'ipsec', 'profile'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'dh-group': KeyInfo(default='modp2048,modp1024'),
                'dpd-interval': KeyInfo(default='2m'),
                'dpd-maximum-failures': KeyInfo(default=5),
                'enc-algorithm': KeyInfo(default='aes-128,3des'),
                'hash-algorithm': KeyInfo(default='sha1'),
                'lifebytes': KeyInfo(can_disable=True, remove_value=0),
                'lifetime': KeyInfo(default='1d'),
                'name': KeyInfo(),
                'nat-traversal': KeyInfo(default=True),
                'prf-algorithm': KeyInfo(can_disable=True, remove_value='auto'),
                'proposal-check': KeyInfo(default='obey'),
            },
        ),
    ),
    ('ip', 'ipsec', 'proposal'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'auth-algorithms': KeyInfo(default='sha1'),
                'disabled': KeyInfo(default=False),
                'enc-algorithms': KeyInfo(default='aes-256-cbc,aes-192-cbc,aes-128-cbc'),
                'lifetime': KeyInfo(default='30m'),
                'name': KeyInfo(),
                'pfs-group': KeyInfo(default='modp1024'),
            },
        ),
    ),
    ('ip', 'pool'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'comment': KeyInfo(),
                'name': KeyInfo(),
                'next-pool': KeyInfo(),
                'ranges': KeyInfo(),
            },
        ),
    ),
    ('ip', 'route'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'blackhole': KeyInfo(can_disable=True),
                'check-gateway': KeyInfo(can_disable=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'distance': KeyInfo(),
                'dst-address': KeyInfo(),
                'gateway': KeyInfo(),
                'pref-src': KeyInfo(),
                'routing-table': KeyInfo(default='main'),
                'route-tag': KeyInfo(can_disable=True),
                'routing-mark': KeyInfo(can_disable=True),
                'scope': KeyInfo(),
                'suppress-hw-offload': KeyInfo(default=False),
                'target-scope': KeyInfo(),
                'type': KeyInfo(can_disable=True, remove_value='unicast'),
                'vrf-interface': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('ip', 'route', 'rule'): APIData(
        versioned=[
            ('7', '<', VersionedAPIData(
                fully_understood=True,
                fields={
                    'action': KeyInfo(default='lookup'),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'dst-address': KeyInfo(can_disable=True),
                    'interface': KeyInfo(can_disable=True),
                    'routing-mark': KeyInfo(can_disable=True),
                    'src-address': KeyInfo(can_disable=True),
                    'table': KeyInfo(default='main'),
                },
            )),
        ],
    ),
    ('ip', 'vrf'): APIData(
        versioned=[
            ('7', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                fields={
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'interfaces': KeyInfo(),
                    'name': KeyInfo(),
                },
            )),
        ]
    ),
    ('ip', 'route', 'vrf'): APIData(
        versioned=[
            ('7', '<', VersionedAPIData(
                fully_understood=True,
                primary_keys=('routing-mark', ),
                fields={
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'interfaces': KeyInfo(),
                    'routing-mark': KeyInfo(),
                },
            )),
        ],
    ),
    ('routing', 'filter'): APIData(
        versioned=[
            ('7', '<', VersionedAPIData(
                fully_understood=True,
                fields={
                    'action': KeyInfo(default='passthrough'),
                    'address-family': KeyInfo(can_disable=True),
                    'append-bgp-communities': KeyInfo(can_disable=True),
                    'append-route-targets': KeyInfo(can_disable=True),
                    'bgp-as-path': KeyInfo(can_disable=True),
                    'bgp-as-path-length': KeyInfo(can_disable=True),
                    'bgp-atomic-aggregate': KeyInfo(can_disable=True),
                    'bgp-communities': KeyInfo(can_disable=True),
                    'bgp-local-pref': KeyInfo(can_disable=True),
                    'bgp-med': KeyInfo(can_disable=True),
                    'bgp-origin': KeyInfo(can_disable=True),
                    'bgp-weight': KeyInfo(can_disable=True),
                    'chain': KeyInfo(required=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'distance': KeyInfo(can_disable=True),
                    'invert-match': KeyInfo(default=False),
                    'jump-target': KeyInfo(),
                    'locally-originated-bgp': KeyInfo(can_disable=True),
                    'match-chain': KeyInfo(can_disable=True),
                    'ospf-type': KeyInfo(can_disable=True),
                    'pref-src': KeyInfo(can_disable=True),
                    'prefix': KeyInfo(default='0.0.0.0/0'),
                    'prefix-length': KeyInfo(can_disable=True),
                    'protocol': KeyInfo(can_disable=True),
                    'route-comment': KeyInfo(can_disable=True),
                    'route-tag': KeyInfo(can_disable=True),
                    'route-targets': KeyInfo(can_disable=True),
                    'routing-mark': KeyInfo(can_disable=True),
                    'scope': KeyInfo(can_disable=True),
                    'set-bgp-communities': KeyInfo(can_disable=True),
                    'set-bgp-local-pref': KeyInfo(can_disable=True),
                    'set-bgp-med': KeyInfo(can_disable=True),
                    'set-bgp-prepend': KeyInfo(can_disable=True),
                    'set-bgp-prepend-path': KeyInfo(),
                    'set-bgp-weight': KeyInfo(can_disable=True),
                    'set-check-gateway': KeyInfo(can_disable=True),
                    'set-disabled': KeyInfo(can_disable=True),
                    'set-distance': KeyInfo(can_disable=True),
                    'set-in-nexthop': KeyInfo(can_disable=True),
                    'set-in-nexthop-direct': KeyInfo(can_disable=True),
                    'set-in-nexthop-ipv6': KeyInfo(can_disable=True),
                    'set-in-nexthop-linklocal': KeyInfo(can_disable=True),
                    'set-out-nexthop': KeyInfo(can_disable=True),
                    'set-out-nexthop-ipv6': KeyInfo(can_disable=True),
                    'set-out-nexthop-linklocal': KeyInfo(can_disable=True),
                    'set-pref-src': KeyInfo(can_disable=True),
                    'set-route-comment': KeyInfo(can_disable=True),
                    'set-route-tag': KeyInfo(can_disable=True),
                    'set-route-targets': KeyInfo(can_disable=True),
                    'set-routing-mark': KeyInfo(can_disable=True),
                    'set-scope': KeyInfo(can_disable=True),
                    'set-site-of-origin': KeyInfo(can_disable=True),
                    'set-target-scope': KeyInfo(can_disable=True),
                    'set-type': KeyInfo(can_disable=True),
                    'set-use-te-nexthop': KeyInfo(can_disable=True),
                    'site-of-origin': KeyInfo(can_disable=True),
                    'target-scope': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('routing', 'filter', 'num-list'): APIData(
        versioned=[
            ('7', '>=', VersionedAPIData(
                fully_understood=True,
                fields={
                    'list': KeyInfo(required=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(can_disable=True),
                    'range': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('routing', 'filter', 'rule'): APIData(
        versioned=[
            ('7', '>=', VersionedAPIData(
                fully_understood=True,
                fields={
                    'chain': KeyInfo(required=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(can_disable=True),
                    'rule': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('routing', 'filter', 'select-rule'): APIData(
        versioned=[
            ('7', '>=', VersionedAPIData(
                fully_understood=True,
                fields={
                    'chain': KeyInfo(required=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(can_disable=True),
                    'do-group-num': KeyInfo(can_disable=True),
                    'do-group-prfx': KeyInfo(can_disable=True),
                    'do-jump': KeyInfo(can_disable=True),
                    'do-select-num': KeyInfo(can_disable=True),
                    'do-select-prfx': KeyInfo(can_disable=True),
                    'do-take': KeyInfo(can_disable=True),
                    'do-where': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('routing', 'filter', 'community-list'): APIData(
        versioned=[
            ('7', '>=', VersionedAPIData(
                fully_understood=True,
                fields={
                    'list': KeyInfo(required=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(can_disable=True),
                    'communities': KeyInfo(can_disable=True),
                    'regexp': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('routing', 'ospf', 'instance'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'domain-id': KeyInfo(can_disable=True),
                'domain-tag': KeyInfo(can_disable=True),
                'in-filter-chain': KeyInfo(can_disable=True),
                'mpls-te-address': KeyInfo(can_disable=True),
                'mpls-te-area': KeyInfo(can_disable=True),
                'name': KeyInfo(),
                'originate-default': KeyInfo(can_disable=True),
                'out-filter-chain': KeyInfo(can_disable=True),
                'out-filter-select': KeyInfo(can_disable=True),
                'redistribute': KeyInfo(can_disable=True),
                'router-id': KeyInfo(default='main'),
                'routing-table': KeyInfo(can_disable=True),
                'use-dn': KeyInfo(can_disable=True),
                'version': KeyInfo(default=2),
                'vrf': KeyInfo(default='main'),
            },
        ),
    ),
    ('routing', 'ospf', 'area'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'area-id': KeyInfo(default='0.0.0.0'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'default-cost': KeyInfo(can_disable=True),
                'disabled': KeyInfo(default=False),
                'instance': KeyInfo(required=True),
                'name': KeyInfo(),
                'no-summaries': KeyInfo(can_disable=True),
                'nssa-translator': KeyInfo(can_disable=True),
                'type': KeyInfo(default='default'),
            },
        ),
    ),
    ('routing', 'ospf', 'area', 'range'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('area', 'prefix', ),
            fields={
                'advertise': KeyInfo(default=True),
                'area': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'cost': KeyInfo(can_disable=True),
                'disabled': KeyInfo(default=False),
                'prefix': KeyInfo(),
            },
        ),
    ),
    ('routing', 'ospf', 'interface-template'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'area': KeyInfo(required=True),
                'auth': KeyInfo(can_disable=True),
                'auth-id': KeyInfo(can_disable=True),
                'auth-key': KeyInfo(can_disable=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'cost': KeyInfo(default=1),
                'dead-interval': KeyInfo(default='40s'),
                'disabled': KeyInfo(default=False),
                'hello-interval': KeyInfo(default='10s'),
                'instance-id': KeyInfo(default=0),
                'interfaces': KeyInfo(can_disable=True),
                'networks': KeyInfo(can_disable=True),
                'passive': KeyInfo(can_disable=True),
                'prefix-list': KeyInfo(can_disable=True),
                'priority': KeyInfo(default=128),
                'retransmit-interval': KeyInfo(default='5s'),
                'transmit-delay': KeyInfo(default='1s'),
                'type': KeyInfo(default='broadcast'),
                'vlink-neighbor-id': KeyInfo(can_disable=True),
                'vlink-transit-area': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('routing', 'ospf', 'static-neighbor'): APIData(
        versioned=[
            ('7', '>=', VersionedAPIData(
                fully_understood=True,
                fields={
                    'address': KeyInfo(required=True),
                    'area': KeyInfo(required=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'instance-id': KeyInfo(default=0),
                    'poll-interval': KeyInfo(default='2m'),
                },
            )),
        ],
    ),
    ('routing', 'ospf-v3', 'instance'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'disabled': KeyInfo(),
                'distribute-default': KeyInfo(),
                'metric-bgp': KeyInfo(),
                'metric-connected': KeyInfo(),
                'metric-default': KeyInfo(),
                'metric-other-ospf': KeyInfo(),
                'metric-rip': KeyInfo(),
                'metric-static': KeyInfo(),
                'name': KeyInfo(),
                'redistribute-bgp': KeyInfo(),
                'redistribute-connected': KeyInfo(),
                'redistribute-other-ospf': KeyInfo(),
                'redistribute-rip': KeyInfo(),
                'redistribute-static': KeyInfo(),
                'router-id': KeyInfo(),
            },
        ),
    ),
    ('routing', 'ospf-v3', 'area'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'area-id': KeyInfo(),
                'disabled': KeyInfo(),
                'instance': KeyInfo(),
                'name': KeyInfo(),
                'type': KeyInfo(),
            },
        ),
    ),
    ('routing', 'pimsm', 'instance'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'afi': KeyInfo(default='ipv4'),
                'bsm-forward-back': KeyInfo(),
                'crp-advertise-contained': KeyInfo(),
                'disabled': KeyInfo(default=False),
                'name': KeyInfo(),
                'rp-hash-mask-length': KeyInfo(),
                'rp-static-override': KeyInfo(default=False),
                'ssm-range': KeyInfo(),
                'switch-to-spt': KeyInfo(default=True),
                'switch-to-spt-bytes': KeyInfo(default=0),
                'switch-to-spt-interval': KeyInfo(),
                'vrf': KeyInfo(default="main"),
            },
        ),
    ),
    ('routing', 'pimsm', 'interface-template'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'disabled': KeyInfo(default=False),
                'hello-delay': KeyInfo(default='5s'),
                'hello-period': KeyInfo(default='30s'),
                'instance': KeyInfo(required=True),
                'interfaces': KeyInfo(can_disable=True),
                'join-prune-period': KeyInfo(default='1m'),
                'join-tracking-support': KeyInfo(default=True),
                'override-interval': KeyInfo(default='2s500ms'),
                'priority': KeyInfo(default=1),
                'propagation-delay': KeyInfo(default='500ms'),
                'source-addresses': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('routing', 'rule'): APIData(
        versioned=[
            ('7', '>=', VersionedAPIData(
                fully_understood=True,
                fields={
                    'action': KeyInfo(can_disable=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'dst-address': KeyInfo(can_disable=True),
                    'interface': KeyInfo(can_disable=True),
                    'min-prefix': KeyInfo(can_disable=True),
                    'routing-mark': KeyInfo(can_disable=True),
                    'src-address': KeyInfo(can_disable=True),
                    'table': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('routing', 'table'): APIData(
        versioned=[
            ('7', '>=', VersionedAPIData(
                fully_understood=True,
                fields={
                    'name': KeyInfo(required=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'fib': KeyInfo(),
                },
            )),
        ],
    ),
    ('snmp', 'community'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'default': KeyInfo(read_only=True),
                'addresses': KeyInfo(default='::/0'),
                'authentication-password': KeyInfo(default=''),
                'authentication-protocol': KeyInfo(default='MD5'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'encryption-password': KeyInfo(default=''),
                'encryption-protocol': KeyInfo(default='DES'),
                'name': KeyInfo(required=True),
                'read-access': KeyInfo(default=True),
                'security': KeyInfo(default='none'),
                'write-access': KeyInfo(default=False),
            },
        ),
    ),
    ('caps-man', 'aaa'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'called-format': KeyInfo(default='mac:ssid'),
                'interim-update': KeyInfo(default='disabled'),
                'mac-caching': KeyInfo(default='disabled'),
                'mac-format': KeyInfo(default='XX:XX:XX:XX:XX:XX'),
                'mac-mode': KeyInfo(default='as-username'),
            },
        ),
    ),
    ('caps-man', 'access-list'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'action': KeyInfo(can_disable=True),
                'allow-signal-out-of-range': KeyInfo(can_disable=True),
                'ap-tx-limit': KeyInfo(can_disable=True),
                'client-to-client-forwarding': KeyInfo(can_disable=True),
                'client-tx-limit': KeyInfo(can_disable=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'interface': KeyInfo(can_disable=True),
                'mac-address': KeyInfo(can_disable=True),
                'mac-address-mask': KeyInfo(can_disable=True),
                'private-passphrase': KeyInfo(can_disable=True),
                'radius-accounting': KeyInfo(can_disable=True),
                'signal-range': KeyInfo(can_disable=True),
                'ssid-regexp': KeyInfo(),
                'time': KeyInfo(can_disable=True),
                'vlan-id': KeyInfo(can_disable=True),
                'vlan-mode': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('caps-man', 'channel'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'band': KeyInfo(can_disable=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'control-channel-width': KeyInfo(can_disable=True),
                'extension-channel': KeyInfo(can_disable=True),
                'frequency': KeyInfo(can_disable=True),
                'name': KeyInfo(),
                'reselect-interval': KeyInfo(can_disable=True),
                'save-selected': KeyInfo(can_disable=True),
                'secondary-frequency': KeyInfo(can_disable=True),
                'skip-dfs-channels': KeyInfo(can_disable=True),
                'tx-power': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('caps-man', 'configuration'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'channel': KeyInfo(can_disable=True),
                'channel.band': KeyInfo(can_disable=True),
                'channel.control-channel-width': KeyInfo(can_disable=True),
                'channel.extension-channel': KeyInfo(can_disable=True),
                'channel.frequency': KeyInfo(can_disable=True),
                'channel.reselect-interval': KeyInfo(can_disable=True),
                'channel.save-selected': KeyInfo(can_disable=True),
                'channel.secondary-frequency': KeyInfo(can_disable=True),
                'channel.skip-dfs-channels': KeyInfo(can_disable=True),
                'channel.tx-power': KeyInfo(can_disable=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'country': KeyInfo(can_disable=True),
                'datapath': KeyInfo(can_disable=True),
                'datapath.arp': KeyInfo(),
                'datapath.bridge': KeyInfo(can_disable=True),
                'datapath.bridge-cost': KeyInfo(can_disable=True),
                'datapath.bridge-horizon': KeyInfo(can_disable=True),
                'datapath.client-to-client-forwarding': KeyInfo(can_disable=True),
                'datapath.interface-list': KeyInfo(can_disable=True),
                'datapath.l2mtu': KeyInfo(),
                'datapath.local-forwarding': KeyInfo(can_disable=True),
                'datapath.mtu': KeyInfo(),
                'datapath.openflow-switch': KeyInfo(can_disable=True),
                'datapath.vlan-id': KeyInfo(can_disable=True),
                'datapath.vlan-mode': KeyInfo(can_disable=True),
                'disconnect-timeout': KeyInfo(can_disable=True),
                'distance': KeyInfo(can_disable=True),
                'frame-lifetime': KeyInfo(can_disable=True),
                'guard-interval': KeyInfo(can_disable=True),
                'hide-ssid': KeyInfo(can_disable=True),
                'hw-protection-mode': KeyInfo(can_disable=True),
                'hw-retries': KeyInfo(can_disable=True),
                'installation': KeyInfo(can_disable=True),
                'keepalive-frames': KeyInfo(can_disable=True),
                'load-balancing-group': KeyInfo(can_disable=True),
                'max-sta-count': KeyInfo(can_disable=True),
                'mode': KeyInfo(can_disable=True),
                'multicast-helper': KeyInfo(can_disable=True),
                'name': KeyInfo(),
                'rates': KeyInfo(can_disable=True),
                'rates.basic': KeyInfo(can_disable=True),
                'rates.ht-basic-mcs': KeyInfo(can_disable=True),
                'rates.ht-supported-mcs': KeyInfo(can_disable=True),
                'rates.supported': KeyInfo(can_disable=True),
                'rates.vht-basic-mcs': KeyInfo(can_disable=True),
                'rates.vht-supported-mcs': KeyInfo(can_disable=True),
                'rx-chains': KeyInfo(can_disable=True),
                'security': KeyInfo(can_disable=True),
                'security.authentication-types': KeyInfo(can_disable=True),
                'security.disable-pmkid': KeyInfo(can_disable=True),
                'security.eap-methods': KeyInfo(can_disable=True),
                'security.eap-radius-accounting': KeyInfo(can_disable=True),
                'security.encryption': KeyInfo(can_disable=True),
                'security.group-encryption': KeyInfo(can_disable=True),
                'security.group-key-update': KeyInfo(),
                'security.passphrase': KeyInfo(can_disable=True),
                'security.tls-certificate': KeyInfo(),
                'security.tls-mode': KeyInfo(),
                'ssid': KeyInfo(can_disable=True),
                'tx-chains': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('caps-man', 'datapath'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'arp': KeyInfo(),
                'bridge': KeyInfo(can_disable=True),
                'bridge-cost': KeyInfo(can_disable=True),
                'bridge-horizon': KeyInfo(can_disable=True),
                'client-to-client-forwarding': KeyInfo(can_disable=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'interface-list': KeyInfo(can_disable=True),
                'l2mtu': KeyInfo(),
                'local-forwarding': KeyInfo(can_disable=True),
                'mtu': KeyInfo(),
                'name': KeyInfo(),
                'openflow-switch': KeyInfo(can_disable=True),
                'vlan-id': KeyInfo(can_disable=True),
                'vlan-mode': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('caps-man', 'manager', 'interface'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('interface', ),
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'default': KeyInfo(),
                'disabled': KeyInfo(default=False),
                'forbid': KeyInfo(default=False),
                'interface': KeyInfo(),
            },
        ),
    ),
    ('caps-man', 'provisioning'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'action': KeyInfo(default='none'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'common-name-regexp': KeyInfo(default=''),
                'disabled': KeyInfo(default=False),
                'hw-supported-modes': KeyInfo(default=''),
                'identity-regexp': KeyInfo(default=''),
                'ip-address-ranges': KeyInfo(default=''),
                'master-configuration': KeyInfo(default='*FFFFFFFF'),
                'name-format': KeyInfo(default='cap'),
                'name-prefix': KeyInfo(default=''),
                'radio-mac': KeyInfo(default='00:00:00:00:00:00'),
                'slave-configurations': KeyInfo(default=''),
            },
        ),
    ),
    ('caps-man', 'security'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'authentication-types': KeyInfo(can_disable=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disable-pmkid': KeyInfo(can_disable=True),
                'eap-methods': KeyInfo(can_disable=True),
                'eap-radius-accounting': KeyInfo(can_disable=True),
                'encryption': KeyInfo(can_disable=True),
                'group-encryption': KeyInfo(can_disable=True),
                'group-key-update': KeyInfo(),
                'name': KeyInfo(),
                'passphrase': KeyInfo(can_disable=True),
                'tls-certificate': KeyInfo(),
                'tls-mode': KeyInfo(),
            }
        ),
    ),
    ('certificate', 'settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'crl-download': KeyInfo(default=False),
                'crl-store': KeyInfo(default='ram'),
                'crl-use': KeyInfo(default=False),
            },
        ),
    ),
    ('interface', 'bridge', 'port'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('interface', ),
            versioned_fields=[
                ([('7.0', '<')], 'ingress-filtering', KeyInfo(default=False)),
                ([('7.0', '>=')], 'ingress-filtering', KeyInfo(default=True)),
            ],
            fields={
                'auto-isolate': KeyInfo(default=False),
                'bpdu-guard': KeyInfo(default=False),
                'bridge': KeyInfo(required=True),
                'broadcast-flood': KeyInfo(default=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'edge': KeyInfo(default='auto'),
                'fast-leave': KeyInfo(default=False),
                'frame-types': KeyInfo(default='admit-all'),
                'horizon': KeyInfo(default='none'),
                'hw': KeyInfo(default=True),
                'interface': KeyInfo(),
                'internal-path-cost': KeyInfo(default=10),
                'learn': KeyInfo(default='auto'),
                'multicast-router': KeyInfo(default='temporary-query'),
                'path-cost': KeyInfo(default=10),
                'point-to-point': KeyInfo(default='auto'),
                'priority': KeyInfo(default='0x80'),
                'pvid': KeyInfo(default=1),
                'restricted-role': KeyInfo(default=False),
                'restricted-tcn': KeyInfo(default=False),
                'tag-stacking': KeyInfo(default=False),
                'trusted': KeyInfo(default=False),
                'unknown-multicast-flood': KeyInfo(default=True),
                'unknown-unicast-flood': KeyInfo(default=True),
            },
        ),
    ),
    ('interface', 'bridge', 'mlag'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'bridge': KeyInfo(default='none'),
                'peer-port': KeyInfo(default='none'),
            }
        ),
    ),
    ('interface', 'bridge', 'port-controller'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'bridge': KeyInfo(default='none'),
                'cascade-ports': KeyInfo(default=''),
                'switch': KeyInfo(default='none'),
            },
        ),
    ),
    ('interface', 'bridge', 'port-extender'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'control-ports': KeyInfo(default=''),
                'excluded-ports': KeyInfo(default=''),
                'switch': KeyInfo(default='none'),
            },
        ),
    ),
    ('interface', 'bridge', 'settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allow-fast-path': KeyInfo(default=True),
                'use-ip-firewall': KeyInfo(default=False),
                'use-ip-firewall-for-pppoe': KeyInfo(default=False),
                'use-ip-firewall-for-vlan': KeyInfo(default=False),
            },
        ),
    ),
    ('interface', 'bridge', 'vlan'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('bridge', 'vlan-ids', ),
            fields={
                'bridge': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'tagged': KeyInfo(default=''),
                'untagged': KeyInfo(default=''),
                'vlan-ids': KeyInfo(),
            },
        ),
    ),
    ('ip', 'firewall', 'connection', 'tracking'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'enabled': KeyInfo(default='auto'),
                'generic-timeout': KeyInfo(default='10m'),
                'icmp-timeout': KeyInfo(default='10s'),
                'loose-tcp-tracking': KeyInfo(default=True),
                'tcp-close-timeout': KeyInfo(default='10s'),
                'tcp-close-wait-timeout': KeyInfo(default='10s'),
                'tcp-established-timeout': KeyInfo(default='1d'),
                'tcp-fin-wait-timeout': KeyInfo(default='10s'),
                'tcp-last-ack-timeout': KeyInfo(default='10s'),
                'tcp-max-retrans-timeout': KeyInfo(default='5m'),
                'tcp-syn-received-timeout': KeyInfo(default='5s'),
                'tcp-syn-sent-timeout': KeyInfo(default='5s'),
                'tcp-time-wait-timeout': KeyInfo(default='10s'),
                'tcp-unacked-timeout': KeyInfo(default='5m'),
                'udp-stream-timeout': KeyInfo(default='3m'),
                'udp-timeout': KeyInfo(default='10s'),
            },
        ),
    ),
    ('ip', 'neighbor', 'discovery-settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            versioned_fields=[
                ([('7.7', '>=')], 'mode', KeyInfo(default='tx-and-rx')),
                ([('7.15', '>=')], 'lldp-mac-phy-config', KeyInfo(default=False)),
                ([('7.16', '>=')], 'discover-interval', KeyInfo(default='30s')),
                ([('7.16', '>=')], 'lldp-vlan-info', KeyInfo(default=False)),
            ],
            fields={
                'discover-interface-list': KeyInfo(),
                'lldp-med-net-policy-vlan': KeyInfo(default='disabled'),
                'protocol': KeyInfo(default='cdp,lldp,mndp'),
            },
        ),
    ),
    ('ip', 'settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            versioned_fields=[
                ([('7.16', '>=')], 'ipv4-multipath-hash-policy', KeyInfo(default='l3')),
            ],
            fields={
                'accept-redirects': KeyInfo(default=False),
                'accept-source-route': KeyInfo(default=False),
                'allow-fast-path': KeyInfo(default=True),
                'arp-timeout': KeyInfo(default='30s'),
                'icmp-rate-limit': KeyInfo(default=10),
                'icmp-rate-mask': KeyInfo(default='0x1818'),
                'ip-forward': KeyInfo(default=True),
                'max-neighbor-entries': KeyInfo(default=8192),
                'route-cache': KeyInfo(default=True),
                'rp-filter': KeyInfo(default=False),
                'secure-redirects': KeyInfo(default=True),
                'send-redirects': KeyInfo(default=True),
                'tcp-syncookies': KeyInfo(default=False),
            },
        ),
    ),
    ('ipv6', 'address'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'address': KeyInfo(),
                'advertise': KeyInfo(default=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'eui-64': KeyInfo(default=False),
                'from-pool': KeyInfo(default=''),
                'interface': KeyInfo(required=True),
                'no-dad': KeyInfo(default=False),
            },
        ),
    ),
    ('ipv6', 'settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            versioned_fields=[
                ([('7.16', '>=')], 'multipath-hash-policy', KeyInfo(default='l3')),
            ],
            fields={
                'accept-redirects': KeyInfo(default='yes-if-forwarding-disabled'),
                'accept-router-advertisements': KeyInfo(default='yes-if-forwarding-disabled'),
                'disable-ipv6': KeyInfo(default=False),
                'forward': KeyInfo(default=True),
                'max-neighbor-entries': KeyInfo(default=8192),
            },
        ),
    ),
    ('interface', 'detect-internet'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'detect-interface-list': KeyInfo(default='none'),
                'internet-interface-list': KeyInfo(default='none'),
                'lan-interface-list': KeyInfo(default='none'),
                'wan-interface-list': KeyInfo(default='none'),
            },
        ),
    ),
    ('interface', 'l2tp-client',): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('name', ),
            single_value=False,
            fully_understood=True,
            fields={
                'add-default-route': KeyInfo(default=False),
                'allow': KeyInfo(default='pap,chap,mschap1,mschap2'),
                'allow-fast-path': KeyInfo(default=False),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'connect-to': KeyInfo(required=True),
                'default-route-distance': KeyInfo(default=False),
                'dial-on-demand': KeyInfo(default=False),
                'disabled': KeyInfo(default=True),
                'ipsec-secret': KeyInfo(default=''),
                'keepalive-timeout': KeyInfo(default=60),
                'l2tp-proto-version': KeyInfo(default='l2tpv2'),
                'l2tpv3-cookie-length': KeyInfo(default=0),
                'l2tpv3-digest-hash': KeyInfo(default='md5'),
                'max-mru': KeyInfo(default=1450),
                'max-mtu': KeyInfo(default=1450),
                'mrru': KeyInfo(default='disabled'),
                'name': KeyInfo(required=True),
                'password': KeyInfo(),
                'profile': KeyInfo(default='default-encryption'),
                'src-address': KeyInfo(),
                'use-ipsec': KeyInfo(default=False),
                'use-peer-dns': KeyInfo(default=False),
                'user': KeyInfo(required=True),
            },
        ),
    ),
    ('interface', 'l2tp-server', 'server'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allow-fast-path': KeyInfo(default=False),
                'authentication': KeyInfo(default='pap,chap,mschap1,mschap2'),
                'caller-id-type': KeyInfo(default='ip-address'),
                'default-profile': KeyInfo(default='default-encryption'),
                'enabled': KeyInfo(default=False),
                'ipsec-secret': KeyInfo(default=''),
                'keepalive-timeout': KeyInfo(default=30),
                'max-mru': KeyInfo(default=1450),
                'max-mtu': KeyInfo(default=1450),
                'max-sessions': KeyInfo(default='unlimited'),
                'mrru': KeyInfo(default='disabled'),
                'one-session-per-host': KeyInfo(default=False),
                'use-ipsec': KeyInfo(default=False),
            },
        ),
    ),
    ('interface', 'ovpn-client'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'add-default-route': KeyInfo(default=False),
                'auth': KeyInfo(default='sha1'),
                'certificate': KeyInfo(),
                'cipher': KeyInfo(default='blowfish128'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'connect-to': KeyInfo(),
                'disabled': KeyInfo(default=True),
                'disconnect-notify': KeyInfo(),
                'mac-address': KeyInfo(),
                'max-mtu': KeyInfo(default=1500),
                'mode': KeyInfo(default='ip'),
                'name': KeyInfo(),
                'password': KeyInfo(),
                'port': KeyInfo(default=1194),
                'profile': KeyInfo(default='default'),
                'protocol': KeyInfo(default='tcp'),
                'route-nopull': KeyInfo(default=False),
                'tls-version': KeyInfo(default='any'),
                'use-peer-dns': KeyInfo(default=True),
                'user': KeyInfo(),
                'verify-server-certificate': KeyInfo(default=False),
            },
        ),
    ),
    ('interface', 'ovpn-server', 'server'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'auth': KeyInfo(),
                'cipher': KeyInfo(),
                'default-profile': KeyInfo(default='default'),
                'enabled': KeyInfo(default=False),
                'keepalive-timeout': KeyInfo(default=60),
                'mac-address': KeyInfo(),
                'max-mtu': KeyInfo(default=1500),
                'mode': KeyInfo(default='ip'),
                'netmask': KeyInfo(default=24),
                'port': KeyInfo(default=1194),
                'require-client-certificate': KeyInfo(default=False),
            },
        ),
    ),
    ('interface', 'pppoe-server', 'server'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('interface', ),
            fields={
                'accept-empty-service': KeyInfo(default=True),
                'authentication': KeyInfo(default='pap,chap,mschap1,mschap2'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'default-profile': KeyInfo(default='default'),
                'disabled': KeyInfo(default=True),
                'interface': KeyInfo(required=True),
                'keepalive-timeout': KeyInfo(default=10),
                'max-mru': KeyInfo(default='auto'),
                'max-mtu': KeyInfo(default='auto'),
                'max-sessions': KeyInfo(default='unlimited'),
                'mrru': KeyInfo(default='disabled'),
                'one-session-per-host': KeyInfo(default=False),
                'pado-delay': KeyInfo(default=0),
                'service-name': KeyInfo(default=''),
            },
        ),
    ),
    ('interface', 'pptp-server', 'server'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'authentication': KeyInfo(default='mschap1,mschap2'),
                'default-profile': KeyInfo(default='default-encryption'),
                'enabled': KeyInfo(default=False),
                'keepalive-timeout': KeyInfo(default=30),
                'max-mru': KeyInfo(default=1450),
                'max-mtu': KeyInfo(default=1450),
                'mrru': KeyInfo(default='disabled'),
            },
        ),
    ),
    ('interface', 'sstp-server', 'server'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'authentication': KeyInfo(default='pap,chap,mschap1,mschap2'),
                'certificate': KeyInfo(default='none'),
                'default-profile': KeyInfo(default='default'),
                'enabled': KeyInfo(default=False),
                'force-aes': KeyInfo(default=False),
                'keepalive-timeout': KeyInfo(default=60),
                'max-mru': KeyInfo(default=1500),
                'max-mtu': KeyInfo(default=1500),
                'mrru': KeyInfo(default='disabled'),
                'pfs': KeyInfo(default=False),
                'port': KeyInfo(default=443),
                'tls-version': KeyInfo(default='any'),
                'verify-client-certificate': KeyInfo(default='no'),
            },
        ),
    ),
    ('interface', 'wifi'): APIData(
        versioned=[
            ('7.13', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                required_one_of=[['default-name', 'radio-mac', 'master-interface']],
                fields={
                    'aaa.called-format': KeyInfo(can_disable=True),
                    'aaa.calling-format': KeyInfo(can_disable=True),
                    'aaa.interim-update': KeyInfo(can_disable=True),
                    'aaa.mac-caching': KeyInfo(can_disable=True),
                    'aaa.nas-identifier': KeyInfo(can_disable=True),
                    'aaa.password-format': KeyInfo(can_disable=True),
                    'aaa.username-format': KeyInfo(can_disable=True),
                    'aaa': KeyInfo(can_disable=True),
                    'arp-timeout': KeyInfo(default='auto'),
                    'arp': KeyInfo(can_disable=True),
                    'channel.band': KeyInfo(can_disable=True),
                    'channel.frequency': KeyInfo(can_disable=True),
                    'channel.secondary-frequency': KeyInfo(can_disable=True),
                    'channel.skip-dfs-channels': KeyInfo(can_disable=True),
                    'channel.width': KeyInfo(can_disable=True),
                    'channel': KeyInfo(can_disable=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'configuration.antenna-gain': KeyInfo(can_disable=True),
                    'configuration.beacon-interval': KeyInfo(can_disable=True),
                    'configuration.chains': KeyInfo(can_disable=True),
                    'configuration.country': KeyInfo(can_disable=True),
                    'configuration.dtim-period': KeyInfo(can_disable=True),
                    'configuration.hide-ssid': KeyInfo(can_disable=True),
                    'configuration.manager': KeyInfo(can_disable=True),
                    'configuration.mode': KeyInfo(can_disable=True),
                    'configuration.multicast-enhance': KeyInfo(can_disable=True),
                    'configuration.qos-classifier': KeyInfo(can_disable=True),
                    'configuration.ssid': KeyInfo(can_disable=True),
                    'configuration.tx-chain': KeyInfo(can_disable=True),
                    'configuration.tx-power': KeyInfo(can_disable=True),
                    'configuration': KeyInfo(can_disable=True),
                    'datapath.bridge-cost': KeyInfo(can_disable=True),
                    'datapath.bridge-horizon': KeyInfo(can_disable=True),
                    'datapath.bridge': KeyInfo(can_disable=True),
                    'datapath.client-isolation': KeyInfo(can_disable=True),
                    'datapath.interface-list': KeyInfo(can_disable=True),
                    'datapath.vlan-id': KeyInfo(can_disable=True),
                    'datapath': KeyInfo(can_disable=True),
                    'default-name': KeyInfo(),
                    'disable-running-check': KeyInfo(can_disable=True),
                    'disabled': KeyInfo(default=True),
                    'interworking.3gpp-info': KeyInfo(can_disable=True),
                    'interworking.authentication-types': KeyInfo(can_disable=True),
                    'interworking.connection-capabilities': KeyInfo(can_disable=True),
                    'interworking.domain-names': KeyInfo(can_disable=True),
                    'interworking.esr': KeyInfo(can_disable=True),
                    'interworking.hessid': KeyInfo(can_disable=True),
                    'interworking.hotspot20-dgaf': KeyInfo(can_disable=True),
                    'interworking.hotspot20': KeyInfo(can_disable=True),
                    'interworking.internet': KeyInfo(can_disable=True),
                    'interworking.ipv4-availability': KeyInfo(can_disable=True),
                    'interworking.ipv6-availability': KeyInfo(can_disable=True),
                    'interworking.network-type': KeyInfo(can_disable=True),
                    'interworking.operational-classes': KeyInfo(can_disable=True),
                    'interworking.operator-names': KeyInfo(can_disable=True),
                    'interworking.realms': KeyInfo(can_disable=True),
                    'interworking.roaming-ois': KeyInfo(can_disable=True),
                    'interworking.uesa': KeyInfo(can_disable=True),
                    'interworking.venue-names': KeyInfo(can_disable=True),
                    'interworking.venue': KeyInfo(can_disable=True),
                    'interworking.wan-at-capacity': KeyInfo(can_disable=True),
                    'interworking.wan-downlink-load': KeyInfo(can_disable=True),
                    'interworking.wan-downlink': KeyInfo(can_disable=True),
                    'interworking.wan-measurement-duration': KeyInfo(can_disable=True),
                    'interworking.wan-status': KeyInfo(can_disable=True),
                    'interworking.wan-symmetric': KeyInfo(can_disable=True),
                    'interworking.wan-uplink-load': KeyInfo(can_disable=True),
                    'interworking.wan-uplink': KeyInfo(can_disable=True),
                    'interworking': KeyInfo(can_disable=True),
                    'l2mtu': KeyInfo(default=1560),
                    'mac-address': KeyInfo(),
                    'master-interface': KeyInfo(),
                    'mtu': KeyInfo(default=1500),
                    'name': KeyInfo(),
                    'radio-mac': KeyInfo(),
                    'security.authentication-types': KeyInfo(can_disable=True),
                    'security.connect-group': KeyInfo(can_disable=True),
                    'security.connect-priority': KeyInfo(can_disable=True),
                    'security.dh-groups': KeyInfo(can_disable=True),
                    'security.disable-pmkid': KeyInfo(can_disable=True),
                    'security.eap-accounting': KeyInfo(can_disable=True),
                    'security.eap-anonymous-identity': KeyInfo(can_disable=True),
                    'security.eap-certificate-mode': KeyInfo(can_disable=True),
                    'security.eap-methods': KeyInfo(can_disable=True),
                    'security.eap-password': KeyInfo(can_disable=True),
                    'security.eap-tls-certificate': KeyInfo(can_disable=True),
                    'security.eap-username': KeyInfo(can_disable=True),
                    'security.encryption': KeyInfo(can_disable=True),
                    'security.ft-mobility-domain': KeyInfo(can_disable=True),
                    'security.ft-nas-identifier': KeyInfo(can_disable=True),
                    'security.ft-over-ds': KeyInfo(can_disable=True),
                    'security.ft-preserve-vlanid': KeyInfo(can_disable=True),
                    'security.ft-r0-key-lifetime': KeyInfo(can_disable=True),
                    'security.ft-reassociation-deadline': KeyInfo(can_disable=True),
                    'security.ft': KeyInfo(can_disable=True),
                    'security.group-encryption': KeyInfo(can_disable=True),
                    'security.group-key-update': KeyInfo(can_disable=True),
                    'security.management-encryption': KeyInfo(can_disable=True),
                    'security.management-protection': KeyInfo(can_disable=True),
                    'security.owe-transition-interface': KeyInfo(can_disable=True),
                    'security.passphrase': KeyInfo(can_disable=True),
                    'security.sae-anti-clogging-threshold': KeyInfo(can_disable=True),
                    'security.sae-max-failure-rate': KeyInfo(can_disable=True),
                    'security.sae-pwe': KeyInfo(can_disable=True),
                    'security.wps': KeyInfo(can_disable=True),
                    'security': KeyInfo(can_disable=True),
                    'steering.neighbor-group': KeyInfo(can_disable=True),
                    'steering.rrm': KeyInfo(can_disable=True),
                    'steering.wnm': KeyInfo(can_disable=True),
                    'steering': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('interface', 'wifi', 'aaa'): APIData(
        versioned=[
            ('7.13', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                fields={
                    'called-format': KeyInfo(can_disable=True),
                    'calling-format': KeyInfo(can_disable=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'interim-update': KeyInfo(can_disable=True),
                    'mac-caching': KeyInfo(can_disable=True),
                    'name': KeyInfo(),
                    'nas-identifier': KeyInfo(can_disable=True),
                    'password-format': KeyInfo(can_disable=True),
                    'username-format': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('interface', 'wifi', 'access-list'): APIData(
        versioned=[
            ('7.13', '>=', VersionedAPIData(
                fully_understood=True,
                fields={
                    'action': KeyInfo(default='accept'),
                    'allow-signal-out-of-range': KeyInfo(can_disable=True),
                    'client-isolation': KeyInfo(can_disable=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'interface': KeyInfo(can_disable=True),
                    'mac-address-mask': KeyInfo(can_disable=True),
                    'mac-address': KeyInfo(can_disable=True),
                    'passphrase': KeyInfo(can_disable=True),
                    'radius-accounting': KeyInfo(can_disable=True),
                    'signal-range': KeyInfo(can_disable=True),
                    'ssid-regexp': KeyInfo(can_disable=True),
                    'time': KeyInfo(can_disable=True),
                    'vlan-id': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('interface', 'wifi', 'cap'): APIData(
        versioned=[
            ('7.13', '>=', VersionedAPIData(
                single_value=True,
                fully_understood=True,
                fields={
                    'caps-man-addresses': KeyInfo(default=''),
                    'caps-man-certificate-common-names': KeyInfo(default=''),
                    'caps-man-names': KeyInfo(default=''),
                    'certificate': KeyInfo(default='none'),
                    'discovery-interfaces': KeyInfo(default=''),
                    'enabled': KeyInfo(default=False),
                    'lock-to-caps-man': KeyInfo(default=False),
                    'slaves-datapath': KeyInfo(),
                    'slaves-static': KeyInfo(),
                },
            )),
        ],
    ),
    ('interface', 'wifi', 'capsman'): APIData(
        versioned=[
            ('7.13', '>=', VersionedAPIData(
                single_value=True,
                fully_understood=True,
                fields={
                    'ca-certificate': KeyInfo(default=''),
                    'certificate': KeyInfo(default='none'),
                    'enabled': KeyInfo(default=False),
                    'interfaces': KeyInfo(default=''),
                    'package-path': KeyInfo(default=''),
                    'require-peer-certificate': KeyInfo(default=False),
                    'upgrade-policy': KeyInfo(default='none'),
                },
            )),
        ],
    ),
    ('interface', 'wifi', 'channel'): APIData(
        versioned=[
            ('7.13', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                fields={
                    'band': KeyInfo(can_disable=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'frequency': KeyInfo(can_disable=True),
                    'name': KeyInfo(),
                    'secondary-frequency': KeyInfo(can_disable=True),
                    'skip-dfs-channels': KeyInfo(can_disable=True),
                    'width': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('interface', 'wifi', 'configuration'): APIData(
        versioned=[
            ('7.13', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                fields={
                    'aaa': KeyInfo(can_disable=True),
                    'antenna-gain': KeyInfo(can_disable=True),
                    'beacon-interval': KeyInfo(can_disable=True),
                    'chains': KeyInfo(can_disable=True),
                    'channel': KeyInfo(can_disable=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'country': KeyInfo(can_disable=True),
                    'datapath': KeyInfo(can_disable=True),
                    'disabled': KeyInfo(default=False),
                    'dtim-period': KeyInfo(can_disable=True),
                    'hide-ssid': KeyInfo(default=False),
                    'interworking': KeyInfo(can_disable=True),
                    'manager': KeyInfo(can_disable=True),
                    'mode': KeyInfo(can_disable=True),
                    'multicast-enhance': KeyInfo(can_disable=True),
                    'name': KeyInfo(),
                    'qos-classifier': KeyInfo(can_disable=True),
                    'security': KeyInfo(can_disable=True),
                    'ssid': KeyInfo(can_disable=True),
                    'steering': KeyInfo(can_disable=True),
                    'tx-chains': KeyInfo(can_disable=True),
                    'tx-power': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('interface', 'wifi', 'datapath'): APIData(
        versioned=[
            ('7.13', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                fields={
                    'bridge-cost': KeyInfo(can_disable=True),
                    'bridge-horizon': KeyInfo(can_disable=True),
                    'bridge': KeyInfo(can_disable=True),
                    'client-isolation': KeyInfo(can_disable=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'interface-list': KeyInfo(can_disable=True),
                    'name': KeyInfo(),
                    'vlan-id': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('interface', 'wifi', 'interworking'): APIData(
        versioned=[
            ('7.13', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                fields={
                    '3gpp-info': KeyInfo(can_disable=True),
                    'authentication-types': KeyInfo(can_disable=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'connection-capabilities': KeyInfo(can_disable=True),
                    'disabled': KeyInfo(default=False),
                    'domain-names': KeyInfo(can_disable=True),
                    'esr': KeyInfo(can_disable=True),
                    'hessid': KeyInfo(can_disable=True),
                    'hotspot20-dgaf': KeyInfo(can_disable=True),
                    'hotspot20': KeyInfo(can_disable=True),
                    'internet': KeyInfo(can_disable=True),
                    'ipv4-availability': KeyInfo(can_disable=True),
                    'ipv6-availability': KeyInfo(can_disable=True),
                    'name': KeyInfo(),
                    'network-type': KeyInfo(can_disable=True),
                    'operational-classes': KeyInfo(can_disable=True),
                    'operator-names': KeyInfo(can_disable=True),
                    'realms': KeyInfo(can_disable=True),
                    'roaming-ois': KeyInfo(can_disable=True),
                    'uesa': KeyInfo(can_disable=True),
                    'venue-names': KeyInfo(can_disable=True),
                    'venue': KeyInfo(can_disable=True),
                    'wan-at-capacity': KeyInfo(can_disable=True),
                    'wan-downlink-load': KeyInfo(can_disable=True),
                    'wan-downlink': KeyInfo(can_disable=True),
                    'wan-measurement-duration': KeyInfo(can_disable=True),
                    'wan-status': KeyInfo(can_disable=True),
                    'wan-symmetric': KeyInfo(can_disable=True),
                    'wan-uplink-load': KeyInfo(can_disable=True),
                    'wan-uplink': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('interface', 'wifi', 'provisioning'): APIData(
        versioned=[
            ('7.13', '>=', VersionedAPIData(
                fully_understood=True,
                fields={
                    'action': KeyInfo(default='none'),
                    'address-ranges': KeyInfo(can_disable=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'common-name-regexp': KeyInfo(can_disable=True),
                    'disabled': KeyInfo(default=False),
                    'identity-regexp': KeyInfo(can_disable=True),
                    'master-configuration': KeyInfo(can_disable=True),
                    'name-format': KeyInfo(can_disable=True),
                    'radio-mac': KeyInfo(can_disable=True),
                    'slave-configurations': KeyInfo(can_disable=True),
                    'supported-bands': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('interface', 'wifi', 'security'): APIData(
        versioned=[
            ('7.13', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                fields={
                    'authentication-types': KeyInfo(can_disable=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'connect-group': KeyInfo(can_disable=True),
                    'connect-priority': KeyInfo(can_disable=True),
                    'dh-groups': KeyInfo(can_disable=True),
                    'disable-pmkid': KeyInfo(can_disable=True),
                    'disabled': KeyInfo(default=False),
                    'eap-accounting': KeyInfo(can_disable=True),
                    'eap-anonymous-identity': KeyInfo(can_disable=True),
                    'eap-certificate-mode': KeyInfo(can_disable=True),
                    'eap-methods': KeyInfo(can_disable=True),
                    'eap-password': KeyInfo(can_disable=True),
                    'eap-tls-certificate': KeyInfo(can_disable=True),
                    'eap-username': KeyInfo(can_disable=True),
                    'encryption': KeyInfo(can_disable=True),
                    'ft-mobility-domain': KeyInfo(can_disable=True),
                    'ft-nas-identifier': KeyInfo(can_disable=True),
                    'ft-over-ds': KeyInfo(can_disable=True),
                    'ft-preserve-vlanid': KeyInfo(can_disable=True),
                    'ft-r0-key-lifetime': KeyInfo(can_disable=True),
                    'ft-reassociation-deadline': KeyInfo(can_disable=True),
                    'ft': KeyInfo(can_disable=True),
                    'group-encryption': KeyInfo(can_disable=True),
                    'group-key-update': KeyInfo(can_disable=True),
                    'management-encryption': KeyInfo(can_disable=True),
                    'management-protection': KeyInfo(can_disable=True),
                    'name': KeyInfo(),
                    'owe-transition-interface': KeyInfo(can_disable=True),
                    'passphrase': KeyInfo(can_disable=True),
                    'sae-anti-clogging-threshold': KeyInfo(can_disable=True),
                    'sae-max-failure-rate': KeyInfo(can_disable=True),
                    'sae-pwe': KeyInfo(can_disable=True),
                    'wps': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('interface', 'wifi', 'steering'): APIData(
        versioned=[
            ('7.13', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                fields={
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'name': KeyInfo(),
                    'neighbor-group': KeyInfo(can_disable=True),
                    'rrm': KeyInfo(can_disable=True),
                    'wnm': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('interface', 'wifiwave2'): APIData(
        versioned=[
            ('7.13', '>=', 'RouterOS 7.13 uses WiFi package'),
            ('7.8', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                required_one_of=[['default-name', 'master-interface']],
                fields={
                    'aaa': KeyInfo(),
                    'arp-timeout': KeyInfo(default='auto'),
                    'arp': KeyInfo(default='enabled'),
                    'channel': KeyInfo(),
                    'configuration': KeyInfo(),
                    'datapath': KeyInfo(),
                    'default-name': KeyInfo(),
                    'disable-running-check': KeyInfo(default=False),
                    'interworking': KeyInfo(),
                    'l2mtu': KeyInfo(default=1600),
                    'mac-address': KeyInfo(),
                    'master-interface': KeyInfo(),
                    'mtu': KeyInfo(default=1500),
                    'name': KeyInfo(),
                    'security': KeyInfo(),
                    'steering': KeyInfo(),
                },
            )),
        ],
    ),
    ('interface', 'wifiwave2', 'aaa'): APIData(
        versioned=[
            ('7.13', '>=', 'RouterOS 7.13 uses WiFi package'),
            ('7.8', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                fields={
                    'called-format': KeyInfo(can_disable=True),
                    'calling-format': KeyInfo(can_disable=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=True),
                    'interim-update': KeyInfo(can_disable=True),
                    'mac-caching': KeyInfo(can_disable=True),
                    'name': KeyInfo(),
                    'nas-identifier': KeyInfo(can_disable=True),
                    'password-format': KeyInfo(can_disable=True),
                    'username-format': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('interface', 'wifiwave2', 'access-list'): APIData(
        versioned=[
            ('7.13', '>=', 'RouterOS 7.13 uses WiFi package'),
            ('7.8', '>=', VersionedAPIData(
                fully_understood=True,
                fields={
                    'action': KeyInfo(can_disable=True),
                    'allow-signal-out-of-range': KeyInfo(can_disable=True),
                    'client-isolation': KeyInfo(can_disable=True),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=True),
                    'interface': KeyInfo(can_disable=True),
                    'mac-address': KeyInfo(can_disable=True),
                    'mac-address-mask': KeyInfo(can_disable=True),
                    'passphrase': KeyInfo(can_disable=True),
                    'radius-accounting': KeyInfo(can_disable=True),
                    'signal-range': KeyInfo(can_disable=True),
                    'ssid-regexp': KeyInfo(),
                    'time': KeyInfo(can_disable=True),
                    'vlan-id': KeyInfo(can_disable=True),
                },
            )),
        ],
    ),
    ('interface', 'wifiwave2', 'cap'): APIData(
        versioned=[
            ('7.13', '>=', 'RouterOS 7.13 uses WiFi package'),
            ('7.8', '>=', VersionedAPIData(
                single_value=True,
                fully_understood=True,
                fields={
                    'caps-man-addresses': KeyInfo(default=''),
                    'caps-man-certificate-common-names': KeyInfo(default=''),
                    'caps-man-names': KeyInfo(default=''),
                    'certificate': KeyInfo(default='none'),
                    'discovery-interfaces': KeyInfo(default=''),
                    'enabled': KeyInfo(default=False),
                    'lock-to-caps-man': KeyInfo(default=False),
                    'slaves-datapath': KeyInfo(),
                    'slaves-static': KeyInfo(),
                },
            )),
        ],
    ),
    ('interface', 'wifiwave2', 'capsman'): APIData(
        versioned=[
            ('7.13', '>=', 'RouterOS 7.13 uses WiFi package'),
            ('7.8', '>=', VersionedAPIData(
                single_value=True,
                fully_understood=True,
                fields={
                    'ca-certificate': KeyInfo(default=''),
                    'certificate': KeyInfo(default='none'),
                    'enabled': KeyInfo(default=False),
                    'package-path': KeyInfo(default=''),
                    'require-peer-certificate': KeyInfo(default=False),
                    'upgrade-policy': KeyInfo(default='none'),
                    'interfaces': KeyInfo(default=''),
                },
            )),
        ],
    ),
    ('interface', 'wifiwave2', 'channel'): APIData(
        versioned=[
            ('7.13', '>=', 'RouterOS 7.13 uses WiFi package'),
            ('7.8', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                fields={
                    'band': KeyInfo(),
                    'frequency': KeyInfo(),
                    'name': KeyInfo(),
                    'secondary-frequency': KeyInfo(),
                    'skip-dfs-channels': KeyInfo(default='disabled'),
                    'width': KeyInfo(),
                },
            )),
        ],
    ),
    ('interface', 'wifiwave2', 'configuration'): APIData(
        versioned=[
            ('7.13', '>=', 'RouterOS 7.13 uses WiFi package'),
            ('7.8', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                fields={
                    'aaa': KeyInfo(),
                    'antenna-gain': KeyInfo(),
                    'beacon-interval': KeyInfo(default=100),
                    'chains': KeyInfo(),
                    'channel': KeyInfo(),
                    'country': KeyInfo(default='United States'),
                    'datapath': KeyInfo(),
                    'dtim-period': KeyInfo(default=1),
                    'hide-ssid': KeyInfo(default=False),
                    'interworking': KeyInfo(),
                    'manager': KeyInfo(),
                    'mode': KeyInfo(default='ap'),
                    'name': KeyInfo(),
                    'security': KeyInfo(),
                    'ssid': KeyInfo(),
                    'steering': KeyInfo(),
                    'tx-chains': KeyInfo(),
                    'tx-power': KeyInfo(),
                },
            )),
        ],
    ),
    ('interface', 'wifiwave2', 'datapath'): APIData(
        versioned=[
            ('7.13', '>=', 'RouterOS 7.13 uses WiFi package'),
            ('7.8', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                fields={
                    'bridge': KeyInfo(),
                    'bridge-cost': KeyInfo(),
                    'bridge-horizon': KeyInfo(),
                    'client-isolation': KeyInfo(default=False),
                    'interface-list': KeyInfo(),
                    'name': KeyInfo(),
                    'openflow-switch': KeyInfo(),
                    'vlan-id': KeyInfo(),
                },
            )),
        ],
    ),
    ('interface', 'wifiwave2', 'interworking'): APIData(
        versioned=[
            ('7.13', '>=', 'RouterOS 7.13 uses WiFi package'),
            ('7.8', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                fields={
                    '3gpp-info': KeyInfo(),
                    'authentication-types': KeyInfo(),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'connection-capabilities': KeyInfo(),
                    'disabled': KeyInfo(default=False),
                    'domain-names': KeyInfo(),
                    'esr': KeyInfo(),
                    'hessid': KeyInfo(),
                    'hotspot20': KeyInfo(),
                    'hotspot20-dgaf': KeyInfo(),
                    'internet': KeyInfo(),
                    'ipv4-availability': KeyInfo(),
                    'ipv6-availability': KeyInfo(),
                    'name': KeyInfo(),
                    'network-type': KeyInfo(),
                    'operational-classes': KeyInfo(),
                    'operator-names': KeyInfo(),
                    'realms': KeyInfo(),
                    'roaming-ois': KeyInfo(),
                    'uesa': KeyInfo(),
                    'venue': KeyInfo(),
                    'venue-names': KeyInfo(),
                    'wan-at-capacity': KeyInfo(),
                    'wan-downlink': KeyInfo(),
                    'wan-downlink-load': KeyInfo(),
                    'wan-measurement-duration': KeyInfo(),
                    'wan-status': KeyInfo(),
                    'wan-symmetric': KeyInfo(),
                    'wan-uplink': KeyInfo(),
                    'wan-uplink-load': KeyInfo(),
                },
            )),
        ],
    ),
    ('interface', 'wifiwave2', 'provisioning'): APIData(
        versioned=[
            ('7.13', '>=', 'RouterOS 7.13 uses WiFi package'),
            ('7.8', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('action', ),
                fields={
                    'action': KeyInfo(default='none'),
                    'address-ranges': KeyInfo(),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'common-name-regexp': KeyInfo(),
                    'disabled': KeyInfo(default=False),
                    'identity-regexp': KeyInfo(),
                    'master-configuration': KeyInfo(),
                    'name-format': KeyInfo(),
                    'radio-mac': KeyInfo(),
                    'slave-configurations': KeyInfo(),
                    'supported-bands': KeyInfo(),
                },
            )),
        ],
    ),
    ('interface', 'wifiwave2', 'security'): APIData(
        versioned=[
            ('7.13', '>=', 'RouterOS 7.13 uses WiFi package'),
            ('7.8', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                fields={
                    'authentication-types': KeyInfo(),
                    'connect-group': KeyInfo(can_disable=True),
                    'connect-priority': KeyInfo(),
                    'dh-groups': KeyInfo(),
                    'disable-pmkid': KeyInfo(default=False),
                    'eap-accounting': KeyInfo(default=False),
                    'eap-anonymous-identity': KeyInfo(),
                    'eap-certificate-mode': KeyInfo(default='dont-verify-certificate'),
                    'eap-methods': KeyInfo(),
                    'eap-password': KeyInfo(),
                    'eap-tls-certificate': KeyInfo(),
                    'eap-username': KeyInfo(),
                    'encryption': KeyInfo(default='ccmp'),
                    'ft-mobility-domain': KeyInfo(default=0xADC4),
                    'ft-nas-identifier': KeyInfo(),
                    'ft-over-ds': KeyInfo(default=False),
                    'ft-preserve-vlanid': KeyInfo(default=True),
                    'ft-r0-key-lifetime': KeyInfo(default='600000s'),
                    'ft-reassociation-deadline': KeyInfo(default='20s'),
                    'ft': KeyInfo(default=False),
                    'group-encryption': KeyInfo(default='ccmp'),
                    'group-key-update': KeyInfo(default='24h'),
                    'management-encryption': KeyInfo(default='cmac'),
                    'management-protection': KeyInfo(),
                    'name': KeyInfo(),
                    'owe-transition-interface': KeyInfo(),
                    'passphrase': KeyInfo(default=''),
                    'sae-anti-clogging-threshold': KeyInfo(can_disable=True),
                    'sae-max-failure-rate': KeyInfo(can_disable=True),
                    'sae-pwe': KeyInfo(default='both'),
                    'wps': KeyInfo(default='push-button'),
                },
            )),
        ],
    ),
    ('interface', 'wifiwave2', 'steering'): APIData(
        versioned=[
            ('7.13', '>=', 'RouterOS 7.13 uses WiFi package'),
            ('7.8', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                fields={
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'name': KeyInfo(),
                    'neighbor-group': KeyInfo(),
                    'rrm': KeyInfo(),
                    'wnm': KeyInfo(),
                },
            )),
        ],
    ),
    ('interface', 'wireguard'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'listen-port': KeyInfo(),
                'mtu': KeyInfo(default=1420),
                'name': KeyInfo(),
                'private-key': KeyInfo(),
            },
        ),
    ),
    ('interface', 'wireguard', 'peers'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('public-key', 'interface'),
            fields={
                'allowed-address': KeyInfo(required=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'endpoint-address': KeyInfo(default=''),
                'endpoint-port': KeyInfo(default=0),
                'interface': KeyInfo(),
                'persistent-keepalive': KeyInfo(can_disable=True, remove_value=0),
                'preshared-key': KeyInfo(can_disable=True, remove_value=''),
                'public-key': KeyInfo(),
            },
            versioned_fields=[
                ([('7.15', '>=')], 'name', KeyInfo()),
                ([('7.15', '>='), ('7.17', '<')], 'is-responder', KeyInfo()),
                ([('7.17', '>=')], 'responder', KeyInfo()),
            ],
        ),
    ),
    ('interface', 'wireless'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            required_one_of=[['default-name', 'master-interface']],
            fields={
                'adaptive-noise-immunity': KeyInfo(default='none'),
                'allow-sharedkey': KeyInfo(default=False),
                'ampdu-priorities': KeyInfo(default=0),
                'amsdu-limit': KeyInfo(default=8192),
                'amsdu-threshold': KeyInfo(default=8192),
                'antenna-gain': KeyInfo(default=0),
                'antenna-mode': KeyInfo(),
                'area': KeyInfo(default=''),
                'arp': KeyInfo(default='enabled'),
                'arp-timeout': KeyInfo(default='auto'),
                'band': KeyInfo(),
                'basic-rates-a/g': KeyInfo(default='6Mbps'),
                'basic-rates-b': KeyInfo(default='1Mbps'),
                'bridge-mode': KeyInfo(default='enabled'),
                'channel-width': KeyInfo(default='20mhz'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'compression': KeyInfo(default=False),
                'country': KeyInfo(default='etsi'),
                'default-ap-tx-limit': KeyInfo(default=0),
                'default-authentication': KeyInfo(default=True),
                'default-client-tx-limit': KeyInfo(default=0),
                'default-forwarding': KeyInfo(default=True),
                'default-name': KeyInfo(),
                'disable-running-check': KeyInfo(default=False),
                'disabled': KeyInfo(default=True),
                'disconnect-timeout': KeyInfo(default='3s'),
                'distance': KeyInfo(default='dynamic'),
                'frame-lifetime': KeyInfo(default=0),
                'frequency': KeyInfo(),
                'frequency-mode': KeyInfo(default='regulatory-domain'),
                'frequency-offset': KeyInfo(default=0),
                'guard-interval': KeyInfo(default='any'),
                'hide-ssid': KeyInfo(default=False),
                'ht-basic-mcs': KeyInfo(),
                'ht-supported-mcs': KeyInfo(),
                'hw-fragmentation-threshold': KeyInfo(default='disabled'),
                'hw-protection-mode': KeyInfo(default='none'),
                'hw-protection-threshold': KeyInfo(default=0),
                'hw-retries': KeyInfo(default=7),
                'installation': KeyInfo(default='any'),
                'interworking-profile': KeyInfo(default='disabled'),
                'keepalive-frames': KeyInfo(default='enabled'),
                'l2mtu': KeyInfo(default=1600),
                'mac-address': KeyInfo(),
                'master-interface': KeyInfo(),
                'max-station-count': KeyInfo(default=2007),
                'mode': KeyInfo(default='ap-bridge'),
                'mtu': KeyInfo(default=1500),
                'multicast-buffering': KeyInfo(default='enabled'),
                'multicast-helper': KeyInfo(default='default'),
                'name': KeyInfo(),
                'noise-floor-threshold': KeyInfo(default='default'),
                'nv2-cell-radius': KeyInfo(default=30),
                'nv2-downlink-ratio': KeyInfo(default=50),
                'nv2-mode': KeyInfo(default='dynamic-downlink'),
                'nv2-noise-floor-offset': KeyInfo(default='default'),
                'nv2-preshared-key': KeyInfo(default=''),
                'nv2-qos': KeyInfo(default='default'),
                'nv2-queue-count': KeyInfo(default=2),
                'nv2-security': KeyInfo(default='disabled'),
                'nv2-sync-secret': KeyInfo(default=''),
                'on-fail-retry-time': KeyInfo(default='100ms'),
                'preamble-mode': KeyInfo(default='both'),
                'radio-name': KeyInfo(),
                'rate-selection': KeyInfo(default='advanced'),
                'rate-set': KeyInfo(default='default'),
                'running': KeyInfo(read_only=True),
                'rx-chains': KeyInfo(default='0,1'),
                'scan-list': KeyInfo(default='default'),
                'secondary-frequency': KeyInfo(default=''),
                'security-profile': KeyInfo(default='default'),
                'skip-dfs-channels': KeyInfo(default='disabled'),
                'ssid': KeyInfo(required=True),
                'station-bridge-clone-mac': KeyInfo(),
                'station-roaming': KeyInfo(default='disabled'),
                'supported-rates-a/g': KeyInfo(),
                'supported-rates-b': KeyInfo(),
                'tdma-period-size': KeyInfo(default=2),
                'tx-chains': KeyInfo(),
                'tx-power': KeyInfo(default=''),
                'tx-power-mode': KeyInfo(default='default'),
                'update-stats-interval': KeyInfo(default='disabled'),
                'vlan-id': KeyInfo(default=1),
                'vlan-mode': KeyInfo(default='no-tag'),
                'wds-cost-range': KeyInfo(default='50-150'),
                'wds-default-bridge': KeyInfo(default='none'),
                'wds-default-cost': KeyInfo(default=100),
                'wds-ignore-ssid': KeyInfo(default=False),
                'wds-mode': KeyInfo(default='disabled'),
                'wireless-protocol': KeyInfo(default='any'),
                'wmm-support': KeyInfo(default='disabled'),
                'wps-mode': KeyInfo(default='push-button'),
            },
        ),
    ),
    ('interface', 'wireless', 'align'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'active-mode': KeyInfo(default=True),
                'audio-max': KeyInfo(default=-20),
                'audio-min': KeyInfo(default=-100),
                'audio-monitor': KeyInfo(default='00:00:00:00:00:00'),
                'filter-mac': KeyInfo(default='00:00:00:00:00:00'),
                'frame-size': KeyInfo(default=300),
                'frames-per-second': KeyInfo(default=25),
                'receive-all': KeyInfo(default=False),
                'ssid-all': KeyInfo(default=False),
            },
        ),
    ),
    ('interface', 'wireless', 'access-list'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'allow-signal-out-of-range': KeyInfo(default='10s'),
                'ap-tx-limit': KeyInfo(default=0),
                'authentication': KeyInfo(default=True),
                'client-tx-limit': KeyInfo(default=0),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'forwarding': KeyInfo(default=True),
                'interface': KeyInfo(default='any'),
                'mac-address': KeyInfo(default='00:00:00:00:00:00'),
                'management-protection-key': KeyInfo(default=''),
                'private-algo': KeyInfo(default='none'),
                'private-key': KeyInfo(default=''),
                'private-pre-shared-key': KeyInfo(default=''),
                'signal-range': KeyInfo(default='-120..120'),
                'time': KeyInfo(),
                'vlan-id': KeyInfo(default=1),
                'vlan-mode': KeyInfo(default='default'),
            },
        ),
    ),
    ('interface', 'wireless', 'cap'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'bridge': KeyInfo(default='none'),
                'caps-man-addresses': KeyInfo(default=''),
                'caps-man-certificate-common-names': KeyInfo(default=''),
                'caps-man-names': KeyInfo(default=''),
                'certificate': KeyInfo(default='none'),
                'discovery-interfaces': KeyInfo(default=''),
                'enabled': KeyInfo(default=False),
                'interfaces': KeyInfo(default=''),
                'lock-to-caps-man': KeyInfo(default=False),
                'static-virtual': KeyInfo(default=False),
            },
        ),
    ),
    ('interface', 'wireless', 'connect-list'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                '3gpp': KeyInfo(default=''),
                'allow-signal-out-of-range': KeyInfo(default='10s'),
                'area-prefix': KeyInfo(default=''),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'connect': KeyInfo(default=True),
                'disabled': KeyInfo(default=False),
                'interface': KeyInfo(required=True),
                'interworking': KeyInfo(default='any'),
                'iw-asra': KeyInfo(default='any'),
                'iw-authentication-types': KeyInfo(),
                'iw-connection-capabilities': KeyInfo(),
                'iw-esr': KeyInfo(default='any'),
                'iw-hessid': KeyInfo(default='00:00:00:00:00:00'),
                'iw-hotspot20': KeyInfo(default='any'),
                'iw-hotspot20-dgaf': KeyInfo(default='any'),
                'iw-internet': KeyInfo(default='any'),
                'iw-ipv4-availability': KeyInfo(default='any'),
                'iw-ipv6-availability': KeyInfo(default='any'),
                'iw-network-type': KeyInfo(default='wildcard'),
                'iw-realms': KeyInfo(),
                'iw-roaming-ois': KeyInfo(default=''),
                'iw-uesa': KeyInfo(default='any'),
                'iw-venue': KeyInfo(default='any'),
                'mac-address': KeyInfo(default='00:00:00:00:00:00'),
                'security-profile': KeyInfo(default='none'),
                'signal-range': KeyInfo(default='-120..120'),
                'ssid': KeyInfo(default=''),
                'wireless-protocol': KeyInfo(default='any'),
            },
        ),
    ),
    ('interface', 'wireless', 'security-profiles'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'authentication-types': KeyInfo(),
                'disable-pmkid': KeyInfo(default=False),
                'disabled': KeyInfo(default=True),
                'eap-methods': KeyInfo(),
                'group-ciphers': KeyInfo(),
                'group-key-update': KeyInfo(default='5m'),
                'interim-update': KeyInfo(),
                'management-protection': KeyInfo(default='disabled'),
                'management-protection-key': KeyInfo(default=''),
                'mode': KeyInfo(default='none'),
                'mschapv2-password': KeyInfo(default=''),
                'mschapv2-username': KeyInfo(default=''),
                'name': KeyInfo(),
                'radius-called-format': KeyInfo(),
                'radius-eap-accounting': KeyInfo(default=False),
                'radius-mac-accounting': KeyInfo(default=False),
                'radius-mac-authentication': KeyInfo(default=False),
                'radius-mac-caching': KeyInfo(default='disabled'),
                'radius-mac-format': KeyInfo(default='XX:XX:XX:XX:XX:XX'),
                'radius-mac-mode': KeyInfo(default='as-username'),
                'static-algo-0': KeyInfo(default='none'),
                'static-algo-1': KeyInfo(default='none'),
                'static-algo-2': KeyInfo(default='none'),
                'static-algo-3': KeyInfo(default='none'),
                'static-key-0': KeyInfo(),
                'static-key-1': KeyInfo(),
                'static-key-2': KeyInfo(),
                'static-key-3': KeyInfo(),
                'static-sta-private-algo': KeyInfo(default='none'),
                'static-sta-private-key': KeyInfo(),
                'static-transmit-key': KeyInfo(),
                'supplicant-identity': KeyInfo(default='MikroTik'),
                'tls-certificate': KeyInfo(default='none'),
                'tls-mode': KeyInfo(),
                'unicast-ciphers': KeyInfo(),
                'wpa-pre-shared-key': KeyInfo(),
                'wpa2-pre-shared-key': KeyInfo(),
            },
        ),
    ),
    ('interface', 'wireless', 'sniffer'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'channel-time': KeyInfo(default='200ms'),
                'file-limit': KeyInfo(default=10),
                'file-name': KeyInfo(default=''),
                'memory-limit': KeyInfo(default=10),
                'multiple-channels': KeyInfo(default=False),
                'only-headers': KeyInfo(default=False),
                'receive-errors': KeyInfo(default=False),
                'streaming-enabled': KeyInfo(default=False),
                'streaming-max-rate': KeyInfo(default=0),
                'streaming-server': KeyInfo(default='0.0.0.0'),
            },
        ),
    ),
    ('interface', 'wireless', 'snooper'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'channel-time': KeyInfo(default='200ms'),
                'multiple-channels': KeyInfo(default=True),
                'receive-errors': KeyInfo(default=False),
            },
        ),
    ),
    ('iot', 'modbus'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'disabled': KeyInfo(default=True),
                'hardware-port': KeyInfo(default='modbus'),
                'tcp-port': KeyInfo(default=502),
                'timeout': KeyInfo(default=1000),
            },
        ),
    ),
    ('ip', 'accounting'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'account-local-traffic': KeyInfo(default=False),
                'enabled': KeyInfo(default=False),
                'threshold': KeyInfo(default=256),
            },
        ),
    ),
    ('ip', 'accounting', 'web-access'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'accessible-via-web': KeyInfo(default=False),
                'address': KeyInfo(default='0.0.0.0/0'),
            },
        ),
    ),
    ('ip', 'address'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('address', 'interface', ),
            fields={
                'address': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'interface': KeyInfo(),
                'network': KeyInfo(automatically_computed_from=('address', )),
            },
        ),
    ),
    ('ip', 'arp'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'address': KeyInfo(default='0.0.0.0'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'interface': KeyInfo(required=True),
                'mac-address': KeyInfo(default='00:00:00:00:00:00'),
                'published': KeyInfo(default=False),
            },
        ),
    ),
    ('ip', 'cloud'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            versioned_fields=[
                ([('7.17', '<')], 'ddns-enabled', KeyInfo(default=False)),
                ([('7.17', '>=')], 'ddns-enabled', KeyInfo(default='auto')),
            ],
            fields={
                'ddns-update-interval': KeyInfo(default='none'),
                'update-time': KeyInfo(default=True),
            },
        ),
    ),
    ('ip', 'cloud', 'advanced'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'use-local-address': KeyInfo(default=False),
            },
        ),
    ),
    ('ip', 'dhcp-client'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('interface', ),
            fields={
                'add-default-route': KeyInfo(default=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'default-route-distance': KeyInfo(default=1),
                'dhcp-options': KeyInfo(default='hostname,clientid', can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'interface': KeyInfo(),
                'script': KeyInfo(can_disable=True),
                'use-peer-dns': KeyInfo(default=True),
                'use-peer-ntp': KeyInfo(default=True),
            },
        ),
    ),
    ('ip', 'dhcp-relay'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name',),
            fields={
                'add-relay-info': KeyInfo(default=False),
                'delay-threshold': KeyInfo(can_disable=True, remove_value='none'),
                'dhcp-server': KeyInfo(required=True),
                'disabled': KeyInfo(default=False),
                'interface': KeyInfo(required=True),
                'local-address': KeyInfo(can_disable=True, remove_value='0.0.0.0'),
                'name': KeyInfo(),
                'relay-info-remote-id': KeyInfo(),
            },
        ),
    ),
    ('ip', 'dhcp-server'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'address-pool': KeyInfo(default='static-only'),
                'allow-dual-stack-queue': KeyInfo(can_disable=True, remove_value=True),
                'always-broadcast': KeyInfo(can_disable=True, remove_value=False),
                'authoritative': KeyInfo(default=True),
                'bootp-lease-time': KeyInfo(default='forever'),
                'bootp-support': KeyInfo(can_disable=True, remove_value='static'),
                'client-mac-limit': KeyInfo(can_disable=True, remove_value='unlimited'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'conflict-detection': KeyInfo(can_disable=True, remove_value=True),
                'delay-threshold': KeyInfo(can_disable=True, remove_value='none'),
                'dhcp-option-set': KeyInfo(can_disable=True, remove_value='none'),
                'disabled': KeyInfo(default=False),
                'insert-queue-before': KeyInfo(can_disable=True, remove_value='first'),
                'interface': KeyInfo(required=True),
                'lease-script': KeyInfo(default=''),
                'lease-time': KeyInfo(default='10m'),
                'name': KeyInfo(),
                'parent-queue': KeyInfo(can_disable=True, remove_value='none'),
                'relay': KeyInfo(can_disable=True, remove_value='0.0.0.0'),
                'server-address': KeyInfo(can_disable=True, remove_value='0.0.0.0'),
                'use-framed-as-classless': KeyInfo(can_disable=True, remove_value=True),
                'use-radius': KeyInfo(default=False),
            },
        ),
    ),
    ('ip', 'dhcp-server', 'config'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'accounting': KeyInfo(default=True),
                'interim-update': KeyInfo(default='0s'),
                'store-leases-disk': KeyInfo(default='5m'),
            },
        ),
    ),
    ('ip', 'dhcp-server', 'lease'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('server', 'address', ),
            fields={
                'address': KeyInfo(),
                'address-lists': KeyInfo(default=''),
                'always-broadcast': KeyInfo(),
                'client-id': KeyInfo(can_disable=True, remove_value=''),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'dhcp-option': KeyInfo(default=''),
                'disabled': KeyInfo(default=False),
                'insert-queue-before': KeyInfo(can_disable=True),
                'mac-address': KeyInfo(can_disable=True, remove_value=''),
                'server': KeyInfo(absent_value='all'),
            },
        ),
    ),
    ('ip', 'dhcp-server', 'network'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('address', ),
            fields={
                'address': KeyInfo(),
                'boot-file-name': KeyInfo(default=''),
                'caps-manager': KeyInfo(default=''),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'dhcp-option': KeyInfo(default=''),
                'dhcp-option-set': KeyInfo(default=''),
                'dns-none': KeyInfo(default=False),
                'dns-server': KeyInfo(default=''),
                'domain': KeyInfo(default=''),
                'gateway': KeyInfo(default=''),
                'netmask': KeyInfo(can_disable=True, remove_value=0),
                'next-server': KeyInfo(can_disable=True),
                'ntp-server': KeyInfo(default=''),
                'wins-server': KeyInfo(default=''),
            },
        ),
    ),
    ('ip', 'dhcp-server', 'option'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name',),
            versioned_fields=[
                ([('7.16', '>=')], 'comment', KeyInfo(can_disable=True, remove_value='')),
            ],
            fields={
                'code': KeyInfo(required=True),
                'name': KeyInfo(),
                'value': KeyInfo(default=''),
                'force': KeyInfo(default=False),
            },
        ),
    ),
    ('ip', 'dhcp-server', 'option', 'sets'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name',),
            versioned_fields=[
                ([('7.16', '>=')], 'comment', KeyInfo(can_disable=True, remove_value='')),
            ],
            fields={
                'name': KeyInfo(required=True),
                'options': KeyInfo(),
            },
        ),
    ),
    ('ip', 'dhcp-server', 'matcher'): APIData(
        versioned=[
            ('7.4', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('name', ),
                versioned_fields=[
                    ([('7.16', '>=')], 'comment', KeyInfo(can_disable=True, remove_value='')),
                    ([('7.16', '>=')], 'matching-type', KeyInfo()),
                ],
                fields={
                    'address-pool': KeyInfo(default='none'),
                    'code': KeyInfo(required=True),
                    'disabled': KeyInfo(default=False),
                    'name': KeyInfo(required=True),
                    'option-set': KeyInfo(),
                    'server': KeyInfo(default='all'),
                    'value': KeyInfo(required=True),
                },
            )),
        ],
    ),
    ('ip', 'dns'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            versioned_fields=[
                ([('7.8', '>=')], 'doh-max-concurrent-queries', KeyInfo(default=50)),
                ([('7.8', '>=')], 'doh-max-server-connections', KeyInfo(default=5)),
                ([('7.8', '>=')], 'doh-timeout', KeyInfo(default='5s')),
                ([('7.16', '>=')], 'mdns-repeat-ifaces', KeyInfo()),
            ],
            fields={
                'allow-remote-requests': KeyInfo(),
                'cache-max-ttl': KeyInfo(default='1w'),
                'cache-size': KeyInfo(default='2048KiB'),
                'max-concurrent-queries': KeyInfo(default=100),
                'max-concurrent-tcp-sessions': KeyInfo(default=20),
                'max-udp-packet-size': KeyInfo(default=4096),
                'query-server-timeout': KeyInfo(default='2s'),
                'query-total-timeout': KeyInfo(default='10s'),
                'servers': KeyInfo(default=''),
                'use-doh-server': KeyInfo(default=''),
                'verify-doh-cert': KeyInfo(default=False),
            },
        ),
    ),
    ('ip', 'dns', 'adlist'): APIData(
        versioned=[
            ('7.15', '>=', VersionedAPIData(
                fully_understood=True,
                fields={
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'file': KeyInfo(default=''),
                    'match-count': KeyInfo(read_only=True),
                    'name-count': KeyInfo(read_only=True),
                    'ssl-verify': KeyInfo(default=True),
                    'url': KeyInfo(default=''),
                },
            )),
        ],
    ),
    ('ip', 'dns', 'forwarders'): APIData(
        versioned=[
            ('7.17', '>=', VersionedAPIData(
                fully_understood=True,
                required_one_of=[['dns-servers', 'doh-servers']],
                fields={
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'dns-servers': KeyInfo(default=''),
                    'doh-servers': KeyInfo(default=''),
                    'name': KeyInfo(required=True),
                    'verify-doh-cert': KeyInfo(default=True),
                },
            )),
        ],
    ),
    ('ip', 'dns', 'static'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            required_one_of=[['name', 'regexp']],
            mutually_exclusive=[['name', 'regexp']],
            versioned_fields=[
                ([('7.5', '>=')], 'address-list', KeyInfo()),
                ([('7.5', '>=')], 'match-subdomain', KeyInfo(default=False)),
            ],
            fields={
                'address': KeyInfo(),
                'cname': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'forward-to': KeyInfo(),
                'mx-exchange': KeyInfo(),
                'mx-preference': KeyInfo(),
                'name': KeyInfo(),
                'ns': KeyInfo(),
                'regexp': KeyInfo(),
                'srv-port': KeyInfo(),
                'srv-priority': KeyInfo(),
                'srv-target': KeyInfo(),
                'srv-weight': KeyInfo(),
                'text': KeyInfo(),
                'ttl': KeyInfo(default='1d'),
                'type': KeyInfo(),
            },
        ),
    ),
    ('ip', 'firewall', 'address-list'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('address', 'list', ),
            fields={
                'address': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'list': KeyInfo(),
            },
        ),
    ),
    ('ip', 'firewall', 'filter'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            stratify_keys=('chain', ),
            fields={
                'action': KeyInfo(),
                'address-list': KeyInfo(can_disable=True),
                'address-list-timeout': KeyInfo(can_disable=True),
                'chain': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'connection-bytes': KeyInfo(can_disable=True),
                'connection-limit': KeyInfo(can_disable=True),
                'connection-mark': KeyInfo(can_disable=True),
                'connection-nat-state': KeyInfo(can_disable=True),
                'connection-rate': KeyInfo(can_disable=True),
                'connection-state': KeyInfo(can_disable=True),
                'connection-type': KeyInfo(can_disable=True),
                'content': KeyInfo(can_disable=True),
                'disabled': KeyInfo(default=False),
                'dscp': KeyInfo(can_disable=True),
                'dst-address': KeyInfo(can_disable=True),
                'dst-address-list': KeyInfo(can_disable=True),
                'dst-address-type': KeyInfo(can_disable=True),
                'dst-limit': KeyInfo(can_disable=True),
                'dst-port': KeyInfo(can_disable=True),
                'fragment': KeyInfo(can_disable=True),
                'hotspot': KeyInfo(can_disable=True),
                'hw-offload': KeyInfo(can_disable=True),
                'icmp-options': KeyInfo(can_disable=True),
                'in-bridge-port': KeyInfo(can_disable=True),
                'in-bridge-port-list': KeyInfo(can_disable=True),
                'in-interface': KeyInfo(can_disable=True),
                'in-interface-list': KeyInfo(can_disable=True),
                'ingress-priority': KeyInfo(can_disable=True),
                'ipsec-policy': KeyInfo(can_disable=True),
                'ipv4-options': KeyInfo(can_disable=True),
                'jump-target': KeyInfo(can_disable=True),
                'layer7-protocol': KeyInfo(can_disable=True),
                'limit': KeyInfo(can_disable=True),
                'log': KeyInfo(default=False),
                'log-prefix': KeyInfo(default=''),
                'nth': KeyInfo(can_disable=True),
                'out-bridge-port': KeyInfo(can_disable=True),
                'out-bridge-port-list': KeyInfo(can_disable=True),
                'out-interface': KeyInfo(can_disable=True),
                'out-interface-list': KeyInfo(can_disable=True),
                'p2p': KeyInfo(can_disable=True),
                'packet-mark': KeyInfo(can_disable=True),
                'packet-size': KeyInfo(can_disable=True),
                'per-connection-classifier': KeyInfo(can_disable=True),
                'port': KeyInfo(can_disable=True),
                'priority': KeyInfo(can_disable=True),
                'protocol': KeyInfo(can_disable=True),
                'psd': KeyInfo(can_disable=True),
                'random': KeyInfo(can_disable=True),
                'realm': KeyInfo(can_disable=True),
                'reject-with': KeyInfo(can_disable=True),
                'routing-mark': KeyInfo(can_disable=True),
                'routing-table': KeyInfo(can_disable=True),
                'src-address': KeyInfo(can_disable=True),
                'src-address-list': KeyInfo(can_disable=True),
                'src-address-type': KeyInfo(can_disable=True),
                'src-mac-address': KeyInfo(can_disable=True),
                'src-port': KeyInfo(can_disable=True),
                'tcp-flags': KeyInfo(can_disable=True),
                'tcp-mss': KeyInfo(can_disable=True),
                'time': KeyInfo(can_disable=True),
                'tls-host': KeyInfo(can_disable=True),
                'ttl': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('ip', 'firewall', 'mangle'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            stratify_keys=('chain', ),
            fields={
                'action': KeyInfo(),
                'address-list': KeyInfo(can_disable=True),
                'address-list-timeout': KeyInfo(can_disable=True),
                'chain': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'connection-bytes': KeyInfo(can_disable=True),
                'connection-limit': KeyInfo(can_disable=True),
                'connection-mark': KeyInfo(can_disable=True),
                'connection-nat-state': KeyInfo(can_disable=True),
                'connection-rate': KeyInfo(can_disable=True),
                'connection-state': KeyInfo(can_disable=True),
                'connection-type': KeyInfo(can_disable=True),
                'content': KeyInfo(can_disable=True),
                'disabled': KeyInfo(default=False),
                'dscp': KeyInfo(can_disable=True),
                'dst-address': KeyInfo(can_disable=True),
                'dst-address-list': KeyInfo(can_disable=True),
                'dst-address-type': KeyInfo(can_disable=True),
                'dst-limit': KeyInfo(can_disable=True),
                'dst-port': KeyInfo(can_disable=True),
                'fragment': KeyInfo(can_disable=True),
                'hotspot': KeyInfo(can_disable=True),
                'icmp-options': KeyInfo(can_disable=True),
                'in-bridge-port': KeyInfo(can_disable=True),
                'in-bridge-port-list': KeyInfo(can_disable=True),
                'in-interface': KeyInfo(can_disable=True),
                'in-interface-list': KeyInfo(can_disable=True),
                'ingress-priority': KeyInfo(can_disable=True),
                'ipsec-policy': KeyInfo(can_disable=True),
                'ipv4-options': KeyInfo(can_disable=True),
                'jump-target': KeyInfo(can_disable=True),
                'layer7-protocol': KeyInfo(can_disable=True),
                'limit': KeyInfo(can_disable=True),
                'log': KeyInfo(default=False),
                'log-prefix': KeyInfo(default=''),
                'new-connection-mark': KeyInfo(can_disable=True),
                'new-dscp': KeyInfo(can_disable=True),
                'new-mss': KeyInfo(can_disable=True),
                'new-packet-mark': KeyInfo(can_disable=True),
                'new-priority': KeyInfo(can_disable=True),
                'new-routing-mark': KeyInfo(can_disable=True),
                'new-ttl': KeyInfo(can_disable=True),
                'nth': KeyInfo(can_disable=True),
                'out-bridge-port': KeyInfo(can_disable=True),
                'out-bridge-port-list': KeyInfo(can_disable=True),
                'out-interface': KeyInfo(can_disable=True),
                'out-interface-list': KeyInfo(can_disable=True),
                'p2p': KeyInfo(can_disable=True),
                'packet-mark': KeyInfo(can_disable=True),
                'packet-size': KeyInfo(can_disable=True),
                'passthrough': KeyInfo(can_disable=True),
                'per-connection-classifier': KeyInfo(can_disable=True),
                'port': KeyInfo(can_disable=True),
                'priority': KeyInfo(can_disable=True),
                'protocol': KeyInfo(can_disable=True),
                'psd': KeyInfo(can_disable=True),
                'random': KeyInfo(can_disable=True),
                'realm': KeyInfo(can_disable=True),
                'route-dst': KeyInfo(can_disable=True),
                'routing-mark': KeyInfo(can_disable=True),
                'routing-table': KeyInfo(can_disable=True),
                'sniff-id': KeyInfo(can_disable=True),
                'sniff-target': KeyInfo(can_disable=True),
                'sniff-target-port': KeyInfo(can_disable=True),
                'src-address': KeyInfo(can_disable=True),
                'src-address-list': KeyInfo(can_disable=True),
                'src-address-type': KeyInfo(can_disable=True),
                'src-mac-address': KeyInfo(can_disable=True),
                'src-port': KeyInfo(can_disable=True),
                'tcp-flags': KeyInfo(can_disable=True),
                'tcp-mss': KeyInfo(can_disable=True),
                'time': KeyInfo(can_disable=True),
                'tls-host': KeyInfo(can_disable=True),
                'ttl': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('ip', 'firewall', 'nat'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            stratify_keys=('chain', ),
            fields={
                'action': KeyInfo(),
                'address-list': KeyInfo(can_disable=True),
                'address-list-timeout': KeyInfo(can_disable=True),
                'chain': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'connection-bytes': KeyInfo(can_disable=True),
                'connection-limit': KeyInfo(can_disable=True),
                'connection-mark': KeyInfo(can_disable=True),
                'connection-rate': KeyInfo(can_disable=True),
                'connection-type': KeyInfo(can_disable=True),
                'content': KeyInfo(can_disable=True),
                'disabled': KeyInfo(default=False),
                'dscp': KeyInfo(can_disable=True),
                'dst-address': KeyInfo(can_disable=True),
                'dst-address-list': KeyInfo(can_disable=True),
                'dst-address-type': KeyInfo(can_disable=True),
                'dst-limit': KeyInfo(can_disable=True),
                'dst-port': KeyInfo(can_disable=True),
                'fragment': KeyInfo(can_disable=True),
                'hotspot': KeyInfo(can_disable=True),
                'icmp-options': KeyInfo(can_disable=True),
                'in-bridge-port': KeyInfo(can_disable=True),
                'in-bridge-port-list': KeyInfo(can_disable=True),
                'in-interface': KeyInfo(can_disable=True),
                'in-interface-list': KeyInfo(can_disable=True),
                'ingress-priority': KeyInfo(can_disable=True),
                'ipsec-policy': KeyInfo(can_disable=True),
                'ipv4-options': KeyInfo(can_disable=True),
                'jump-target': KeyInfo(can_disable=True),
                'layer7-protocol': KeyInfo(can_disable=True),
                'limit': KeyInfo(can_disable=True),
                'log': KeyInfo(default=False),
                'log-prefix': KeyInfo(default=''),
                'nth': KeyInfo(can_disable=True),
                'out-bridge-port': KeyInfo(can_disable=True),
                'out-bridge-port-list': KeyInfo(can_disable=True),
                'out-interface': KeyInfo(can_disable=True),
                'out-interface-list': KeyInfo(can_disable=True),
                'packet-mark': KeyInfo(can_disable=True),
                'packet-size': KeyInfo(can_disable=True),
                'per-connection-classifier': KeyInfo(can_disable=True),
                'port': KeyInfo(can_disable=True),
                'priority': KeyInfo(can_disable=True),
                'protocol': KeyInfo(can_disable=True),
                'psd': KeyInfo(can_disable=True),
                'random': KeyInfo(can_disable=True),
                'randomise-ports': KeyInfo(can_disable=True),
                'realm': KeyInfo(can_disable=True),
                'routing-mark': KeyInfo(can_disable=True),
                'same-not-by-dst': KeyInfo(can_disable=True),
                'src-address': KeyInfo(can_disable=True),
                'src-address-list': KeyInfo(can_disable=True),
                'src-address-type': KeyInfo(can_disable=True),
                'src-mac-address': KeyInfo(can_disable=True),
                'src-port': KeyInfo(can_disable=True),
                'tcp-mss': KeyInfo(can_disable=True),
                'time': KeyInfo(can_disable=True),
                'tls-host': KeyInfo(can_disable=True),
                'to-addresses': KeyInfo(can_disable=True),
                'to-ports': KeyInfo(can_disable=True),
                'ttl': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('ip', 'firewall', 'raw'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            stratify_keys=('chain',),
            fields={
                'action': KeyInfo(),
                'address-list': KeyInfo(can_disable=True),
                'address-list-timeout': KeyInfo(can_disable=True),
                'chain': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'content': KeyInfo(can_disable=True),
                'disabled': KeyInfo(default=False),
                'dscp': KeyInfo(can_disable=True),
                'dst-address': KeyInfo(can_disable=True),
                'dst-address-list': KeyInfo(can_disable=True),
                'dst-address-type': KeyInfo(can_disable=True),
                'dst-limit': KeyInfo(can_disable=True),
                'dst-port': KeyInfo(can_disable=True),
                'fragment': KeyInfo(can_disable=True),
                'hotspot': KeyInfo(can_disable=True),
                'icmp-options': KeyInfo(can_disable=True),
                'in-bridge-port': KeyInfo(can_disable=True),
                'in-bridge-port-list': KeyInfo(can_disable=True),
                'in-interface': KeyInfo(can_disable=True),
                'in-interface-list': KeyInfo(can_disable=True),
                'ingress-priority': KeyInfo(can_disable=True),
                'ipsec-policy': KeyInfo(can_disable=True),
                'ipv4-options': KeyInfo(can_disable=True),
                'jump-target': KeyInfo(can_disable=True),
                'limit': KeyInfo(can_disable=True),
                'log': KeyInfo(default=False),
                'log-prefix': KeyInfo(default=''),
                'nth': KeyInfo(can_disable=True),
                'out-bridge-port': KeyInfo(can_disable=True),
                'out-bridge-port-list': KeyInfo(can_disable=True),
                'out-interface': KeyInfo(can_disable=True),
                'out-interface-list': KeyInfo(can_disable=True),
                'packet-mark': KeyInfo(can_disable=True),
                'packet-size': KeyInfo(can_disable=True),
                'per-connection-classifier': KeyInfo(can_disable=True),
                'port': KeyInfo(can_disable=True),
                'priority': KeyInfo(can_disable=True),
                'protocol': KeyInfo(can_disable=True),
                'psd': KeyInfo(can_disable=True),
                'random': KeyInfo(can_disable=True),
                'src-address': KeyInfo(can_disable=True),
                'src-address-list': KeyInfo(can_disable=True),
                'src-address-type': KeyInfo(can_disable=True),
                'src-mac-address': KeyInfo(can_disable=True),
                'src-port': KeyInfo(can_disable=True),
                'tcp-flags': KeyInfo(can_disable=True),
                'tcp-mss': KeyInfo(can_disable=True),
                'time': KeyInfo(can_disable=True),
                'tls-host': KeyInfo(can_disable=True),
                'ttl': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('ip', 'hotspot', 'user'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(),
                'name': KeyInfo(),
            },
        ),
    ),
    ('ip', 'ipsec', 'settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'accounting': KeyInfo(default=True),
                'interim-update': KeyInfo(default='0s'),
                'xauth-use-radius': KeyInfo(default=False),
            },
        ),
    ),
    ('ip', 'proxy'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'always-from-cache': KeyInfo(default=False),
                'anonymous': KeyInfo(default=False),
                'cache-administrator': KeyInfo(default='webmaster'),
                'cache-hit-dscp': KeyInfo(default=4),
                'cache-on-disk': KeyInfo(default=False),
                'cache-path': KeyInfo(default='web-proxy'),
                'enabled': KeyInfo(default=False),
                'max-cache-object-size': KeyInfo(default='2048KiB'),
                'max-cache-size': KeyInfo(default='unlimited'),
                'max-client-connections': KeyInfo(default=600),
                'max-fresh-time': KeyInfo(default='3d'),
                'max-server-connections': KeyInfo(default=600),
                'parent-proxy': KeyInfo(default='::'),
                'parent-proxy-port': KeyInfo(default=0),
                'port': KeyInfo(default=8080),
                'serialize-connections': KeyInfo(default=False),
                'src-address': KeyInfo(default='::'),
            },
        ),
    ),
    ('ip', 'smb'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allow-guests': KeyInfo(default=True),
                'comment': KeyInfo(default='MikrotikSMB'),
                'domain': KeyInfo(default='MSHOME'),
                'enabled': KeyInfo(default=False),
                'interfaces': KeyInfo(default='all'),
            },
        ),
    ),
    ('ip', 'smb', 'shares'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'directory': KeyInfo(),
                'disabled': KeyInfo(),
                'max-sessions': KeyInfo(),
                'name': KeyInfo(),
            },
        ),
    ),
    ('ip', 'smb', 'users'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'disabled': KeyInfo(),
                'name': KeyInfo(),
                'password': KeyInfo(),
                'read-only': KeyInfo(),
            },
        ),
    ),
    ('ip', 'socks'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'auth-method': KeyInfo(default='none'),
                'connection-idle-timeout': KeyInfo(default='2m'),
                'enabled': KeyInfo(default=False),
                'max-connections': KeyInfo(default=200),
                'port': KeyInfo(default=1080),
                'version': KeyInfo(default=4),
            },
        ),
    ),
    ('ip', 'ssh'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            versioned_fields=[
                ([('7.9', '>=')], 'host-key-type', KeyInfo(default='rsa')),
            ],
            fields={
                'allow-none-crypto': KeyInfo(default=False),
                'always-allow-password-login': KeyInfo(default=False),
                'forwarding-enabled': KeyInfo(default=False),
                'host-key-size': KeyInfo(default=2048),
                'strong-crypto': KeyInfo(default=False),
            },
        ),
    ),
    ('ip', 'tftp', 'settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'max-block-size': KeyInfo(default=4096),
            },
        ),
    ),
    ('ip', 'traffic-flow'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'active-flow-timeout': KeyInfo(default='30m'),
                'cache-entries': KeyInfo(default='32k'),
                'enabled': KeyInfo(default=False),
                'inactive-flow-timeout': KeyInfo(default='15s'),
                'interfaces': KeyInfo(default='all'),
                'packet-sampling': KeyInfo(default=False),
                'sampling-interval': KeyInfo(default=0),
                'sampling-space': KeyInfo(default=0),
            },
        ),
    ),
    ('ip', 'traffic-flow', 'ipfix'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'bytes': KeyInfo(default=True),
                'dst-address': KeyInfo(default=True),
                'dst-address-mask': KeyInfo(default=True),
                'dst-mac-address': KeyInfo(default=True),
                'dst-port': KeyInfo(default=True),
                'first-forwarded': KeyInfo(default=True),
                'gateway': KeyInfo(default=True),
                'icmp-code': KeyInfo(default=True),
                'icmp-type': KeyInfo(default=True),
                'igmp-type': KeyInfo(default=True),
                'in-interface': KeyInfo(default=True),
                'ip-header-length': KeyInfo(default=True),
                'ip-total-length': KeyInfo(default=True),
                'ipv6-flow-label': KeyInfo(default=True),
                'is-multicast': KeyInfo(default=True),
                'last-forwarded': KeyInfo(default=True),
                'nat-dst-address': KeyInfo(default=True),
                'nat-dst-port': KeyInfo(default=True),
                'nat-events': KeyInfo(default=False),
                'nat-src-address': KeyInfo(default=True),
                'nat-src-port': KeyInfo(default=True),
                'out-interface': KeyInfo(default=True),
                'packets': KeyInfo(default=True),
                'protocol': KeyInfo(default=True),
                'src-address': KeyInfo(default=True),
                'src-address-mask': KeyInfo(default=True),
                'src-mac-address': KeyInfo(default=True),
                'src-port': KeyInfo(default=True),
                'sys-init-time': KeyInfo(default=True),
                'tcp-ack-num': KeyInfo(default=True),
                'tcp-flags': KeyInfo(default=True),
                'tcp-seq-num': KeyInfo(default=True),
                'tcp-window-size': KeyInfo(default=True),
                'tos': KeyInfo(default=True),
                'ttl': KeyInfo(default=True),
                'udp-length': KeyInfo(default=True),
            },
        ),
    ),
    ('ip', 'traffic-flow', 'target'): APIData(
        unversioned=VersionedAPIData(
            single_value=False,
            fully_understood=True,
            fields={
                'address': KeyInfo(),
                'disabled': KeyInfo(default=False),
                'dst-address': KeyInfo(),
                'port': KeyInfo(default=2055),
                'src-address': KeyInfo(),
                'v9-template-refresh': KeyInfo(default=20),
                'v9-template-timeout': KeyInfo(),
                'version': KeyInfo(),
            },
        ),
    ),
    ('ip', 'upnp'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allow-disable-external-interface': KeyInfo(default=False),
                'enabled': KeyInfo(default=False),
                'show-dummy-rule': KeyInfo(default=True),
            },
        ),
    ),
    ('ip', 'upnp', 'interfaces'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('interface', 'type'),
            fields={
                'disabled': KeyInfo(default=False),
                'interface': KeyInfo(),
                'type': KeyInfo(),
                'forced-ip': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('ipv6', 'dhcp-client'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('interface', 'request'),
            fields={
                'add-default-route': KeyInfo(default=False),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'default-route-distance': KeyInfo(default=1),
                'dhcp-options': KeyInfo(default=''),
                'disabled': KeyInfo(default=False),
                'interface': KeyInfo(),
                'pool-name': KeyInfo(required=True),
                'pool-prefix-length': KeyInfo(default=64),
                'prefix-hint': KeyInfo(default='::/0'),
                'request': KeyInfo(),
                'use-peer-dns': KeyInfo(default=True),
            },
            versioned_fields=[
                # Mikrotik does not provide exact version in official changelogs.
                # The 7.15 version is the earliest, found option in router config backups:
                ([('7.15', '>=')], 'script', KeyInfo(default='')),
                ([('7.15', '>=')], 'custom-duid', KeyInfo(default='')),
                ([('7.15', '>=')], 'use-interface-duid', KeyInfo(default=False)),
                ([('7.15', '>=')], 'validate-server-duid', KeyInfo(default=True)),
            ],
        ),
    ),
    ('ipv6', 'dhcp-server'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'address-pool': KeyInfo(required=True),
                'allow-dual-stack-queue': KeyInfo(can_disable=True, remove_value=True),
                'binding-script': KeyInfo(can_disable=True, remove_value=''),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'dhcp-option': KeyInfo(default=''),
                'disabled': KeyInfo(default=False),
                'insert-queue-before': KeyInfo(can_disable=True, remove_value='first'),
                'interface': KeyInfo(required=True),
                'lease-time': KeyInfo(default='3d'),
                'name': KeyInfo(),
                'parent-queue': KeyInfo(can_disable=True, remove_value='none'),
                'preference': KeyInfo(default=255),
                'rapid-commit': KeyInfo(default=True),
                'route-distance': KeyInfo(default=1),
                'use-radius': KeyInfo(default=False),
            },
        ),
    ),
    ('ipv6', 'dhcp-server', 'option'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name',),
            fields={
                'code': KeyInfo(required=True),
                'name': KeyInfo(),
                'value': KeyInfo(default=''),
            },
        ),
    ),
    ('ipv6', 'firewall', 'address-list'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('address', 'list', ),
            fields={
                'address': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'list': KeyInfo(),
            },
        ),
    ),
    ('ipv6', 'firewall', 'filter'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            stratify_keys=('chain', ),
            fields={
                'action': KeyInfo(),
                'address-list': KeyInfo(can_disable=True),
                'address-list-timeout': KeyInfo(can_disable=True),
                'chain': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'connection-bytes': KeyInfo(can_disable=True),
                'connection-limit': KeyInfo(can_disable=True),
                'connection-mark': KeyInfo(can_disable=True),
                'connection-rate': KeyInfo(can_disable=True),
                'connection-state': KeyInfo(can_disable=True),
                'connection-type': KeyInfo(can_disable=True),
                'content': KeyInfo(can_disable=True),
                'disabled': KeyInfo(default=False),
                'dscp': KeyInfo(can_disable=True),
                'dst-address': KeyInfo(can_disable=True),
                'dst-address-list': KeyInfo(can_disable=True),
                'dst-address-type': KeyInfo(can_disable=True),
                'dst-limit': KeyInfo(can_disable=True),
                'dst-port': KeyInfo(can_disable=True),
                'headers': KeyInfo(can_disable=True),
                'hop-limit': KeyInfo(can_disable=True),
                'icmp-options': KeyInfo(can_disable=True),
                'in-bridge-port': KeyInfo(can_disable=True),
                'in-bridge-port-list': KeyInfo(can_disable=True),
                'in-interface': KeyInfo(can_disable=True),
                'in-interface-list': KeyInfo(can_disable=True),
                'ingress-priority': KeyInfo(can_disable=True),
                'ipsec-policy': KeyInfo(can_disable=True),
                'jump-target': KeyInfo(can_disable=True),
                'limit': KeyInfo(can_disable=True),
                'log': KeyInfo(can_disable=False),
                'log-prefix': KeyInfo(can_disable=False),
                'nth': KeyInfo(can_disable=True),
                'out-bridge-port': KeyInfo(can_disable=True),
                'out-bridge-port-list': KeyInfo(can_disable=True),
                'out-interface': KeyInfo(can_disable=True),
                'out-interface-list': KeyInfo(can_disable=True),
                'packet-mark': KeyInfo(can_disable=True),
                'packet-size': KeyInfo(can_disable=True),
                'per-connection-classifier': KeyInfo(can_disable=True),
                'port': KeyInfo(can_disable=True),
                'priority': KeyInfo(can_disable=True),
                'protocol': KeyInfo(can_disable=True),
                'random': KeyInfo(can_disable=True),
                'reject-with': KeyInfo(),
                'src-address': KeyInfo(can_disable=True),
                'src-address-list': KeyInfo(can_disable=True),
                'src-address-type': KeyInfo(can_disable=True),
                'src-mac-address': KeyInfo(can_disable=True),
                'src-port': KeyInfo(can_disable=True),
                'tcp-flags': KeyInfo(can_disable=True),
                'tcp-mss': KeyInfo(can_disable=True),
                'time': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('ipv6', 'firewall', 'mangle'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            stratify_keys=('chain', ),
            fields={
                'action': KeyInfo(),
                'address-list': KeyInfo(),
                'address-list-timeout': KeyInfo(),
                'chain': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'connection-bytes': KeyInfo(can_disable=True),
                'connection-limit': KeyInfo(can_disable=True),
                'connection-mark': KeyInfo(can_disable=True),
                'connection-rate': KeyInfo(can_disable=True),
                'connection-state': KeyInfo(can_disable=True),
                'connection-type': KeyInfo(can_disable=True),
                'content': KeyInfo(can_disable=True),
                'disabled': KeyInfo(default=False),
                'dscp': KeyInfo(can_disable=True),
                'dst-address': KeyInfo(can_disable=True),
                'dst-address-list': KeyInfo(can_disable=True),
                'dst-address-type': KeyInfo(can_disable=True),
                'dst-limit': KeyInfo(can_disable=True),
                'dst-port': KeyInfo(can_disable=True),
                'dst-prefix': KeyInfo(),
                'headers': KeyInfo(can_disable=True),
                'hop-limit': KeyInfo(can_disable=True),
                'icmp-options': KeyInfo(can_disable=True),
                'in-bridge-port': KeyInfo(can_disable=True),
                'in-bridge-port-list': KeyInfo(can_disable=True),
                'in-interface': KeyInfo(can_disable=True),
                'in-interface-list': KeyInfo(can_disable=True),
                'ingress-priority': KeyInfo(can_disable=True),
                'ipsec-policy': KeyInfo(can_disable=True),
                'jump-target': KeyInfo(),
                'limit': KeyInfo(can_disable=True),
                'log': KeyInfo(),
                'log-prefix': KeyInfo(),
                'new-connection-mark': KeyInfo(),
                'new-dscp': KeyInfo(),
                'new-hop-limit': KeyInfo(),
                'new-mss': KeyInfo(),
                'new-packet-mark': KeyInfo(),
                'new-routing-mark': KeyInfo(),
                'nth': KeyInfo(can_disable=True),
                'out-bridge-port': KeyInfo(can_disable=True),
                'out-bridge-port-list': KeyInfo(can_disable=True),
                'out-interface': KeyInfo(can_disable=True),
                'out-interface-list': KeyInfo(can_disable=True),
                'packet-mark': KeyInfo(can_disable=True),
                'packet-size': KeyInfo(can_disable=True),
                'passthrough': KeyInfo(),
                'per-connection-classifier': KeyInfo(can_disable=True),
                'port': KeyInfo(can_disable=True),
                'priority': KeyInfo(can_disable=True),
                'protocol': KeyInfo(can_disable=True),
                'random': KeyInfo(can_disable=True),
                'routing-mark': KeyInfo(can_disable=True),
                'sniff-id': KeyInfo(),
                'sniff-target': KeyInfo(),
                'sniff-target-port': KeyInfo(),
                'src-address': KeyInfo(can_disable=True),
                'src-address-list': KeyInfo(can_disable=True),
                'src-address-type': KeyInfo(can_disable=True),
                'src-mac-address': KeyInfo(can_disable=True),
                'src-port': KeyInfo(can_disable=True),
                'src-prefix': KeyInfo(),
                'tcp-flags': KeyInfo(can_disable=True),
                'tcp-mss': KeyInfo(can_disable=True),
                'time': KeyInfo(can_disable=True),
                'tls-host': KeyInfo(can_disable=True),
            }
        ),
    ),
    ('ipv6', 'firewall', 'nat'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            stratify_keys=('chain', ),
            fields={
                'action': KeyInfo(),
                'address-list': KeyInfo(),
                'address-list-timeout': KeyInfo(),
                'chain': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'connection-bytes': KeyInfo(can_disable=True),
                'connection-limit': KeyInfo(can_disable=True),
                'connection-mark': KeyInfo(can_disable=True),
                'connection-rate': KeyInfo(can_disable=True),
                'connection-state': KeyInfo(can_disable=True),
                'connection-type': KeyInfo(can_disable=True),
                'content': KeyInfo(can_disable=True),
                'disabled': KeyInfo(default=False),
                'dscp': KeyInfo(can_disable=True),
                'dst-address': KeyInfo(can_disable=True),
                'dst-address-list': KeyInfo(can_disable=True),
                'dst-address-type': KeyInfo(can_disable=True),
                'dst-limit': KeyInfo(can_disable=True),
                'dst-port': KeyInfo(can_disable=True),
                'icmp-options': KeyInfo(can_disable=True),
                'in-bridge-port': KeyInfo(can_disable=True),
                'in-bridge-port-list': KeyInfo(can_disable=True),
                'in-interface': KeyInfo(can_disable=True),
                'in-interface-list': KeyInfo(can_disable=True),
                'ingress-priority': KeyInfo(can_disable=True),
                'ipsec-policy': KeyInfo(can_disable=True),
                'jump-target': KeyInfo(),
                'layer7-protocol': KeyInfo(can_disable=True),
                'limit': KeyInfo(can_disable=True),
                'log': KeyInfo(),
                'log-prefix': KeyInfo(),
                'out-bridge-port': KeyInfo(can_disable=True),
                'out-bridge-port-list': KeyInfo(can_disable=True),
                'out-interface': KeyInfo(can_disable=True),
                'out-interface-list': KeyInfo(can_disable=True),
                'packet-mark': KeyInfo(can_disable=True),
                'packet-size': KeyInfo(can_disable=True),
                'per-connection-classifier': KeyInfo(can_disable=True),
                'port': KeyInfo(can_disable=True),
                'priority': KeyInfo(can_disable=True),
                'protocol': KeyInfo(can_disable=True),
                'random': KeyInfo(can_disable=True),
                'routing-mark': KeyInfo(can_disable=True),
                'src-address': KeyInfo(can_disable=True),
                'src-address-list': KeyInfo(can_disable=True),
                'src-address-type': KeyInfo(can_disable=True),
                'src-mac-address': KeyInfo(can_disable=True),
                'src-port': KeyInfo(can_disable=True),
                'tcp-flags': KeyInfo(can_disable=True),
                'tcp-mss': KeyInfo(can_disable=True),
                'time': KeyInfo(can_disable=True),
                'tls-host': KeyInfo(can_disable=True),
                'to-addresses': KeyInfo(can_disable=True),
                'to-ports': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('ipv6', 'firewall', 'raw'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            stratify_keys=('chain',),
            fields={
                'action': KeyInfo(),
                'address-list': KeyInfo(),
                'address-list-timeout': KeyInfo(),
                'chain': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'content': KeyInfo(can_disable=True),
                'disabled': KeyInfo(default=False),
                'dscp': KeyInfo(can_disable=True),
                'dst-address': KeyInfo(can_disable=True),
                'dst-address-list': KeyInfo(can_disable=True),
                'dst-address-type': KeyInfo(can_disable=True),
                'dst-limit': KeyInfo(can_disable=True),
                'dst-port': KeyInfo(can_disable=True),
                'headers': KeyInfo(can_disable=True),
                'hop-limit': KeyInfo(can_disable=True),
                'icmp-options': KeyInfo(can_disable=True),
                'in-bridge-port': KeyInfo(can_disable=True),
                'in-bridge-port-list': KeyInfo(can_disable=True),
                'in-interface': KeyInfo(can_disable=True),
                'in-interface-list': KeyInfo(can_disable=True),
                'ingress-priority': KeyInfo(can_disable=True),
                'ipsec-policy': KeyInfo(can_disable=True),
                'jump-target': KeyInfo(),
                'limit': KeyInfo(can_disable=True),
                'log': KeyInfo(),
                'log-prefix': KeyInfo(),
                'nth': KeyInfo(can_disable=True),
                'out-bridge-port': KeyInfo(can_disable=True),
                'out-bridge-port-list': KeyInfo(can_disable=True),
                'out-interface': KeyInfo(can_disable=True),
                'out-interface-list': KeyInfo(can_disable=True),
                'packet-mark': KeyInfo(can_disable=True),
                'packet-size': KeyInfo(can_disable=True),
                'per-connection-classifier': KeyInfo(can_disable=True),
                'port': KeyInfo(can_disable=True),
                'priority': KeyInfo(can_disable=True),
                'protocol': KeyInfo(can_disable=True),
                'random': KeyInfo(can_disable=True),
                'src-address': KeyInfo(can_disable=True),
                'src-address-list': KeyInfo(can_disable=True),
                'src-address-type': KeyInfo(can_disable=True),
                'src-mac-address': KeyInfo(can_disable=True),
                'src-port': KeyInfo(can_disable=True),
                'tcp-flags': KeyInfo(can_disable=True),
                'tcp-mss': KeyInfo(can_disable=True),
                'time': KeyInfo(can_disable=True),
                'tls-host': KeyInfo(can_disable=True),
            }
        ),
    ),
    ('ipv6', 'nd'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('interface', ),
            fields={
                'advertise-dns': KeyInfo(default=True),
                'advertise-mac-address': KeyInfo(default=True),
                'disabled': KeyInfo(default=False),
                'dns': KeyInfo(default=''),
                'hop-limit': KeyInfo(default='unspecified'),
                'interface': KeyInfo(),
                'managed-address-configuration': KeyInfo(default=False),
                'mtu': KeyInfo(default='unspecified'),
                'other-configuration': KeyInfo(default=False),
                'ra-delay': KeyInfo(default='3s'),
                'ra-interval': KeyInfo(default='3m20s-10m'),
                'ra-lifetime': KeyInfo(default='30m'),
                'ra-preference': KeyInfo(default='medium'),
                'reachable-time': KeyInfo(default='unspecified'),
                'retransmit-interval': KeyInfo(default='unspecified'),
            },
        ),
    ),
    ('ipv6', 'nd', 'prefix'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                '6to4-interface': KeyInfo(default='none'),
                'autonomous': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'interface': KeyInfo(required=True),
                'on-link': KeyInfo(default=True),
                'preferred-lifetime': KeyInfo(),
                'prefix': KeyInfo(),
                'valid-lifetime': KeyInfo(),
            },
        ),
    ),
    ('ipv6', 'nd', 'prefix', 'default'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'autonomous': KeyInfo(default=True),
                'preferred-lifetime': KeyInfo(default='1w'),
                'valid-lifetime': KeyInfo(default='4w2d'),
            },
        ),
    ),
    ('ipv6', 'route'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'bgp-as-path': KeyInfo(can_disable=True),
                'bgp-atomic-aggregate': KeyInfo(can_disable=True),
                'bgp-communities': KeyInfo(can_disable=True),
                'bgp-local-pref': KeyInfo(can_disable=True),
                'bgp-med': KeyInfo(can_disable=True),
                'bgp-origin': KeyInfo(can_disable=True),
                'bgp-prepend': KeyInfo(can_disable=True),
                'type': KeyInfo(can_disable=True, remove_value='unicast'),
                'blackhole': KeyInfo(can_disable=True),
                'check-gateway': KeyInfo(can_disable=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'distance': KeyInfo(default=1),
                'dst-address': KeyInfo(),
                'gateway': KeyInfo(),
                'route-tag': KeyInfo(can_disable=True),
                'routing-table': KeyInfo(default='main'),
                'scope': KeyInfo(default=30),
                'target-scope': KeyInfo(default=10),
                'vrf-interface': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('mpls', ): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allow-fast-path': KeyInfo(default=True),
                'dynamic-label-range': KeyInfo(default='16-1048575'),
                'propagate-ttl': KeyInfo(default=True),
            },
        ),
    ),
    ('mpls', 'interface'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'disabled': KeyInfo(default=False),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'interface': KeyInfo(required=True),
                'mpls-mtu': KeyInfo(),
                'info': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('mpls', 'ldp'): APIData(
        versioned=[
            ('7.1', '>=', VersionedAPIData(
                fully_understood=True,
                primary_keys=('vrf', ),
                fields={
                    'afi': KeyInfo(can_disable=True),
                    'distribute-for-default': KeyInfo(can_disable=True),
                    'path-vector-limit': KeyInfo(can_disable=True),
                    'vrf': KeyInfo(),
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'hop-limit': KeyInfo(can_disable=True),
                    'preferred-afi': KeyInfo(can_disable=True),
                    'loop-detect': KeyInfo(can_disable=True),
                    'transport-addresses': KeyInfo(can_disable=True),
                    'disabled': KeyInfo(default=False),
                    'lsr-id': KeyInfo(can_disable=True),
                    'use-explicit-null': KeyInfo(can_disable=True),
                },
            )),
            ('7.1', '<', VersionedAPIData(
                single_value=True,
                fully_understood=True,
                fields={
                    'distribute-for-default-route': KeyInfo(default=False),
                    'enabled': KeyInfo(default=False),
                    'hop-limit': KeyInfo(default=255),
                    'loop-detect': KeyInfo(default=False),
                    'lsr-id': KeyInfo(default='0.0.0.0'),
                    'path-vector-limit': KeyInfo(default=255),
                    'transport-address': KeyInfo(default='0.0.0.0'),
                    'use-explicit-null': KeyInfo(default=False),
                },
            )),
        ],
    ),
    ('mpls', 'ldp', 'accept-filter'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'accept': KeyInfo(can_disable=True),
                'disabled': KeyInfo(default=False),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'neighbor': KeyInfo(can_disable=True),
                'prefix': KeyInfo(can_disable=True),
                'vrf': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('mpls', 'ldp', 'advertise-filter'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'advertise': KeyInfo(default=''),
                'disabled': KeyInfo(default=False),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'neighbor': KeyInfo(),
                'prefix': KeyInfo(),
                'vrf': KeyInfo(),
            },
        ),
    ),
    ('mpls', 'ldp', 'interface'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'disabled': KeyInfo(default=False),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'accept-dynamic-neighbors': KeyInfo(can_disable=True),
                'afi': KeyInfo(can_disable=True),
                'hello-interval': KeyInfo(can_disable=True),
                'hold-time': KeyInfo(can_disable=True),
                'interface': KeyInfo(required=True),
                'transport-addresses': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('port', 'firmware'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'directory': KeyInfo(default='firmware'),
                'ignore-directip-modem': KeyInfo(default=False),
            },
        ),
    ),
    ('port', 'remote-access'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'allowed-addresses': KeyInfo(default='0.0.0.0/0'),
                'channel': KeyInfo(default=0),
                'disabled': KeyInfo(default=False),
                'log-file': KeyInfo(default=""),
                'port': KeyInfo(required=True),
                'protocol': KeyInfo(default='rfc2217'),
                'tcp-port': KeyInfo(default=0),
            },
        ),
    ),
    ('ppp', 'aaa'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'accounting': KeyInfo(default=True),
                'interim-update': KeyInfo(default='0s'),
                'use-circuit-id-in-nas-port-id': KeyInfo(default=False),
                'use-radius': KeyInfo(default=False),
            },
        ),
    ),
    ('radius', ): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'accounting-backup': KeyInfo(default=False),
                'accounting-port': KeyInfo(default=1813),
                'address': KeyInfo(default='0.0.0.0'),
                'authentication-port': KeyInfo(default=1812),
                'called-id': KeyInfo(),
                'certificate': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'domain': KeyInfo(),
                'protocol': KeyInfo(default='udp'),
                'realm': KeyInfo(),
                'secret': KeyInfo(),
                'service': KeyInfo(),
                'src-address': KeyInfo(default='0.0.0.0'),
                'timeout': KeyInfo(default='300ms'),
            },
            versioned_fields=[
                ([('7.15', '>=')], 'require-message-auth', KeyInfo(default='yes-for-request-resp')),
            ],
        ),
    ),
    ('radius', 'incoming'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'accept': KeyInfo(default=False),
                'port': KeyInfo(default=3799),
            },
        ),
    ),
    ('routing', 'id'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'id': KeyInfo(),
                'name': KeyInfo(),
                'select-dynamic-id': KeyInfo(),
                'select-from-vrf': KeyInfo(),
            },
        ),
    ),
    ('routing', 'igmp-proxy'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'query-interval': KeyInfo(),
                'query-response-interval': KeyInfo(),
                'quick-leave': KeyInfo(default=False),
            },
        ),
    ),
    ('routing', 'igmp-proxy', 'interface'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('interface', ),
            fields={
                'alternative-subnets': KeyInfo(default=''),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'interface': KeyInfo(),
                'threshold': KeyInfo(),
                'upstream': KeyInfo(default=False),
            },
        ),
    ),
    ('routing', 'bfd', 'interface'): APIData(
        unversioned=VersionedAPIData(
            unknown_mechanism=True,
            # primary_keys=('default', ),
            fields={
                'default': KeyInfo(),
                'disabled': KeyInfo(),
                'interface': KeyInfo(),
                'interval': KeyInfo(),
                'min-rx': KeyInfo(),
                'multiplier': KeyInfo(),
            },
        ),
    ),
    ('routing', 'mme'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'bidirectional-timeout': KeyInfo(default=2),
                'gateway-class': KeyInfo(default='none'),
                'gateway-keepalive': KeyInfo(default='1m'),
                'gateway-selection': KeyInfo(default='no-gateway'),
                'origination-interval': KeyInfo(default='5s'),
                'preferred-gateway': KeyInfo(default='0.0.0.0'),
                'timeout': KeyInfo(default='1m'),
                'ttl': KeyInfo(default=50),
            },
        ),
    ),
    ('routing', 'rip'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'distribute-default': KeyInfo(default='never'),
                'garbage-timer': KeyInfo(default='2m'),
                'metric-bgp': KeyInfo(default=1),
                'metric-connected': KeyInfo(default=1),
                'metric-default': KeyInfo(default=1),
                'metric-ospf': KeyInfo(default=1),
                'metric-static': KeyInfo(default=1),
                'redistribute-bgp': KeyInfo(default=False),
                'redistribute-connected': KeyInfo(default=False),
                'redistribute-ospf': KeyInfo(default=False),
                'redistribute-static': KeyInfo(default=False),
                'routing-table': KeyInfo(default='main'),
                'timeout-timer': KeyInfo(default='3m'),
                'update-timer': KeyInfo(default='30s'),
            },
        ),
    ),
    ('routing', 'ripng'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'distribute-default': KeyInfo(default='never'),
                'garbage-timer': KeyInfo(default='2m'),
                'metric-bgp': KeyInfo(default=1),
                'metric-connected': KeyInfo(default=1),
                'metric-default': KeyInfo(default=1),
                'metric-ospf': KeyInfo(default=1),
                'metric-static': KeyInfo(default=1),
                'redistribute-bgp': KeyInfo(default=False),
                'redistribute-connected': KeyInfo(default=False),
                'redistribute-ospf': KeyInfo(default=False),
                'redistribute-static': KeyInfo(default=False),
                'timeout-timer': KeyInfo(default='3m'),
                'update-timer': KeyInfo(default='30s'),
            },
        ),
    ),
    ('snmp', ): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'contact': KeyInfo(default=''),
                'enabled': KeyInfo(default=False),
                'location': KeyInfo(default=''),
                'src-address': KeyInfo(default='::'),
                'trap-community': KeyInfo(default='public'),
                'trap-generators': KeyInfo(default='temp-exception'),
                'trap-target': KeyInfo(default=''),
                'trap-version': KeyInfo(default=1),
                'trap-interfaces': KeyInfo(default=''),
            },
            versioned_fields=[
                ([('7.10', '<')], 'engine-id', KeyInfo(default='')),
                ([('7.10', '>=')], 'engine-id', KeyInfo(read_only=True)),
                ([('7.10', '>=')], 'engine-id-suffix', KeyInfo(default='')),
            ],
        ),
    ),
    ('system', 'clock'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'date': KeyInfo(),
                'gmt-offset': KeyInfo(),
                'time': KeyInfo(),
                'time-zone-autodetect': KeyInfo(default=True),
                'time-zone-name': KeyInfo(default='manual'),
            },
        ),
    ),
    ('system', 'clock', 'manual'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'dst-delta': KeyInfo(default='00:00'),
                'dst-end': KeyInfo(default='jan/01/1970 00:00:00'),
                'dst-start': KeyInfo(default='jan/01/1970 00:00:00'),
                'time-zone': KeyInfo(default='+00:00'),
            },
        ),
    ),
    ('system', 'health', 'settings'): APIData(
        versioned=[
            ('7.14', '<', VersionedAPIData(
                single_value=True,
                fully_understood=True,
                fields={
                    'cpu-overtemp-check': KeyInfo(),
                    'cpu-overtemp-startup-delay': KeyInfo(),
                    'cpu-overtemp-threshold': KeyInfo(),
                    'fan-control-interval': KeyInfo(can_disable=True, default='30s'),
                    'fan-full-speed-temp': KeyInfo(default=65),
                    'fan-min-speed-percent': KeyInfo(default=0),
                    'fan-mode': KeyInfo(),
                    'fan-on-threshold': KeyInfo(),
                    'fan-switch': KeyInfo(),
                    'fan-target-temp': KeyInfo(default=58),
                    'use-fan': KeyInfo(),
                },
            )),
            ('7.14', '>=', VersionedAPIData(
                single_value=True,
                fully_understood=True,
                fields={
                    'cpu-overtemp-check': KeyInfo(),
                    'cpu-overtemp-startup-delay': KeyInfo(),
                    'cpu-overtemp-threshold': KeyInfo(),
                    'fan-control-interval': KeyInfo(default=30),
                    'fan-full-speed-temp': KeyInfo(default=65),
                    'fan-min-speed-percent': KeyInfo(default=12),
                    'fan-mode': KeyInfo(),
                    'fan-on-threshold': KeyInfo(),
                    'fan-switch': KeyInfo(),
                    'fan-target-temp': KeyInfo(default=58),
                    'use-fan': KeyInfo(),
                },
            )),
        ],
    ),
    ('system', 'identity'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'name': KeyInfo(default='Mikrotik'),
            },
        ),
    ),
    ('system', 'leds', 'settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'all-leds-off': KeyInfo(default='never'),
            },
        ),
    ),
    ('system', 'note'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'note': KeyInfo(default=''),
                'show-at-login': KeyInfo(default=True),
            },
        ),
    ),
    ('system', 'ntp', 'client'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'enabled': KeyInfo(default=False),
                'primary-ntp': KeyInfo(default='0.0.0.0'),
                'secondary-ntp': KeyInfo(default='0.0.0.0'),
                'server-dns-names': KeyInfo(default=''),
                'servers': KeyInfo(default=''),
                'mode': KeyInfo(default='unicast'),
                'vrf': KeyInfo(default='main'),
            },
        ),
    ),
    ('system', 'ntp', 'client', 'servers'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('address', ),
            fully_understood=True,
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'address': KeyInfo(),
                'auth-key': KeyInfo(default='none'),
                'iburst': KeyInfo(default=True),
                'max-poll': KeyInfo(default=10),
                'min-poll': KeyInfo(default=6),
            },
        ),
    ),
    ('system', 'ntp', 'server'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'auth-key': KeyInfo(default='none'),
                'broadcast': KeyInfo(default=False),
                'broadcast-addresses': KeyInfo(default=''),
                'enabled': KeyInfo(default=False),
                'local-clock-stratum': KeyInfo(default=5),
                'manycast': KeyInfo(default=False),
                'multicast': KeyInfo(default=False),
                'use-local-clock': KeyInfo(default=False),
                'vrf': KeyInfo(default='main'),
            },
        ),
    ),
    ('system', 'package', 'update'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'channel': KeyInfo(default='stable'),
                'installed-version': KeyInfo(read_only=True),
                'latest-version': KeyInfo(read_only=True),
                'status': KeyInfo(read_only=True),
            },
        ),
    ),
    ('system', 'routerboard', 'settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'auto-upgrade': KeyInfo(default=False),
                'baud-rate': KeyInfo(default=115200),
                'boot-delay': KeyInfo(default='2s'),
                'boot-device': KeyInfo(default='nand-if-fail-then-ethernet'),
                'boot-protocol': KeyInfo(default='bootp'),
                'cpu-frequency': KeyInfo(),
                'enable-jumper-reset': KeyInfo(default=True),
                'enter-setup-on': KeyInfo(default='any-key'),
                'force-backup-booter': KeyInfo(default=False),
                'memory-frequency': KeyInfo(),
                'preboot-etherboot': KeyInfo(),
                'preboot-etherboot-server': KeyInfo(),
                'protected-routerboot': KeyInfo(default='disabled'),
                'reformat-hold-button': KeyInfo(default='20s'),
                'reformat-hold-button-max': KeyInfo(default='10m'),
                'silent-boot': KeyInfo(default=False),
            },
        ),
    ),
    ('system', 'upgrade', 'mirror'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'check-interval': KeyInfo(default='1d'),
                'enabled': KeyInfo(default=False),
                'primary-server': KeyInfo(default='0.0.0.0'),
                'secondary-server': KeyInfo(default='0.0.0.0'),
                'user': KeyInfo(default=''),
            },
        ),
    ),
    ('system', 'ups'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'alarm-setting': KeyInfo(default='immediate'),
                'check-capabilities': KeyInfo(can_disable=True, remove_value=True),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=True),
                'min-runtime': KeyInfo(default='never'),
                'name': KeyInfo(),
                'offline-time': KeyInfo(default='0s'),
                'port': KeyInfo(required=True),
            },
        ),
    ),
    ('system', 'watchdog'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'auto-send-supout': KeyInfo(default=False),
                'automatic-supout': KeyInfo(default=True),
                'ping-start-after-boot': KeyInfo(default='5m'),
                'ping-timeout': KeyInfo(default='1m'),
                'watch-address': KeyInfo(default='none'),
                'watchdog-timer': KeyInfo(default=True),
            },
        ),
    ),
    ('tool', 'bandwidth-server'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allocate-udp-ports-from': KeyInfo(default=2000),
                'authenticate': KeyInfo(default=True),
                'enabled': KeyInfo(default=True),
                'max-sessions': KeyInfo(default=100),
            },
        ),
    ),
    ('tool', 'e-mail'): APIData(
        versioned=[
            ('7.12', '>=', VersionedAPIData(
                single_value=True,
                fully_understood=True,
                fields={
                    'from': KeyInfo(default='<>'),
                    'password': KeyInfo(default=''),
                    'port': KeyInfo(default=25),
                    'server': KeyInfo(default='0.0.0.0'),
                    'start-tls': KeyInfo(default=False),
                    'tls': KeyInfo(default=False),
                    'user': KeyInfo(default=''),
                    'vfr': KeyInfo(default=''),
                },
            )),
            ('7.12', '<', VersionedAPIData(
                single_value=True,
                fully_understood=True,
                fields={
                    'address': KeyInfo(default='0.0.0.0'),
                    'from': KeyInfo(default='<>'),
                    'password': KeyInfo(default=''),
                    'port': KeyInfo(default=25),
                    'start-tls': KeyInfo(default=False),
                    'tls': KeyInfo(default=False),
                    'user': KeyInfo(default=''),
                    'vfr': KeyInfo(default=''),
                },
            )),
        ],
    ),
    ('tool', 'graphing'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'page-refresh': KeyInfo(default=300),
                'store-every': KeyInfo(default='5min'),
            },
        ),
    ),
    ('tool', 'graphing', 'interface'): APIData(
        versioned=[
            ('7', '>=', VersionedAPIData(
                fully_understood=True,
                fields={
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'allow-address': KeyInfo(default='0.0.0.0/0'),
                    'interface': KeyInfo(default='all'),
                    'store-on-disk': KeyInfo(default=True),
                },
            )),
        ],
    ),
    ('tool', 'graphing', 'resource'): APIData(
        versioned=[
            ('7', '>=', VersionedAPIData(
                fully_understood=True,
                fields={
                    'comment': KeyInfo(can_disable=True, remove_value=''),
                    'disabled': KeyInfo(default=False),
                    'allow-address': KeyInfo(default='0.0.0.0/0'),
                    'store-on-disk': KeyInfo(default=True),
                },
            )),
        ],
    ),
    ('tool', 'mac-server'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allowed-interface-list': KeyInfo(),
            },
        ),
    ),
    ('tool', 'mac-server', 'mac-winbox'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allowed-interface-list': KeyInfo(),
            },
        ),
    ),
    ('tool', 'mac-server', 'ping'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'enabled': KeyInfo(default=True),
            },
        ),
    ),
    ('tool', 'netwatch'): APIData(
        versioned=[
            ('7', '>=', VersionedAPIData(
                fully_understood=True,
                versioned_fields=[
                    ([('7.16', '>=')], 'accept-icmp-time-exceeded', KeyInfo(default=False)),
                    ([('7.16', '>=')], 'ttl', KeyInfo(default=255)),
                ],
                fields={
                    'certificate': KeyInfo(),
                    'check-certificate': KeyInfo(),
                    'comment': KeyInfo(),
                    'disabled': KeyInfo(default=False),
                    'down-script': KeyInfo(),
                    'host': KeyInfo(required=True),
                    'http-codes': KeyInfo(),
                    'interval': KeyInfo(),
                    'name': KeyInfo(),
                    'packet-count': KeyInfo(),
                    'packet-interval': KeyInfo(),
                    'packet-size': KeyInfo(),
                    'port': KeyInfo(),
                    'src-address': KeyInfo(),
                    'start-delay': KeyInfo(),
                    'startup-delay': KeyInfo(),
                    'test-script': KeyInfo(),
                    'thr-avg': KeyInfo(),
                    'thr-http-time': KeyInfo(),
                    'thr-jitter': KeyInfo(),
                    'thr-loss-count': KeyInfo(),
                    'thr-loss-percent': KeyInfo(),
                    'thr-max': KeyInfo(),
                    'thr-stdev': KeyInfo(),
                    'thr-tcp-conn-time': KeyInfo(),
                    'timeout': KeyInfo(),
                    'type': KeyInfo(default='simple'),
                    'up-script': KeyInfo(),
                },
            )),
        ],
    ),
    ('tool', 'romon'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'enabled': KeyInfo(default=False),
                'id': KeyInfo(default='00:00:00:00:00:00'),
                'secrets': KeyInfo(default=''),
            },
        ),
    ),
    ('tool', 'romon', 'port'): APIData(
        unversioned=VersionedAPIData(
            fields={
                'cost': KeyInfo(),
                'disabled': KeyInfo(),
                'forbid': KeyInfo(),
                'interface': KeyInfo(),
                'secrets': KeyInfo(),
            },
        ),
    ),
    ('tool', 'sms'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'allowed-number': KeyInfo(default=''),
                'auto-erase': KeyInfo(default=False),
                'channel': KeyInfo(default=0),
                'port': KeyInfo(default='none'),
                'receive-enabled': KeyInfo(default=False),
                'secret': KeyInfo(default=''),
                'sim-pin': KeyInfo(default=''),
            },
        ),
    ),
    ('tool', 'sniffer'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'file-limit': KeyInfo(default='1000KiB'),
                'file-name': KeyInfo(default=''),
                'filter-cpu': KeyInfo(default=''),
                'filter-direction': KeyInfo(default='any'),
                'filter-interface': KeyInfo(default=''),
                'filter-ip-address': KeyInfo(default=''),
                'filter-ip-protocol': KeyInfo(default=''),
                'filter-ipv6-address': KeyInfo(default=''),
                'filter-mac-address': KeyInfo(default=''),
                'filter-mac-protocol': KeyInfo(default=''),
                'filter-operator-between-entries': KeyInfo(default='or'),
                'filter-port': KeyInfo(default=''),
                'filter-size': KeyInfo(default=''),
                'filter-stream': KeyInfo(default=False),
                'memory-limit': KeyInfo(default='100KiB'),
                'memory-scroll': KeyInfo(default=True),
                'only-headers': KeyInfo(default=False),
                'streaming-enabled': KeyInfo(default=False),
                'streaming-server': KeyInfo(default='0.0.0.0:37008'),
            },
        ),
    ),
    ('tool', 'traffic-generator'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'latency-distribution-max': KeyInfo(default='100us'),
                'measure-out-of-order': KeyInfo(default=True),
                'stats-samples-to-keep': KeyInfo(default=100),
                'test-id': KeyInfo(default=0),
            },
        ),
    ),
    ('user',): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'address': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'expired': KeyInfo(read_only=True),
                'group': KeyInfo(),
                'last-logged-in': KeyInfo(read_only=True),
                'name': KeyInfo(),
                'password': KeyInfo(write_only=True),
            },
        ),
    ),
    ('user', 'aaa'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'accounting': KeyInfo(default=True),
                'default-group': KeyInfo(default='read'),
                'exclude-groups': KeyInfo(default=''),
                'interim-update': KeyInfo(default='0s'),
                'use-radius': KeyInfo(default=False),
            },
        ),
    ),
    ('user', 'settings'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'minimum-categories': KeyInfo(),
                'minimum-password-length': KeyInfo(),
            },
        ),
    ),
    ('queue', 'interface'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('interface', ),
            fully_understood=True,
            fixed_entries=True,
            fields={
                'interface': KeyInfo(required=True),
                'queue': KeyInfo(required=True),
            },
        ),
    ),
    ('queue', 'simple'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'dst': KeyInfo(can_disable=True, remove_value=''),
                'time': KeyInfo(can_disable=True, remove_value=''),
                'bucket-size': KeyInfo(default='0.1/0.1'),
                'burst-limit': KeyInfo(default='0/0'),
                'burst-threshold': KeyInfo(default='0/0'),
                'burst-time': KeyInfo(default='0s/0s'),
                'disabled': KeyInfo(default=False),
                'limit-at': KeyInfo(default='0/0'),
                'max-limit': KeyInfo(default='0/0'),
                'name': KeyInfo(),
                'packet-marks': KeyInfo(default=''),
                'parent': KeyInfo(default='none'),
                'priority': KeyInfo(default='8/8'),
                'queue': KeyInfo(default='default-small/default-small'),
                'target': KeyInfo(required=True),
            },
        ),
    ),
    ('queue', 'tree'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'bucket-size': KeyInfo(default='0.1'),
                'burst-limit': KeyInfo(default=0),
                'burst-threshold': KeyInfo(default=0),
                'burst-time': KeyInfo(default='0s'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'limit-at': KeyInfo(default=0),
                'max-limit': KeyInfo(default=0),
                'name': KeyInfo(),
                'packet-mark': KeyInfo(default=''),
                'parent': KeyInfo(required=True),
                'priority': KeyInfo(default=8),
                'queue': KeyInfo(default='default-small'),
            },
        ),
    ),
    ('queue', 'type'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'name': KeyInfo(),
                'kind': KeyInfo(required=True),
                'bfifo-limit': KeyInfo(default=15000),
                'cake-ack-filter': KeyInfo(default='none'),
                'cake-atm': KeyInfo(default='none'),
                'cake-autorate-ingress': KeyInfo(can_disable=True),
                'cake-bandwidth': KeyInfo(can_disable=True, remove_value=0),
                'cake-diffserv': KeyInfo(default='diffserv3'),
                'cake-flowmode': KeyInfo(default='triple-isolate'),
                'cake-memlimit': KeyInfo(default=0),
                'cake-mpu': KeyInfo(can_disable=True, remove_value=''),
                'cake-nat': KeyInfo(can_disable=True, remove_value=False),
                'cake-overhead': KeyInfo(default=0),
                'cake-overhead-scheme': KeyInfo(can_disable=True, remove_value=''),
                'cake-rtt': KeyInfo(default='100ms'),
                'cake-rtt-scheme': KeyInfo(default='none'),
                'cake-wash': KeyInfo(can_disable=True, remove_value=False),
                'codel-ce-threshold': KeyInfo(can_disable=True, remove_value=''),
                'codel-ecn': KeyInfo(can_disable=True, remove_value=False),
                'codel-interval': KeyInfo(default='100ms'),
                'codel-limit': KeyInfo(default=1000),
                'codel-target': KeyInfo(default='5ms'),
                'fq-codel-ce-threshold': KeyInfo(can_disable=True, remove_value=''),
                'fq-codel-ecn': KeyInfo(default=True),
                'fq-codel-flows': KeyInfo(default=1024),
                'fq-codel-interval': KeyInfo(default='100ms'),
                'fq-codel-limit': KeyInfo(default=10240),
                'fq-codel-memlimit': KeyInfo(default=33554432),
                'fq-codel-quantum': KeyInfo(default=1514),
                'fq-codel-target': KeyInfo(default='5ms'),
                'mq-pfifo-limit': KeyInfo(default=50),
                'pcq-burst-rate': KeyInfo(default=0),
                'pcq-burst-threshold': KeyInfo(default=0),
                'pcq-burst-time': KeyInfo(default='10s'),
                'pcq-classifier': KeyInfo(can_disable=True, remove_value=''),
                'pcq-dst-address-mask': KeyInfo(default=32),
                'pcq-dst-address6-mask': KeyInfo(default=128),
                'pcq-limit': KeyInfo(default=50),
                'pcq-rate': KeyInfo(default=0),
                'pcq-src-address-mask': KeyInfo(default=32),
                'pcq-src-address6-mask': KeyInfo(default=128),
                'pcq-total-limit': KeyInfo(default=2000),
                'pfifo-limit': KeyInfo(default=50),
                'red-avg-packet': KeyInfo(default=1000),
                'red-burst': KeyInfo(default=20),
                'red-limit': KeyInfo(default=60),
                'red-max-threshold': KeyInfo(default=50),
                'red-min-threshold': KeyInfo(default=10),
                'sfq-allot': KeyInfo(default=1514),
                'sfq-perturb': KeyInfo(default=5),
            },
        ),
    ),
    ('interface', 'ethernet', 'switch'): APIData(
        unversioned=VersionedAPIData(
            fixed_entries=True,
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'cpu-flow-control': KeyInfo(default=True),
                'mirror-source': KeyInfo(default='none'),
                'mirror-target': KeyInfo(default='none'),
                'name': KeyInfo(),
            },
        ),
    ),
    ('interface', 'ethernet', 'switch', 'port'): APIData(
        unversioned=VersionedAPIData(
            fixed_entries=True,
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'default-vlan-id': KeyInfo(),
                'name': KeyInfo(),
                'vlan-header': KeyInfo(default='leave-as-is'),
                'vlan-mode': KeyInfo(default='disabled'),
            },
        ),
    ),
    ('ip', 'dhcp-client', 'option'): APIData(
        unversioned=VersionedAPIData(
            fixed_entries=True,
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'code': KeyInfo(),
                'name': KeyInfo(),
                'value': KeyInfo(),
            },
        ),
    ),
    ('ppp', 'profile'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'address-list': KeyInfo(default=''),
                'bridge': KeyInfo(can_disable=True),
                'bridge-horizon': KeyInfo(can_disable=True),
                'bridge-learning': KeyInfo(default='default'),
                'bridge-path-cost': KeyInfo(can_disable=True),
                'bridge-port-priority': KeyInfo(can_disable=True),
                'change-tcp-mss': KeyInfo(default=True),
                'dns-server': KeyInfo(can_disable=True),
                'idle-timeout': KeyInfo(can_disable=True),
                'incoming-filter': KeyInfo(can_disable=True),
                'insert-queue-before': KeyInfo(can_disable=True),
                'interface-list': KeyInfo(can_disable=True),
                'local-address': KeyInfo(can_disable=True),
                'name': KeyInfo(required=True),
                'on-down': KeyInfo(default=''),
                'on-up': KeyInfo(default=''),
                'only-one': KeyInfo(default='default'),
                'outgoing-filter': KeyInfo(can_disable=True),
                'parent-queue': KeyInfo(can_disable=True),
                'queue-type': KeyInfo(can_disable=True),
                'rate-limit': KeyInfo(can_disable=True),
                'remote-address': KeyInfo(can_disable=True),
                'session-timeout': KeyInfo(can_disable=True),
                'use-compression': KeyInfo(default='default'),
                'use-encryption': KeyInfo(default='default'),
                'use-ipv6': KeyInfo(default=True),
                'use-mpls': KeyInfo(default='default'),
                'use-upnp': KeyInfo(default='default'),
                'wins-server': KeyInfo(can_disable=True),
            },
        ),
    ),
    ('ppp', 'secret'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'caller-id': KeyInfo(default=''),
                'disabled': KeyInfo(default=False),
                'ipv6-routes': KeyInfo(default=''),
                'limit-bytes-in': KeyInfo(default=0),
                'limit-bytes-out': KeyInfo(default=0),
                'local-address': KeyInfo(can_disable=True),
                'name': KeyInfo(required=True),
                'password': KeyInfo(),
                'profile': KeyInfo(default='default'),
                'remote-address': KeyInfo(can_disable=True),
                'remote-ipv6-prefix': KeyInfo(can_disable=True),
                'routes': KeyInfo(can_disable=True),
                'service': KeyInfo(default='any'),
            },
        ),
    ),
    ('routing', 'bgp', 'aggregate'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('prefix',),
            fully_understood=True,
            fields={
                'advertise-filter': KeyInfo(),
                'attribute-filter': KeyInfo(),
                'disabled': KeyInfo(default=False),
                'include-igp': KeyInfo(default=False),
                'inherit-attributes': KeyInfo(default=True),
                'instance': KeyInfo(required=True),
                'prefix': KeyInfo(required=True),
                'summary-only': KeyInfo(default=True),
                'suppress-filter': KeyInfo(),
            },
        ),
    ),
    ('routing', 'bgp', 'connection'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            versioned_fields=[
                ([('7.19', '<')], 'address-families', KeyInfo()),
                ([('7.19', '>=')], 'afi', KeyInfo()),
            ],
            fields={
                'as': KeyInfo(),
                'add-path-out': KeyInfo(),
                'cisco-vpls-nlri-len-fmt': KeyInfo(),
                'cluster-id': KeyInfo(),
                'comment': KeyInfo(),
                'connect': KeyInfo(default=True),
                'disabled': KeyInfo(default=False),
                'hold-time': KeyInfo(),
                'input.accept-communities': KeyInfo(),
                'input.accept-ext-communities': KeyInfo(),
                'input.accept-large-communities': KeyInfo(),
                'input.accpet-nlri': KeyInfo(),
                'input.accept-unknown': KeyInfo(),
                'input.affinity': KeyInfo(),
                'input.allow-as': KeyInfo(),
                'input.filter': KeyInfo(),
                'input.ignore-as-path-len': KeyInfo(),
                'input.limit-process-routes-ipv4': KeyInfo(),
                'input.limit-process-routes-ipv6': KeyInfo(),
                'keepalive-time': KeyInfo(),
                'listen': KeyInfo(default=True),
                'local.address': KeyInfo(),
                'local.port': KeyInfo(),
                'local.role': KeyInfo(required=True),
                'local.ttl': KeyInfo(),
                'multihop': KeyInfo(),
                'name': KeyInfo(required=True),
                'nexthop-choice': KeyInfo(),
                'output.affinity': KeyInfo(),
                'output.as-override': KeyInfo(),
                'output.default-originate': KeyInfo(),
                'output.default-prepend': KeyInfo(),
                'output.filter-chain': KeyInfo(),
                'output.filter-select': KeyInfo(),
                'output.keep-sent-attributes': KeyInfo(),
                'output.network': KeyInfo(),
                'output.no-client-to-client-reflection': KeyInfo(),
                'output.no-early-cut': KeyInfo(),
                'output.redistribute': KeyInfo(),
                'output.remote-private-as': KeyInfo(),
                'remote.address': KeyInfo(required=True),
                'remote.port': KeyInfo(),
                'remote.as': KeyInfo(),
                'remote.allowed-as': KeyInfo(),
                'remote.ttl': KeyInfo(),
                'router-id': KeyInfo(),
                'routing-table': KeyInfo(),
                'save-to': KeyInfo(),
                'tcp-md5-key': KeyInfo(),
                'templates': KeyInfo(),
                'use-bfd': KeyInfo(),
                'vrf': KeyInfo(),
            },
        ),
    ),
    ('routing', 'bgp', 'instance'): APIData(
        unversioned=VersionedAPIData(
            fixed_entries=True,
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'as': KeyInfo(),
                'client-to-client-reflection': KeyInfo(),
                'cluster-id': KeyInfo(can_disable=True),
                'confederation': KeyInfo(can_disable=True),
                'disabled': KeyInfo(default=False),
                'ignore-as-path-len': KeyInfo(),
                'name': KeyInfo(),
                'out-filter': KeyInfo(),
                'redistribute-connected': KeyInfo(),
                'redistribute-ospf': KeyInfo(),
                'redistribute-other-bgp': KeyInfo(),
                'redistribute-rip': KeyInfo(),
                'redistribute-static': KeyInfo(),
                'router-id': KeyInfo(),
                'routing-table': KeyInfo(),
            },
        ),
    ),
    ('routing', 'bgp', 'network'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('network',),
            fully_understood=True,
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'network': KeyInfo(required=True),
                'synchronize': KeyInfo(default=True),
            },
        ),
    ),
    ('routing', 'bgp', 'peer'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'address-families': KeyInfo(default='ip'),
                'allow-as-in': KeyInfo(can_disable=True, remove_value=''),
                'as-override': KeyInfo(default=False),
                'cisco-vpls-nlri-len-fmt': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'default-originate': KeyInfo(default='never'),
                'disabled': KeyInfo(default=False),
                'hold-time': KeyInfo(default='3m'),
                'in-filter': KeyInfo(),
                'instance': KeyInfo(),
                'keepalive-time': KeyInfo(can_disable=True, remove_value=''),
                'max-prefix-limit': KeyInfo(can_disable=True, remove_value=''),
                'max-prefix-restart-time': KeyInfo(can_disable=True, remove_value=''),
                'multihop': KeyInfo(default=False),
                'name': KeyInfo(),
                'nexthop-choice': KeyInfo(default='default'),
                'passive': KeyInfo(default=False),
                'out-filter': KeyInfo(),
                'remote-address': KeyInfo(required=True),
                'remote-as': KeyInfo(required=True),
                'remote-port': KeyInfo(can_disable=True, remove_value=''),
                'remove-private-as': KeyInfo(default=False),
                'route-reflect': KeyInfo(default=False),
                'tcp-md5-key': KeyInfo(),
                'ttl': KeyInfo(default='default'),
                'update-source': KeyInfo(can_disable=True, remove_value='none'),
                'use-bfd': KeyInfo(default=False),
            },
        ),
    ),
    ('routing', 'bgp', 'template'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'add-path-out': KeyInfo(),
                'address-families': KeyInfo(default='ip'),
                'as': KeyInfo(),
                'as-override': KeyInfo(default=False),
                'cisco-vpls-nlri-len-fmt': KeyInfo(),
                'cluster-id': KeyInfo(),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'hold-time': KeyInfo(default='3m'),
                'input.accept-communities': KeyInfo(),
                'input.accept-ext-communities': KeyInfo(),
                'input.accept-large-communities': KeyInfo(),
                'input.accept-unknown': KeyInfo(),
                'input.accept-nlri': KeyInfo(),
                'input.affinity': KeyInfo(),
                'input.allow-as': KeyInfo(),
                'input.filter': KeyInfo(),
                'input.ignore-as-path-len': KeyInfo(default=False),
                'input.limit-nlri-diversity': KeyInfo(),
                'input.limit-process-routes-ipv4': KeyInfo(),
                'input.limit-process-routes-ipv6': KeyInfo(),
                'keepalive-time': KeyInfo(default='3m'),
                'multihop': KeyInfo(default=False),
                'name': KeyInfo(),
                'nexthop-choice': KeyInfo(default='default'),
                'output.affinity': KeyInfo(),
                'output.default-originate': KeyInfo(default='never'),
                'output.default-prepent': KeyInfo(),
                'output.filter-chain': KeyInfo(),
                'output.filter-select': KeyInfo(),
                'output.keep-sent-attributes': KeyInfo(default=False),
                'output.network': KeyInfo(),
                'output.no-client-to-client-reflection': KeyInfo(),
                'output.no-early-cut': KeyInfo(),
                'output.redistribute': KeyInfo(),
                'remove-private-as': KeyInfo(default=False),
                'router-id': KeyInfo(default='main'),
                'routing-table': KeyInfo(default='main'),
                'save-to': KeyInfo(),
                'templates': KeyInfo(),
                'use-bfd': KeyInfo(default=False),
                'vrf': KeyInfo(default='main'),
            },
        ),
    ),
    ('system', 'logging', 'action'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name',),
            fields={
                'bsd-syslog': KeyInfo(default=False),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disk-file-count': KeyInfo(default=2),
                'disk-file-name': KeyInfo(default='log'),
                'disk-lines-per-file': KeyInfo(default=1000),
                'disk-stop-on-full': KeyInfo(default=False),
                'email-start-tls': KeyInfo(default=False),
                'email-to': KeyInfo(default=''),
                'memory-lines': KeyInfo(default=1000),
                'memory-stop-on-full': KeyInfo(default=False),
                'name': KeyInfo(),
                'remember': KeyInfo(default=True),
                'remote': KeyInfo(default='0.0.0.0'),
                'remote-port': KeyInfo(default=514),
                'src-address': KeyInfo(default='0.0.0.0'),
                'syslog-facility': KeyInfo(default='daemon'),
                'syslog-severity': KeyInfo(default='auto'),
                'syslog-time-format': KeyInfo(default='bsd-syslog'),
                'target': KeyInfo(required=True),
            },
        ),
    ),
    ('user', 'group'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'name': KeyInfo(),
                'policy': KeyInfo(),
                'skin': KeyInfo(default='default'),
            },
        ),
    ),
    ('caps-man', 'manager'): APIData(
        unversioned=VersionedAPIData(
            single_value=True,
            fully_understood=True,
            fields={
                'ca-certificate': KeyInfo(default='none'),
                'certificate': KeyInfo(default='none'),
                'enabled': KeyInfo(default=False),
                'package-path': KeyInfo(default=''),
                'require-peer-certificate': KeyInfo(default=False),
                'upgrade-policy': KeyInfo(default='none'),
            },
        ),
    ),
    ('ip', 'firewall', 'service-port'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'disabled': KeyInfo(default=False),
                'name': KeyInfo(),
                'ports': KeyInfo(),
                'sip-direct-media': KeyInfo(),
                'sip-timeout': KeyInfo(),
            },
        ),
    ),
    ('ip', 'firewall', 'layer7-protocol'): APIData(
        unversioned=VersionedAPIData(
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'name': KeyInfo(),
                'regexp': KeyInfo(),
            },
        ),
    ),
    ('ip', 'hotspot', 'service-port'): APIData(
        unversioned=VersionedAPIData(
            fixed_entries=True,
            primary_keys=('name', ),
            fully_understood=True,
            fields={
                'disabled': KeyInfo(default=False),
                'name': KeyInfo(),
                'ports': KeyInfo(),
            },
        ),
    ),
    ('ip', 'ipsec', 'policy'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'action': KeyInfo(default='encrypt'),
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'dst-address': KeyInfo(),
                'dst-port': KeyInfo(default='any'),
                'group': KeyInfo(can_disable=True, remove_value='default'),
                'ipsec-protocols': KeyInfo(default='esp'),
                'level': KeyInfo(default='require'),
                'peer': KeyInfo(),
                'proposal': KeyInfo(default='default'),
                'protocol': KeyInfo(default='all'),
                'src-address': KeyInfo(),
                'src-port': KeyInfo(default='any'),
                # The template field ca not really be changed once the item is
                # created. This config captures the behavior best as it can
                # i.e. template=yes is shown, template=no is hidden.
                'template': KeyInfo(can_disable=True, remove_value=False),
                'tunnel': KeyInfo(default=False),
            },
        ),
    ),
    ('ip', 'service'): APIData(
        unversioned=VersionedAPIData(
            fixed_entries=True,
            primary_keys=('name', ),
            fully_understood=True,
            versioned_fields=[
                ([('7.16', '>=')], 'max-sessions', KeyInfo(default=20)),
            ],
            fields={
                'address': KeyInfo(),
                'certificate': KeyInfo(),
                'disabled': KeyInfo(default=False),
                'name': KeyInfo(),
                'port': KeyInfo(),
                'tls-version': KeyInfo(),
            },
        ),
    ),
    ('system', 'logging'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            fields={
                'action': KeyInfo(default='memory'),
                'disabled': KeyInfo(default=False),
                'prefix': KeyInfo(default=''),
                'topics': KeyInfo(default=''),
            },
        ),
    ),
    ('system', 'resource', 'irq'): APIData(
        unversioned=VersionedAPIData(
            has_identifier=True,
            fields={
                'cpu': KeyInfo(),
            },
        ),
    ),
    ('system', 'resource', 'irq', 'rps'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name',),
            fields={
                'disabled': KeyInfo(default=False),
                'name': KeyInfo(),
            },
        ),
    ),
    ('system', 'scheduler'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name', ),
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'disabled': KeyInfo(default=False),
                'interval': KeyInfo(default='0s'),
                'name': KeyInfo(),
                'on-event': KeyInfo(default=''),
                'policy': KeyInfo(default='ftp,reboot,read,write,policy,test,password,sniff,sensitive,romon'),
                'start-date': KeyInfo(),
                'start-time': KeyInfo(),
            },
        ),
    ),
    ('system', 'script'): APIData(
        unversioned=VersionedAPIData(
            fully_understood=True,
            primary_keys=('name',),
            fields={
                'comment': KeyInfo(can_disable=True, remove_value=''),
                'dont-require-permissions': KeyInfo(default=False),
                'name': KeyInfo(),
                'owner': KeyInfo(),
                'policy': KeyInfo(default='ftp,reboot,read,write,policy,test,password,sniff,sensitive,romon'),
                'source': KeyInfo(default=''),
            },
        ),
    ),
}
