// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "base/allocator/allocator_extension.h"

#include "base/logging.h"

#if defined(OXIDE_BUILD)
#include "base/process/process_handle.h"
#include "base/process/process_metrics.h"
#endif

#if defined(USE_TCMALLOC)
#include "third_party/tcmalloc/chromium/src/gperftools/heap-profiler.h"
#include "third_party/tcmalloc/chromium/src/gperftools/malloc_extension.h"
#include "third_party/tcmalloc/chromium/src/gperftools/malloc_hook.h"
#endif

namespace base {
namespace allocator {

#if defined(OXIDE_BUILD)
namespace {

oxide::ReleaseFreeMemoryFunc g_release_free_memory_func;
oxide::UncheckedAllocFunc g_unchecked_alloc_func;

bool g_enable_termination_on_oom_locked = false;
oxide::EnableTerminationOnOOMFunc g_enable_termination_on_oom_func;

}
#endif

void ReleaseFreeMemory() {
#if defined(USE_TCMALLOC)
  ::MallocExtension::instance()->ReleaseFreeMemory();
#elif defined(OXIDE_BUILD)
  if (g_release_free_memory_func) {
    g_release_free_memory_func();
  }
#endif
}

bool GetNumericProperty(const char* name, size_t* value) {
#if defined(USE_TCMALLOC)
  return ::MallocExtension::instance()->GetNumericProperty(name, value);
#endif
  return false;
}

bool IsHeapProfilerRunning() {
#if defined(USE_TCMALLOC)
  return ::IsHeapProfilerRunning();
#endif
  return false;
}

void SetHooks(AllocHookFunc alloc_hook, FreeHookFunc free_hook) {
// TODO(sque): Use allocator shim layer instead.
#if defined(USE_TCMALLOC)
  // Make sure no hooks get overwritten.
  auto prev_alloc_hook = MallocHook::SetNewHook(alloc_hook);
  if (alloc_hook)
    DCHECK(!prev_alloc_hook);

  auto prev_free_hook = MallocHook::SetDeleteHook(free_hook);
  if (free_hook)
    DCHECK(!prev_free_hook);
#endif
}

int GetCallStack(void** stack, int max_stack_size) {
#if defined(USE_TCMALLOC)
  return MallocHook::GetCallerStackTrace(stack, max_stack_size, 0);
#endif
  return 0;
}

#if defined(OXIDE_BUILD)
namespace oxide {

void SetReleaseFreeMemoryFunc(ReleaseFreeMemoryFunc release_free_memory_func) {
  DCHECK(!g_release_free_memory_func);
  DCHECK_EQ(1, base::GetNumberOfThreads(base::GetCurrentProcessHandle()));
  g_release_free_memory_func = release_free_memory_func;
}

void SetUncheckedAllocFunc(UncheckedAllocFunc unchecked_alloc_func) {
  DCHECK(!g_unchecked_alloc_func);
  DCHECK_EQ(1, base::GetNumberOfThreads(base::GetCurrentProcessHandle()));
  g_unchecked_alloc_func = unchecked_alloc_func;
}

void SetEnableTerminationOnOOMFunc(
    EnableTerminationOnOOMFunc enable_termination_on_oom_func) {
  DCHECK(!g_enable_termination_on_oom_locked);
  DCHECK_EQ(1, base::GetNumberOfThreads(base::GetCurrentProcessHandle()));
  g_enable_termination_on_oom_func = enable_termination_on_oom_func;
}

void* UncheckedAlloc(size_t size) {
  if (g_unchecked_alloc_func) {
    return g_unchecked_alloc_func(size);
  }

  return malloc(size);
}

void EnableTerminationOnOutOfMemory() {
  g_enable_termination_on_oom_locked = true;
  if (g_enable_termination_on_oom_func) {
    g_enable_termination_on_oom_func();
  }
}

} // namespace oxide
#endif

}  // namespace allocator
}  // namespace base
