/////////////////////////////////////////////////////////////////////////
// $Id$
/////////////////////////////////////////////////////////////////////////
//
//   Copyright (c) 2025 Stanislav Shwartsman
//          Written by Stanislav Shwartsman [sshwarts at sourceforge net]
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA B 02110-1301 USA
//
/////////////////////////////////////////////////////////////////////////

#define NEED_CPU_REG_SHORTCUTS 1
#include "bochs.h"
#include "cpu.h"
#define LOG_THIS BX_CPU_THIS_PTR

#if BX_SUPPORT_3DNOW && BX_CPU_LEVEL >= 5

extern softfloat_status_t prepare_softfloat_status_word_3dnow(int rounding_mode);

// Thanks for Cristina Iordache and David W. Matula for reverse-engineering of 3dNow! Reciprocal instructions and writing the article
// Analysis of Reciprocal and Square Root Reciprocal Instructions in the AMD K6-2 Implementation of 3DNow!
// https://www.sciencedirect.com/science/article/pii/S1571066105806218

#include "softfloat3e/include/softfloat.h"
#include "fpu/softfloat-specialize.h"

static Bit16u rcp_tableH[1024] = {
  0xFFBE, 0xFF3E, 0xFEBF, 0xFE40, 0xFDC1, 0xFD42, 0xFCC3, 0xFC45, 0xFBC7, 0xFB49, 0xFACC, 0xFA4F, 0xF9D1, 0xF955, 0xF8D8, 0xF85C, // 0x00    FRACTION MSB[b1..b5]
  0xF7E0, 0xF764, 0xF6E8, 0xF66D, 0xF5F2, 0xF577, 0xF4FC, 0xF482, 0xF407, 0xF38D, 0xF314, 0xF29A, 0xF221, 0xF1A8, 0xF12F, 0xF0B7,
  0xF03F, 0xEFC6, 0xEF4F, 0xEED7, 0xEE5F, 0xEDE8, 0xED71, 0xECFA, 0xEC84, 0xEC0E, 0xEB97, 0xEB22, 0xEAAC, 0xEA36, 0xE9C1, 0xE94C, // 0x01
  0xE8D7, 0xE863, 0xE7EE, 0xE77A, 0xE706, 0xE693, 0xE61F, 0xE5AC, 0xE539, 0xE4C6, 0xE453, 0xE3E1, 0xE36F, 0xE2FD, 0xE28B, 0xE219,
  0xE1A8, 0xE136, 0xE0C5, 0xE055, 0xDFE4, 0xDF74, 0xDF04, 0xDE94, 0xDE24, 0xDDB4, 0xDD45, 0xDCD6, 0xDC67, 0xDBF8, 0xDB8A, 0xDB1C, // 0x02
  0xDAAD, 0xDA3F, 0xD9D2, 0xD964, 0xD8F7, 0xD88A, 0xD81D, 0xD7B0, 0xD744, 0xD6D7, 0xD66B, 0xD5FF, 0xD593, 0xD528, 0xD4BC, 0xD451,
  0xD3E6, 0xD37B, 0xD311, 0xD2A6, 0xD23C, 0xD1D2, 0xD168, 0xD0FF, 0xD095, 0xD02C, 0xCFC3, 0xCF5A, 0xCEF1, 0xCE88, 0xCE20, 0xCDB8, // 0x03
  0xCD50, 0xCCE8, 0xCC80, 0xCC19, 0xCBB2, 0xCB4B, 0xCAE4, 0xCA7D, 0xCA16, 0xC9B0, 0xC94A, 0xC8E4, 0xC87E, 0xC818, 0xC7B3, 0xC74D,
  0xC6E9, 0xC684, 0xC61F, 0xC5BA, 0xC556, 0xC4F2, 0xC48D, 0xC42A, 0xC3C6, 0xC362, 0xC2FF, 0xC29C, 0xC239, 0xC1D6, 0xC173, 0xC110, // 0x04
  0xC0AE, 0xC04C, 0xBFEA, 0xBF88, 0xBF26, 0xBEC5, 0xBE63, 0xBE02, 0xBDA1, 0xBD40, 0xBCDF, 0xBC7F, 0xBC1E, 0xBBBE, 0xBB5E, 0xBAFE,
  0xBA9E, 0xBA3E, 0xB9DF, 0xB980, 0xB921, 0xB8C2, 0xB863, 0xB804, 0xB7A6, 0xB747, 0xB6E9, 0xB68B, 0xB62D, 0xB5D0, 0xB572, 0xB515, // 0x05
  0xB4B8, 0xB45B, 0xB3FE, 0xB3A1, 0xB344, 0xB2E8, 0xB28C, 0xB230, 0xB1D4, 0xB178, 0xB11C, 0xB0C1, 0xB065, 0xB00A, 0xAFAF, 0xAF54,
  0xAEFA, 0xAE9F, 0xAE45, 0xADEB, 0xAD90, 0xAD36, 0xACDC, 0xAC83, 0xAC29, 0xABD0, 0xAB76, 0xAB1D, 0xAAC4, 0xAA6B, 0xAA13, 0xA9BA, // 0x06
  0xA962, 0xA909, 0xA8B1, 0xA859, 0xA801, 0xA7AA, 0xA752, 0xA6FB, 0xA6A3, 0xA64C, 0xA5F5, 0xA59E, 0xA548, 0xA4F1, 0xA49B, 0xA444,
  0xA3EE, 0xA398, 0xA342, 0xA2EC, 0xA297, 0xA241, 0xA1EC, 0xA197, 0xA141, 0xA0ED, 0xA098, 0xA043, 0x9FEE, 0x9F9A, 0x9F46, 0x9EF2, // 0x07
  0x9E9E, 0x9E4A, 0x9DF6, 0x9DA2, 0x9D4F, 0x9CFC, 0x9CA8, 0x9C55, 0x9C02, 0x9BB0, 0x9B5D, 0x9B0A, 0x9AB8, 0x9A66, 0x9A13, 0x99C1,
  0x996F, 0x991E, 0x98CC, 0x987A, 0x9829, 0x97D8, 0x9787, 0x9736, 0x96E5, 0x9694, 0x9643, 0x95F3, 0x95A2, 0x9552, 0x9502, 0x94B2, // 0x08
  0x9462, 0x9412, 0x93C2, 0x9373, 0x9323, 0x92D4, 0x9285, 0x9236, 0x91E7, 0x9198, 0x9149, 0x90FB, 0x90AC, 0x905E, 0x9010, 0x8FC2,
  0x8F74, 0x8F26, 0x8ED8, 0x8E8B, 0x8E3D, 0x8DF0, 0x8DA3, 0x8D56, 0x8D08, 0x8CBC, 0x8C6F, 0x8C22, 0x8BD6, 0x8B89, 0x8B3D, 0x8AF1, // 0x09
  0x8AA4, 0x8A58, 0x8A0D, 0x89C1, 0x8975, 0x892A, 0x88DE, 0x8893, 0x8848, 0x87FD, 0x87B2, 0x8767, 0x871C, 0x86D1, 0x8687, 0x863C,
  0x85F2, 0x85A8, 0x855E, 0x8514, 0x84CA, 0x8481, 0x8437, 0x83ED, 0x83A4, 0x835B, 0x8311, 0x82C8, 0x827F, 0x8236, 0x81EE, 0x81A5, // 0x0A
  0x815C, 0x8114, 0x80CC, 0x8083, 0x803B, 0x7FF3, 0x7FAB, 0x7F63, 0x7F1C, 0x7ED4, 0x7E8D, 0x7E45, 0x7DFE, 0x7DB7, 0x7D70, 0x7D29,
  0x7CE1, 0x7C9A, 0x7C54, 0x7C0D, 0x7BC7, 0x7B80, 0x7B3A, 0x7AF4, 0x7AAE, 0x7A68, 0x7A22, 0x79DC, 0x7996, 0x7951, 0x790B, 0x78C6, // 0x0B
  0x7881, 0x783C, 0x77F6, 0x77B2, 0x776D, 0x7728, 0x76E3, 0x769F, 0x765A, 0x7616, 0x75D1, 0x758D, 0x7549, 0x7505, 0x74C1, 0x747E,
  0x743A, 0x73F7, 0x73B3, 0x7370, 0x732C, 0x72E9, 0x72A6, 0x7263, 0x7220, 0x71DD, 0x719B, 0x7158, 0x7115, 0x70D3, 0x7090, 0x704E, // 0x0C
  0x700C, 0x6FCA, 0x6F88, 0x6F46, 0x6F04, 0x6EC2, 0x6E81, 0x6E3F, 0x6DFE, 0x6DBC, 0x6D7B, 0x6D3A, 0x6CF9, 0x6CB8, 0x6C77, 0x6C36,
  0x6BF5, 0x6BB4, 0x6B74, 0x6B33, 0x6AF3, 0x6AB3, 0x6A73, 0x6A33, 0x69F2, 0x69B3, 0x6973, 0x6933, 0x68F3, 0x68B4, 0x6874, 0x6835, // 0x0D
  0x67F6, 0x67B6, 0x6777, 0x6738, 0x66F9, 0x66BA, 0x667B, 0x663D, 0x65FE, 0x65C0, 0x6581, 0x6543, 0x6505, 0x64C6, 0x6488, 0x644A,
  0x640D, 0x63CF, 0x6391, 0x6354, 0x6316, 0x62D8, 0x629B, 0x625E, 0x6220, 0x61E3, 0x61A6, 0x6169, 0x612C, 0x60EF, 0x60B2, 0x6076, // 0x0E
  0x6039, 0x5FFD, 0x5FC0, 0x5F84, 0x5F48, 0x5F0B, 0x5ECF, 0x5E93, 0x5E57, 0x5E1B, 0x5DDF, 0x5DA4, 0x5D68, 0x5D2C, 0x5CF1, 0x5CB6,
  0x5C7A, 0x5C3F, 0x5C04, 0x5BC8, 0x5B8D, 0x5B53, 0x5B18, 0x5ADD, 0x5AA2, 0x5A68, 0x5A2D, 0x59F2, 0x59B8, 0x597E, 0x5944, 0x5909, // 0x0F
  0x58CF, 0x5895, 0x585B, 0x5821, 0x57E8, 0x57AE, 0x5774, 0x573B, 0x5701, 0x56C8, 0x568F, 0x5655, 0x561C, 0x55E3, 0x55AA, 0x5571,
  0x5538, 0x54FF, 0x54C6, 0x548E, 0x5455, 0x541D, 0x53E4, 0x53AC, 0x5373, 0x533B, 0x5303, 0x52CB, 0x5293, 0x525B, 0x5223, 0x51EB, // 0x10
  0x51B4, 0x517C, 0x5144, 0x510D, 0x50D5, 0x509E, 0x5067, 0x5030, 0x4FF8, 0x4FC1, 0x4F8A, 0x4F53, 0x4F1D, 0x4EE6, 0x4EAF, 0x4E78,
  0x4E42, 0x4E0B, 0x4DD5, 0x4D9E, 0x4D68, 0x4D32, 0x4CFC, 0x4CC5, 0x4C8F, 0x4C59, 0x4C24, 0x4BEE, 0x4BB8, 0x4B82, 0x4B4D, 0x4B17, // 0x11
  0x4AE2, 0x4AAC, 0x4A77, 0x4A42, 0x4A0C, 0x49D7, 0x49A2, 0x496D, 0x4938, 0x4903, 0x48CE, 0x489A, 0x4865, 0x4830, 0x47FC, 0x47C7,
  0x4793, 0x475E, 0x472A, 0x46F6, 0x46C2, 0x468E, 0x465A, 0x4626, 0x45F2, 0x45BE, 0x458A, 0x4556, 0x4523, 0x44EF, 0x44BC, 0x4488, // 0x12
  0x4455, 0x4421, 0x43EE, 0x43BB, 0x4388, 0x4355, 0x4322, 0x42EF, 0x42BC, 0x4289, 0x4256, 0x4224, 0x41F1, 0x41BE, 0x418C, 0x4159,
  0x4127, 0x40F5, 0x40C2, 0x4090, 0x405E, 0x402C, 0x3FFA, 0x3FC8, 0x3F96, 0x3F64, 0x3F32, 0x3F01, 0x3ECF, 0x3E9D, 0x3E6C, 0x3E3A, // 0x13
  0x3E09, 0x3DD8, 0x3DA6, 0x3D75, 0x3D44, 0x3D13, 0x3CE2, 0x3CB1, 0x3C80, 0x3C4F, 0x3C1E, 0x3BED, 0x3BBD, 0x3B8C, 0x3B5B, 0x3B2B,
  0x3AFB, 0x3ACA, 0x3A9A, 0x3A69, 0x3A39, 0x3A09, 0x39D9, 0x39A9, 0x3979, 0x3949, 0x3919, 0x38E9, 0x38B9, 0x388A, 0x385A, 0x382A, // 0x14
  0x37FB, 0x37CB, 0x379C, 0x376C, 0x373D, 0x370E, 0x36DF, 0x36AF, 0x3680, 0x3651, 0x3622, 0x35F3, 0x35C5, 0x3596, 0x3567, 0x3538,
  0x350A, 0x34DB, 0x34AD, 0x347E, 0x3450, 0x3421, 0x33F3, 0x33C5, 0x3396, 0x3368, 0x333A, 0x330C, 0x32DE, 0x32B0, 0x3282, 0x3254, // 0x15
  0x3227, 0x31F9, 0x31CB, 0x319D, 0x3170, 0x3142, 0x3115, 0x30E7, 0x30BA, 0x308D, 0x305F, 0x3032, 0x3005, 0x2FD8, 0x2FAB, 0x2F7E,
  0x2F51, 0x2F24, 0x2EF7, 0x2ECB, 0x2E9E, 0x2E71, 0x2E45, 0x2E18, 0x2DEB, 0x2DBF, 0x2D93, 0x2D66, 0x2D3A, 0x2D0E, 0x2CE1, 0x2CB5, // 0x16
  0x2C89, 0x2C5D, 0x2C31, 0x2C05, 0x2BD9, 0x2BAD, 0x2B81, 0x2B56, 0x2B2A, 0x2AFE, 0x2AD3, 0x2AA7, 0x2A7B, 0x2A50, 0x2A25, 0x29F9,
  0x29CE, 0x29A3, 0x2978, 0x294C, 0x2921, 0x28F6, 0x28CB, 0x28A0, 0x2875, 0x284A, 0x281F, 0x27F5, 0x27CA, 0x279F, 0x2775, 0x274A, // 0x17
  0x271F, 0x26F5, 0x26CA, 0x26A0, 0x2676, 0x264B, 0x2621, 0x25F7, 0x25CD, 0x25A3, 0x2578, 0x254E, 0x2524, 0x24FB, 0x24D1, 0x24A7,
  0x247C, 0x2453, 0x2429, 0x23FF, 0x23D6, 0x23AC, 0x2382, 0x2359, 0x2330, 0x2306, 0x22DD, 0x22B4, 0x228A, 0x2261, 0x2238, 0x220F, // 0x18
  0x21E6, 0x21BD, 0x2194, 0x216B, 0x2142, 0x2119, 0x20F0, 0x20C8, 0x209F, 0x2076, 0x204E, 0x2025, 0x1FFD, 0x1FD4, 0x1FAC, 0x1F83,
  0x1F5B, 0x1F33, 0x1F0B, 0x1EE3, 0x1EBA, 0x1E92, 0x1E6A, 0x1E42, 0x1E1A, 0x1DF2, 0x1DCA, 0x1DA2, 0x1D7B, 0x1D53, 0x1D2B, 0x1D03, // 0x19
  0x1CDC, 0x1CB4, 0x1C8D, 0x1C65, 0x1C3E, 0x1C16, 0x1BEF, 0x1BC8, 0x1BA0, 0x1B79, 0x1B52, 0x1B2B, 0x1B03, 0x1ADC, 0x1AB5, 0x1A8E,
  0x1A68, 0x1A41, 0x1A1A, 0x19F3, 0x19CC, 0x19A5, 0x197F, 0x1958, 0x1931, 0x190B, 0x18E4, 0x18BE, 0x1897, 0x1871, 0x184B, 0x1824, // 0x1A
  0x17FE, 0x17D8, 0x17B1, 0x178B, 0x1765, 0x173F, 0x1719, 0x16F3, 0x16CD, 0x16A7, 0x1681, 0x165B, 0x1635, 0x1610, 0x15EA, 0x15C4,
  0x159E, 0x1578, 0x1553, 0x152D, 0x1508, 0x14E2, 0x14BD, 0x1498, 0x1472, 0x144D, 0x1428, 0x1402, 0x13DD, 0x13B8, 0x1393, 0x136E, // 0x1B
  0x1349, 0x1324, 0x12FF, 0x12DA, 0x12B5, 0x1290, 0x126C, 0x1247, 0x1222, 0x11FD, 0x11D9, 0x11B4, 0x1190, 0x116B, 0x1147, 0x1122,
  0x10FE, 0x10DA, 0x10B6, 0x1091, 0x106D, 0x1049, 0x1025, 0x1000, 0x0FDC, 0x0FB8, 0x0F94, 0x0F70, 0x0F4C, 0x0F28, 0x0F04, 0x0EE1, // 0x1C
  0x0EBD, 0x0E99, 0x0E75, 0x0E52, 0x0E2E, 0x0E0A, 0x0DE7, 0x0DC3, 0x0DA0, 0x0D7C, 0x0D59, 0x0D35, 0x0D12, 0x0CEF, 0x0CCB, 0x0CA8,
  0x0C85, 0x0C62, 0x0C3F, 0x0C1C, 0x0BF9, 0x0BD6, 0x0BB3, 0x0B90, 0x0B6D, 0x0B4A, 0x0B27, 0x0B04, 0x0AE1, 0x0ABF, 0x0A9C, 0x0A79, // 0x1D
  0x0A57, 0x0A34, 0x0A11, 0x09EF, 0x09CC, 0x09AA, 0x0987, 0x0965, 0x0943, 0x0920, 0x08FE, 0x08DC, 0x08B9, 0x0897, 0x0875, 0x0853,
  0x0830, 0x080E, 0x07EC, 0x07CA, 0x07A8, 0x0786, 0x0764, 0x0743, 0x0721, 0x06FF, 0x06DD, 0x06BB, 0x069A, 0x0678, 0x0656, 0x0635, // 0x1E
  0x0613, 0x05F2, 0x05D0, 0x05AF, 0x058D, 0x056C, 0x054B, 0x0529, 0x0508, 0x04E7, 0x04C6, 0x04A4, 0x0483, 0x0462, 0x0441, 0x0420,
  0x03FF, 0x03DF, 0x03BE, 0x039D, 0x037C, 0x035B, 0x033A, 0x0319, 0x02F8, 0x02D8, 0x02B7, 0x0296, 0x0276, 0x0255, 0x0235, 0x0214, // 0x1F
  0x01F4, 0x01D3, 0x01B3, 0x0192, 0x0172, 0x0151, 0x0131, 0x0111, 0x00F1, 0x00D0, 0x00B0, 0x0090, 0x0070, 0x0050, 0x0030, 0x0010
};

// correction table
static Bit8s rcp_tableL[1024] = {
  62,  58,  54,  50,  46,  43,  39,  35,  31,  27,  23,  19,  15,  12,   8,   4,
   0,  -4,  -8, -12, -16, -20, -23, -27, -31, -35, -39, -43, -47, -51, -54, -58,
  58,  54,  51,  47,  43,  40,  36,  32,  29,  25,  21,  18,  14,  10,   7,   3,
   0,  -4,  -8, -11, -15, -19, -22, -26, -30, -33, -37, -41, -44, -48, -52, -55,
  55,  52,  48,  45,  41,  38,  34,  31,  27,  24,  21,  17,  14,  10,   7,   3,
   0,  -4,  -7, -10, -14, -17, -21, -24, -28, -31, -35, -38, -41, -45, -48, -52,
  52,  49,  45,  42,  39,  36,  32,  29,  26,  23,  19,  16,  13,  10,   6,   3,
   0,  -3,  -7, -10, -13, -16, -20, -23, -26, -29, -33, -36, -39, -42, -46, -49,
  49,  46,  43,  40,  37,  34,  31,  27,  24,  21,  18,  15,  12,   9,   6,   3,
   0,  -3,  -6,  -9, -13, -16, -19, -22, -25, -28, -31, -34, -37, -40, -43, -46,
  47,  44,  41,  38,  35,  32,  30,  27,  24,  21,  18,  15,  12,   9,   6,   3,
   0,  -3,  -5,  -8, -11, -14, -17, -20, -23, -26, -29, -32, -35, -38, -40, -43,
  44,  41,  38,  36,  33,  30,  27,  25,  22,  19,  16,  14,  11,   8,   5,   3,
   0,  -3,  -6,  -9, -11, -14, -17, -20, -22, -25, -28, -31, -33, -36, -39, -42,
  42,  39,  37,  34,  31,  29,  26,  24,  21,  18,  16,  13,  10,   8,   5,   3,
   0,  -3,  -5,  -8, -11, -13, -16, -18, -21, -24, -26, -29, -32, -34, -37, -39,
  40,  37,  35,  33,  30,  28,  25,  23,  20,  18,  15,  13,  10,   8,   5,   3,
   0,  -2,  -5,  -7, -10, -12, -15, -17, -20, -22, -25, -27, -30, -32, -35, -37,
  38,  36,  33,  31,  28,  26,  24,  21,  19,  17,  14,  12,   9,   7,   5,   2,
   0,  -2,  -5,  -7, -10, -12, -14, -17, -19, -21, -24, -26, -29, -31, -33, -36,
  36,  34,  31,  29,  27,  25,  22,  20,  18,  16,  13,  11,   9,   6,   4,   2,
   0,  -3,  -5,  -7,  -9, -12, -14, -16, -18, -21, -23, -25, -28, -30, -32, -34,
  35,  33,  31,  28,  26,  24,  22,  20,  18,  16,  13,  11,   9,   7,   5,   3,
   0,  -2,  -4,  -6,  -8, -10, -13, -15, -17, -19, -21, -23, -26, -28, -30, -32,
  33,  31,  29,  27,  25,  23,  21,  19,  16,  14,  12,  10,   8,   6,   4,   2,
   0,  -2,  -4,  -6,  -8, -10, -13, -15, -17, -19, -21, -23, -25, -27, -29, -31,
  32,  30,  28,  26,  24,  22,  20,  18,  16,  14,  12,  10,   8,   6,   4,   2,
   0,  -2,  -4,  -6,  -8, -10, -12, -14, -16, -17, -19, -21, -23, -25, -27, -29,
  30,  28,  26,  24,  22,  21,  19,  17,  15,  13,  11,   9,   7,   5,   3,   2,
   0,  -2,  -4,  -6,  -8, -10, -12, -14, -15, -17, -19, -21, -23, -25, -27, -29,
  29,  27,  25,  24,  22,  20,  18,  16,  14,  13,  11,   9,   7,   5,   4,   2,
   0,  -2,  -4,  -6,  -7,  -9, -11, -13, -15, -16, -18, -20, -22, -24, -25, -27,
  28,  26,  25,  23,  21,  19,  18,  16,  14,  12,  11,   9,   7,   5,   4,   2,
   0,  -2,  -3,  -5,  -7,  -9, -10, -12, -14, -16, -17, -19, -21, -23, -24, -26,
  27,  25,  24,  22,  20,  19,  17,  15,  14,  12,  10,   9,   7,   5,   4,   2,
   0,  -1,  -3,  -5,  -6,  -8, -10, -11, -13, -15, -16, -18, -20, -21, -23, -25,
  26,  24,  23,  21,  20,  18,  16,  15,  13,  12,  10,   8,   7,   5,   4,   2,
   0,  -1,  -3,  -5,  -6,  -8,  -9, -11, -13, -14, -16, -17, -19, -21, -22, -24,
  25,  23,  22,  20,  19,  17,  16,  14,  13,  11,  10,   8,   6,   5,   3,   2,
   0,  -1,  -3,  -4,  -6,  -7,  -9, -11, -12, -14, -15, -17, -18, -20, -21, -23,
  24,  23,  21,  20,  18,  17,  15,  14,  12,  11,   9,   8,   6,   5,   3,   2,
   0,  -1,  -3,  -4,  -6,  -7,  -9, -10, -12, -13, -15, -16, -18, -19, -21, -22,
  23,  22,  20,  19,  17,  16,  14,  13,  12,  10,   9,   7,   6,   4,   3,   2,
   0,  -1,  -3,  -4,  -6,  -7,  -8, -10, -11, -13, -14, -16, -17, -19, -20, -21,
  22,  21,  19,  18,  16,  15,  14,  12,  11,  10,   8,   7,   5,   4,   3,   1,
   0,  -1,  -3,  -4,  -6,  -7,  -8, -10, -11, -12, -14, -15, -17, -18, -19, -21,
  21,  20,  18,  17,  16,  14,  13,  12,  10,   9,   8,   6,   5,   4,   2,   1,
   0,  -2,  -3,  -4,  -6,  -7,  -8, -10, -11, -12, -14, -15, -16, -18, -19, -20,
  21,  20,  18,  17,  16,  15,  13,  12,  11,   9,   8,   7,   6,   4,   3,   2,
   0,  -1,  -2,  -3,  -5,  -6,  -7,  -9, -10, -11, -12, -14, -15, -16, -18, -19,
  20,  19,  18,  16,  15,  14,  13,  11,  10,   9,   8,   6,   5,   4,   3,   1,
   0,  -1,  -2,  -4,  -5,  -6,  -7,  -9, -10, -11, -12, -13, -15, -16, -17, -18,
  19,  18,  17,  15,  14,  13,  12,  11,   9,   8,   7,   6,   5,   3,   2,   1,
   0,  -1,  -3,  -4,  -5,  -6,  -7,  -9, -10, -11, -12, -13, -15, -16, -17, -18,
  19,  18,  17,  16,  14,  13,  12,  11,  10,   9,   7,   6,   5,   4,   3,   2,
   0,  -1,  -2,  -3,  -4,  -5,  -6,  -8, -9,  -10, -11, -12, -13, -15, -16, -17,
  18,  17,  16,  15,  14,  12,  11,  10,  9,    8,   7,   6,   5,   3,   2,   1,
   0,  -1,  -2,  -3,  -4,  -6,  -7,  -8, -9,  -10, -11, -12, -13, -14, -16, -17,
  17,  16,  15,  14,  13,  12,  10,   9,  8,    7,   6,   5,   4,   3,   2,   1,
   0,  -1,  -3,  -4,  -5,  -6,  -7,  -8, -9,  -10, -11, -12, -13, -14, -15, -17,
  17,  16,  15,  14,  13,  12,  11,  10,  9,    8,   7,   5,   4,   3,   2,   1,
   0,  -1,  -2,  -3,  -4,  -5,  -6,  -7, -8,   -9, -10, -11, -12, -13, -14, -16,
  16,  15,  14,  13,  12,  11,  10,   9,  8,    7,   6,   5,   4,   3,   2,   1,
   0,  -1,  -2,  -3,  -4,  -5,  -6,  -7, -8,   -9, -10, -11, -12, -13, -14, -15
};

// approximate reciprocal of scalar single precision FP with 14-bit precision
float32 approximate_rcp14_3dnow(float32 op)
{
  softfloat_class_t op_class = f32_class(op);
  bool sign = f32_sign(op);

  // Note that NaN handling is not documented in 3Dnow! manuals
  // The manual doesn't specify what result going to be if argument is NaN (undefined behavior)
  // This implementation choose IEEE-754 behavior which might not necessary match actual AMD's hardware
  switch(op_class) {
    case softfloat_zero:             // return maximum normal value with sign of the argument
    case softfloat_denormal:
      return packFloat32(sign, 0x7F, 0x7FFFFF);

    case softfloat_negative_inf:     // this is that actual AMD hardware returns
    case softfloat_positive_inf:
      return packFloat32(sign, 0x7F, 0x7FFE00);

    case softfloat_SNaN:
    case softfloat_QNaN:
      return convert_to_QNaN(op);

    case softfloat_normalized:
      break;
  }

  Bit32u fraction = f32_fraction(op);
  Bit16s exp = f32_exp(op);

  exp = 2 * FLOAT32_EXP_BIAS - 1 - exp;

  /* check for underflow */
  if (exp <= 0)
      return packFloat32(sign, 0, 0);

  // Approximate 1/(1.b1b2b3..bn) with T1(b1..bk..bm) + T2(b1..bk,bm+1..bn)

  // Table 1 is indexed with bits[b1..b5][b6...b10]
  // Table 2 is indexed with bits[b1..b5][b11..b15]

  Bit32u fraction_b1_b5   = (fraction >> 18) & 0x01F;
  Bit32u fraction_b1_b10  = (fraction >> 13) & 0x3FF;
  Bit32u fraction_b11_b15 = (fraction >>  8) & 0x01F;

  return packFloat32(sign, exp, Bit32u(int(rcp_tableH[fraction_b1_b10]) + int(rcp_tableL[(fraction_b1_b5 << 5) | fraction_b11_b15])) << 7);
}

void BX_CPP_AttrRegparmN(1) BX_CPU_C::PFRCP_PqQq(bxInstruction_c *i)
{
  BX_CPU_THIS_PTR FPU_check_pending_exceptions();

  BxPackedMmxRegister dst;
  float32 op;

  /* op is a register or memory reference */
  if (i->modC0()) {
    dst = BX_READ_MMX_REG(i->src());
    op = MMXUD0(dst);
  }
  else {
    bx_address eaddr = BX_CPU_RESOLVE_ADDR(i);
    /* pointer, segment address pair */
    op = read_virtual_dword(i->seg(), eaddr);
  }

  BX_CPU_THIS_PTR prepareFPU2MMX(); /* FPU2MMX transition */

  op = f32_denormal_to_zero(op);

  // for zero argument produce maximum normal value with the sign of source operand
  if ((op << 1) == 0) {
    op = packFloat32(f32_sign(op), 0x7F, 0x7FFFFF);
  }
  else {
    // Note that Inf/NaN handling is not documented in 3Dnow! manuals
    // This implementation choose IEEE-754 behavior which might not necessary match actual AMD's hardware
    op = approximate_rcp14_3dnow(op);
  }

  MMXUD0(dst) = op;
  MMXUD1(dst) = op;

  /* now write result back to destination */
  BX_WRITE_MMX_REG(i->dst(), dst);

  BX_NEXT_INSTR(i);
}

void BX_CPP_AttrRegparmN(1) BX_CPU_C::PFRCPIT1_PqQq(bxInstruction_c *i)
{
  BX_CPU_THIS_PTR FPU_check_pending_exceptions();

  BxPackedMmxRegister op1 = BX_READ_MMX_REG(i->dst()), op2;

  /* op2 is a register or memory reference */
  if (i->modC0()) {
    op2 = BX_READ_MMX_REG(i->src());
  }
  else {
    bx_address eaddr = BX_CPU_RESOLVE_ADDR(i);
    /* pointer, segment address pair */
    op2 = read_virtual_qword(i->seg(), eaddr);
  }

  BX_CPU_THIS_PTR prepareFPU2MMX(); /* FPU2MMX transition */

  // Compute first step of Newton-Raphson Approximation iteration
  // One iteration of the 1/b reciprocal approximation is:
  //       X1 = X0 * (2 - b * X0)

  // Use FMA call to compute (2 - b * X0) as the first step

  // Second step in instruction PFRCPIT2 just has to multiply arguments to complete approximation
  // PFRCPIT2 is aliased to PFMUL and doesn't have Bochs handler

  static softfloat_status_t status = prepare_softfloat_status_word_3dnow(softfloat_round_near_even);
  const float32 f32_two = 0x40000000;

  MMXUD0(op1) = f32_fnmadd(MMXUD0(op1), MMXUD0(op2), f32_two, &status);
  MMXUD1(op1) = f32_fnmadd(MMXUD1(op1), MMXUD1(op2), f32_two, &status);

  /* now write result back to destination */
  BX_WRITE_MMX_REG(i->dst(), op1);

  BX_NEXT_INSTR(i);
}

static Bit16u rsqrt_table1H[1024] = {
  0xFFDF, 0xFF9F, 0xFF60, 0xFF20, 0xFEE0, 0xFEA1, 0xFE61, 0xFE22, 0xFDE3, 0xFDA4, 0xFD64, 0xFD26, 0xFCE7, 0xFCA8, 0xFC69, 0xFC2A,
  0xFBEC, 0xFBAE, 0xFB6F, 0xFB31, 0xFAF3, 0xFAB5, 0xFA77, 0xFA39, 0xF9FB, 0xF9BD, 0xF980, 0xF942, 0xF905, 0xF8C7, 0xF88A, 0xF84D,
  0xF80F, 0xF7D2, 0xF795, 0xF758, 0xF71C, 0xF6DF, 0xF6A2, 0xF666, 0xF629, 0xF5ED, 0xF5B1, 0xF575, 0xF539, 0xF4FD, 0xF4C1, 0xF485,
  0xF449, 0xF40D, 0xF3D2, 0xF396, 0xF35B, 0xF320, 0xF2E4, 0xF2A9, 0xF26E, 0xF233, 0xF1F8, 0xF1BD, 0xF183, 0xF148, 0xF10D, 0xF0D3,
  0xF098, 0xF05E, 0xF023, 0xEFE9, 0xEFAF, 0xEF75, 0xEF3B, 0xEF01, 0xEEC7, 0xEE8E, 0xEE54, 0xEE1B, 0xEDE1, 0xEDA8, 0xED6E, 0xED35,
  0xECFC, 0xECC3, 0xEC8A, 0xEC51, 0xEC18, 0xEBDF, 0xEBA6, 0xEB6E, 0xEB35, 0xEAFD, 0xEAC4, 0xEA8C, 0xEA54, 0xEA1C, 0xE9E4, 0xE9AC,
  0xE974, 0xE93D, 0xE905, 0xE8CD, 0xE895, 0xE85E, 0xE826, 0xE7EF, 0xE7B8, 0xE780, 0xE749, 0xE712, 0xE6DB, 0xE6A4, 0xE66D, 0xE636,
  0xE5FF, 0xE5C9, 0xE592, 0xE55B, 0xE525, 0xE4EF, 0xE4B8, 0xE482, 0xE44C, 0xE416, 0xE3E0, 0xE3AA, 0xE374, 0xE33E, 0xE308, 0xE2D2,
  0xE29D, 0xE267, 0xE232, 0xE1FC, 0xE1C7, 0xE191, 0xE15C, 0xE127, 0xE0F2, 0xE0BD, 0xE088, 0xE053, 0xE01E, 0xDFEA, 0xDFB5, 0xDF80,
  0xDF4C, 0xDF17, 0xDEE3, 0xDEAF, 0xDE7A, 0xDE46, 0xDE12, 0xDDDE, 0xDDAA, 0xDD76, 0xDD42, 0xDD0E, 0xDCDB, 0xDCA7, 0xDC73, 0xDC40,
  0xDC0C, 0xDBD9, 0xDBA6, 0xDB72, 0xDB3F, 0xDB0C, 0xDAD9, 0xDAA6, 0xDA73, 0xDA40, 0xDA0D, 0xD9DA, 0xD9A8, 0xD975, 0xD942, 0xD910,
  0xD8DD, 0xD8AB, 0xD879, 0xD846, 0xD814, 0xD7E2, 0xD7B0, 0xD77E, 0xD74C, 0xD71A, 0xD6E8, 0xD6B7, 0xD685, 0xD653, 0xD622, 0xD5F0,
  0xD5BF, 0xD58D, 0xD55C, 0xD52B, 0xD4FA, 0xD4C8, 0xD497, 0xD466, 0xD435, 0xD404, 0xD3D4, 0xD3A3, 0xD372, 0xD341, 0xD311, 0xD2E0,
  0xD2B0, 0xD27F, 0xD24F, 0xD21F, 0xD1EE, 0xD1BE, 0xD18E, 0xD15E, 0xD12E, 0xD0FE, 0xD0CE, 0xD09E, 0xD06E, 0xD03F, 0xD00F, 0xCFDF,
  0xCFB0, 0xCF80, 0xCF51, 0xCF21, 0xCEF2, 0xCEC3, 0xCE94, 0xCE64, 0xCE35, 0xCE06, 0xCDD7, 0xCDA8, 0xCD79, 0xCD4B, 0xCD1C, 0xCCED,
  0xCCBE, 0xCC90, 0xCC61, 0xCC33, 0xCC04, 0xCBD6, 0xCBA8, 0xCB79, 0xCB4B, 0xCB1D, 0xCAEF, 0xCAC1, 0xCA93, 0xCA65, 0xCA37, 0xCA09,
  0xC9DA, 0xC9AD, 0xC97F, 0xC951, 0xC924, 0xC8F6, 0xC8C9, 0xC89B, 0xC86E, 0xC841, 0xC813, 0xC7E6, 0xC7B9, 0xC78C, 0xC75F, 0xC732,
  0xC705, 0xC6D8, 0xC6AB, 0xC67E, 0xC652, 0xC625, 0xC5F8, 0xC5CC, 0xC59F, 0xC573, 0xC546, 0xC51A, 0xC4ED, 0xC4C1, 0xC495, 0xC469,
  0xC43D, 0xC411, 0xC3E5, 0xC3B9, 0xC38D, 0xC361, 0xC335, 0xC309, 0xC2DE, 0xC2B2, 0xC286, 0xC25B, 0xC22F, 0xC204, 0xC1D8, 0xC1AD,
  0xC182, 0xC156, 0xC12B, 0xC100, 0xC0D5, 0xC0AA, 0xC07F, 0xC054, 0xC029, 0xBFFE, 0xBFD3, 0xBFA8, 0xBF7D, 0xBF53, 0xBF28, 0xBEFD,
  0xBED3, 0xBEA9, 0xBE7E, 0xBE54, 0xBE29, 0xBDFF, 0xBDD5, 0xBDAA, 0xBD80, 0xBD56, 0xBD2C, 0xBD02, 0xBCD8, 0xBCAE, 0xBC84, 0xBC5A,
  0xBC30, 0xBC07, 0xBBDD, 0xBBB3, 0xBB8A, 0xBB60, 0xBB37, 0xBB0D, 0xBAE4, 0xBABA, 0xBA91, 0xBA68, 0xBA3E, 0xBA15, 0xB9EC, 0xB9C3,
  0xB99A, 0xB971, 0xB948, 0xB91F, 0xB8F6, 0xB8CD, 0xB8A4, 0xB87C, 0xB853, 0xB82A, 0xB802, 0xB7D9, 0xB7B0, 0xB788, 0xB75F, 0xB737,
  0xB70F, 0xB6E6, 0xB6BE, 0xB696, 0xB66D, 0xB645, 0xB61D, 0xB5F5, 0xB5CD, 0xB5A5, 0xB57D, 0xB555, 0xB52D, 0xB506, 0xB4DE, 0xB4B6,
  0xB48E, 0xB466, 0xB43E, 0xB417, 0xB3EF, 0xB3C8, 0xB3A0, 0xB379, 0xB352, 0xB32A, 0xB303, 0xB2DC, 0xB2B5, 0xB28D, 0xB266, 0xB23F,
  0xB218, 0xB1F1, 0xB1CA, 0xB1A3, 0xB17D, 0xB156, 0xB12F, 0xB108, 0xB0E2, 0xB0BB, 0xB094, 0xB06E, 0xB047, 0xB021, 0xAFFA, 0xAFD4,
  0xAFAE, 0xAF87, 0xAF61, 0xAF3B, 0xAF15, 0xAEEE, 0xAEC8, 0xAEA2, 0xAE7C, 0xAE56, 0xAE30, 0xAE0A, 0xADE4, 0xADBE, 0xAD99, 0xAD73,
  0xAD4D, 0xAD27, 0xAD02, 0xACDC, 0xACB7, 0xAC91, 0xAC6B, 0xAC46, 0xAC21, 0xABFB, 0xABD6, 0xABB0, 0xAB8B, 0xAB66, 0xAB41, 0xAB1C,
  0xAAF7, 0xAAD2, 0xAAAD, 0xAA88, 0xAA63, 0xAA3E, 0xAA19, 0xA9F4, 0xA9CF, 0xA9AA, 0xA986, 0xA961, 0xA93C, 0xA917, 0xA8F3, 0xA8CE,
  0xA8AA, 0xA885, 0xA861, 0xA83C, 0xA818, 0xA7F4, 0xA7CF, 0xA7AB, 0xA787, 0xA763, 0xA73E, 0xA71A, 0xA6F6, 0xA6D2, 0xA6AE, 0xA68A,
  0xA666, 0xA642, 0xA61E, 0xA5FA, 0xA5D6, 0xA5B2, 0xA58F, 0xA56B, 0xA547, 0xA524, 0xA500, 0xA4DC, 0xA4B9, 0xA495, 0xA472, 0xA44F,
  0xA42B, 0xA408, 0xA3E4, 0xA3C1, 0xA39E, 0xA37B, 0xA358, 0xA334, 0xA311, 0xA2EE, 0xA2CB, 0xA2A8, 0xA285, 0xA262, 0xA23F, 0xA21C,
  0xA1FA, 0xA1D7, 0xA1B5, 0xA192, 0xA16F, 0xA14C, 0xA12A, 0xA107, 0xA0E5, 0xA0C2, 0xA0A0, 0xA07D, 0xA05B, 0xA038, 0xA016, 0x9FF4,
  0x9FD1, 0x9FAF, 0x9F8D, 0x9F6B, 0x9F48, 0x9F26, 0x9F04, 0x9EE2, 0x9EC0, 0x9E9E, 0x9E7C, 0x9E5A, 0x9E38, 0x9E16, 0x9DF5, 0x9DD3,
  0x9DB0, 0x9D8F, 0x9D6D, 0x9D4B, 0x9D2A, 0x9D08, 0x9CE6, 0x9CC5, 0x9CA3, 0x9C82, 0x9C60, 0x9C3F, 0x9C1E, 0x9BFC, 0x9BDB, 0x9BBA,
  0x9B98, 0x9B77, 0x9B56, 0x9B35, 0x9B14, 0x9AF3, 0x9AD1, 0x9AB0, 0x9A8F, 0x9A6E, 0x9A4D, 0x9A2D, 0x9A0C, 0x99EB, 0x99CA, 0x99A9,
  0x9989, 0x9968, 0x9947, 0x9927, 0x9906, 0x98E6, 0x98C5, 0x98A4, 0x9884, 0x9863, 0x9843, 0x9822, 0x9802, 0x97E2, 0x97C1, 0x97A1,
  0x9781, 0x9761, 0x9740, 0x9720, 0x9700, 0x96E0, 0x96C0, 0x96A0, 0x9680, 0x9660, 0x9640, 0x9620, 0x9600, 0x95E0, 0x95C0, 0x95A0,
  0x9580, 0x9560, 0x9540, 0x9521, 0x9501, 0x94E1, 0x94C2, 0x94A2, 0x9482, 0x9463, 0x9443, 0x9424, 0x9404, 0x93E5, 0x93C6, 0x93A6,
  0x9387, 0x9368, 0x9348, 0x9329, 0x930A, 0x92EB, 0x92CB, 0x92AC, 0x928D, 0x926E, 0x924F, 0x9230, 0x9211, 0x91F2, 0x91D3, 0x91B4,
  0x9196, 0x9177, 0x9158, 0x9139, 0x911A, 0x90FC, 0x90DD, 0x90BE, 0x90A0, 0x9081, 0x9062, 0x9044, 0x9025, 0x9007, 0x8FE8, 0x8FCA,
  0x8FAB, 0x8F8D, 0x8F6E, 0x8F50, 0x8F32, 0x8F13, 0x8EF5, 0x8ED7, 0x8EB8, 0x8E9A, 0x8E7C, 0x8E5E, 0x8E40, 0x8E22, 0x8E04, 0x8DE6,
  0x8DC7, 0x8DA9, 0x8D8B, 0x8D6D, 0x8D4F, 0x8D31, 0x8D13, 0x8CF6, 0x8CD8, 0x8CBA, 0x8C9C, 0x8C7F, 0x8C61, 0x8C43, 0x8C26, 0x8C08,
  0x8BEA, 0x8BCD, 0x8BAF, 0x8B92, 0x8B74, 0x8B57, 0x8B39, 0x8B1C, 0x8AFE, 0x8AE1, 0x8AC4, 0x8AA6, 0x8A89, 0x8A6C, 0x8A4E, 0x8A31,
  0x8A15, 0x89F8, 0x89DA, 0x89BD, 0x89A0, 0x8983, 0x8966, 0x8949, 0x892C, 0x890F, 0x88F2, 0x88D5, 0x88B8, 0x889B, 0x887F, 0x8862,
  0x8845, 0x8828, 0x880B, 0x87EF, 0x87D2, 0x87B5, 0x8799, 0x877C, 0x8760, 0x8743, 0x8726, 0x870A, 0x86ED, 0x86D1, 0x86B4, 0x8698,
  0x867B, 0x865F, 0x8642, 0x8626, 0x860A, 0x85EE, 0x85D1, 0x85B5, 0x8599, 0x857D, 0x8561, 0x8544, 0x8528, 0x850C, 0x84F0, 0x84D4,
  0x84B8, 0x849C, 0x8480, 0x8464, 0x8448, 0x842C, 0x8410, 0x83F5, 0x83D9, 0x83BD, 0x83A1, 0x8385, 0x836A, 0x834E, 0x8332, 0x8317,
  0x82FB, 0x82DF, 0x82C3, 0x82A8, 0x828C, 0x8271, 0x8255, 0x823A, 0x821E, 0x8203, 0x81E7, 0x81CC, 0x81B1, 0x8195, 0x817A, 0x815F,
  0x8143, 0x8128, 0x810D, 0x80F2, 0x80D6, 0x80BB, 0x80A0, 0x8085, 0x806A, 0x804F, 0x8034, 0x8019, 0x7FFE, 0x7FE3, 0x7FC8, 0x7FAD,
  0x7F93, 0x7F78, 0x7F5D, 0x7F42, 0x7F27, 0x7F0C, 0x7EF2, 0x7ED7, 0x7EBC, 0x7EA1, 0x7E87, 0x7E6C, 0x7E51, 0x7E37, 0x7E1C, 0x7E01,
  0x7DE7, 0x7DCC, 0x7DB2, 0x7D97, 0x7D7D, 0x7D62, 0x7D48, 0x7D2D, 0x7D13, 0x7CF9, 0x7CDE, 0x7CC4, 0x7CAA, 0x7C8F, 0x7C75, 0x7C5B,
  0x7C40, 0x7C26, 0x7C0C, 0x7BF2, 0x7BD8, 0x7BBD, 0x7BA3, 0x7B89, 0x7B6F, 0x7B55, 0x7B3B, 0x7B21, 0x7B07, 0x7AED, 0x7AD3, 0x7AB9,
  0x7AA0, 0x7A86, 0x7A6C, 0x7A52, 0x7A38, 0x7A1E, 0x7A05, 0x79EB, 0x79D1, 0x79B7, 0x799E, 0x7984, 0x796A, 0x7951, 0x7937, 0x791E,
  0x7904, 0x78EA, 0x78D1, 0x78B7, 0x789E, 0x7884, 0x786B, 0x7851, 0x7838, 0x781F, 0x7805, 0x77EC, 0x77D3, 0x77B9, 0x77A0, 0x7787,
  0x776D, 0x7754, 0x773B, 0x7722, 0x7709, 0x76F0, 0x76D6, 0x76BD, 0x76A4, 0x768B, 0x7672, 0x7659, 0x7640, 0x7627, 0x760E, 0x75F5,
  0x75DD, 0x75C4, 0x75AB, 0x7592, 0x7579, 0x7561, 0x7548, 0x752F, 0x7516, 0x74FD, 0x74E5, 0x74CC, 0x74B3, 0x749B, 0x7482, 0x7469,
  0x7451, 0x7438, 0x7420, 0x7407, 0x73EE, 0x73D6, 0x73BD, 0x73A5, 0x738C, 0x7374, 0x735C, 0x7343, 0x732B, 0x7312, 0x72FA, 0x72E2,
  0x72C9, 0x72B1, 0x7299, 0x7280, 0x7268, 0x7250, 0x7238, 0x721F, 0x7207, 0x71EF, 0x71D7, 0x71BF, 0x71A7, 0x718F, 0x7177, 0x715F,
  0x7147, 0x712F, 0x7117, 0x70FF, 0x70E7, 0x70CF, 0x70B7, 0x709F, 0x7087, 0x706F, 0x7057, 0x7040, 0x7028, 0x7010, 0x6FF8, 0x6FE0,
  0x6FC8, 0x6FB1, 0x6F99, 0x6F81, 0x6F6A, 0x6F52, 0x6F3A, 0x6F23, 0x6F0B, 0x6EF4, 0x6EDC, 0x6EC5, 0x6EAD, 0x6E96, 0x6E7E, 0x6E67,
  0x6E4F, 0x6E38, 0x6E20, 0x6E09, 0x6DF2, 0x6DDA, 0x6DC3, 0x6DAC, 0x6D94, 0x6D7D, 0x6D66, 0x6D4E, 0x6D37, 0x6D20, 0x6D09, 0x6CF2,
  0x6CDA, 0x6CC3, 0x6CAC, 0x6C95, 0x6C7E, 0x6C67, 0x6C4F, 0x6C38, 0x6C21, 0x6C0A, 0x6BF3, 0x6BDC, 0x6BC5, 0x6BAF, 0x6B98, 0x6B81,
  0x6B6A, 0x6B53, 0x6B3C, 0x6B25, 0x6B0E, 0x6AF8, 0x6AE1, 0x6ACA, 0x6AB3, 0x6A9C, 0x6A86, 0x6A6F, 0x6A58, 0x6A42, 0x6A2B, 0x6A14
};

static Bit8s rsqrt_table1L[1024] = {
  31, 29, 27, 25, 23,  21,  19,  17,  15,  13,  11,   9,   7,   6,   4,   2,
   0, -2, -4, -6, -8, -10, -12, -14, -16, -18, -20, -22, -24, -26, -28, -30,
  30, 28, 26, 24, 22,  21,  19,  17,  15,  13,  11,   9,   8,   6,   4,   2,
   0, -2, -4, -6, -7,  -9, -11, -13, -15, -17, -19, -20, -22, -24, -26, -28,
  29, 27, 25, 24, 22,  20,  18,  16,  15,  13,  11,   9,   7,   6,   4,   2,
   0, -1, -3, -5, -7,  -9, -10, -12, -14, -16, -18, -19, -21, -23, -25, -26,
  27, 25, 24, 22, 20,  18,  17,  15,  13,  12,  10,   8,   6,   5,   3,   1,
   0, -2, -4, -6, -7,  -9, -11, -12, -14, -16, -18, -19, -21, -23, -24, -26,
  26, 24, 23, 21, 19,  18,  16,  14,  13,  11,  10,   8,   6,   5,   3,   1,
   0, -2, -4, -5, -7,  -9, -10, -12, -13, -15, -17, -18, -20, -22, -23, -25,
  25, 23, 22, 20, 19,  17,  15,  14,  12,  11,   9,   8,   6,   4,   3,   1,
   0, -2, -3, -5, -7,  -8, -10, -11, -13, -14, -16, -18, -19, -21, -22, -24,
  24, 22, 21, 19, 18,  16,  15,  13,  12,  10,   9,   7,   6,   4,   3,   1,
   0, -2, -3, -5, -6,  -8,  -9, -11, -12, -14, -15, -17, -18, -20, -22, -23,
  23, 21, 20, 19, 17,  16,  14,  13,  11,  10,   8,   7,   5,   4,   3,   1,
   0, -2, -3, -5, -6,  -8,  -9, -11, -12, -14, -15, -16, -18, -19, -21, -22,
  23, 22, 20, 19, 17,  16,  15,  13,  12,  10,   9,   7,   6,   5,   3,   2,
   0, -1, -2, -4, -5,  -7,  -8,  -9, -11, -12, -14, -15, -16, -18, -19, -21,
  22, 21, 19, 18, 17,  15,  14,  12,  11,  10,   8,   7,   6,   4,   3,   2,
   0, -1, -2, -4, -5,  -7,  -8,  -9, -11, -12, -13, -15, -16, -17, -19, -20,
  21, 20, 18, 17, 16,  14,  13,  12,  10,   9,   8,   7,   5,   4,   3,   1,
   0, -1, -3, -4, -5,  -7,  -8,  -9, -10, -12, -13, -14, -16, -17, -18, -20,
  20, 19, 17, 16, 15,  14,  12,  11,  10,   9,   7,   6,   5,   4,   2,   1,
   0, -2, -3, -4, -5,  -7,  -8,  -9, -10, -12, -13, -14, -15, -17, -18, -19,
  20, 19, 18, 16, 15,  14,  13,  11,  10,   9,   8,   7,   5,   4,   3,   2,
   0, -1, -2, -3, -4,  -6,  -7,  -8,  -9, -11, -12, -13, -14, -15, -17, -18,
  19, 18, 17, 15, 14,  13,  12,  11,  10,   8,   7,   6,   5,   4,   2,   1,
   0, -1, -2, -3, -5,  -6,  -7,  -8,  -9, -11, -12, -13, -14, -15, -16, -18,
  18, 17, 16, 15, 13,  12,  11,  10,   9,   8,   7,   5,   4,   3,   2,   1,
   0, -1, -3, -4, -5,  -6,  -7,  -8,  -9, -11, -12, -13, -14, -15, -16, -17,
  18, 17, 16, 15, 14,  12,  11,  10,   9,   8,   7,   6,   5,   4,   2,   1,
   0, -1, -2, -3, -4,  -5,  -6,  -7,  -9, -10, -11, -12, -13, -14, -15, -16,
  17, 16, 15, 14, 13,  12,  11,   9,   8,   7,   6,   5,   4,   3,   2,   1,
   0, -1, -2, -3, -4,  -6,  -7,  -8,  -9, -10, -11, -12, -13, -14, -15, -16,
  17, 16, 15, 14, 13,  12,  11,  10,   9,   8,   7,   6,   4,   3,   2,   1,
   0, -1, -2, -3, -4,  -5,  -6,  -7,  -8,  -9, -10, -11, -12, -13, -14, -15,
  16, 15, 14, 13, 12,  11,  10,   9,   8,   7,   6,   5,   4,   3,   2,   1,
   0, -1, -2, -3, -4,  -5,  -6,  -7,  -8,  -9, -10, -11, -12, -13, -14, -15,
  16, 15, 14, 13, 12,  11,  10,   9,   8,   7,   6,   5,   4,   3,   2,   1,
   0, -1, -2, -3, -4,  -5,  -6,  -7,  -8,  -9, -10, -11, -11, -12, -13, -14,
  15, 14, 13, 12, 11,  10,   9,   8,   7,   6,   5,   4,   4,   3,   2,   1,
   0, -1, -2, -3, -4,  -5,  -6,  -7,  -8,  -9, -10, -11, -12, -13, -14, -15,
  15, 14, 13, 12, 11,  10,   9,   8,   8,   7,   6,   5,   4,   3,   2,   1,
   0, -1, -2, -3, -4,  -4,  -5,  -6,  -7,  -8,  -9, -10, -11, -12, -13, -14,
  14, 13, 12, 11, 10,   9,   9,   8,   7,   6,   5,   4,   3,   2,   1,   0,
   0, -1, -2, -3, -4,  -5,  -6,  -7,  -8,  -9,  -9, -10, -11, -12, -13, -14,
  14, 13, 12, 11, 10,  10,   9,   8,   7,   6,   5,   4,   3,   3,   2,   1,
   0, -1, -2, -3, -4,  -4,  -5,  -6,  -7,  -8,  -9, -10, -11, -11, -12, -13,
  14, 13, 12, 11, 11,  10,   9,   8,   7,   6,   5,   5,   4,   3,   2,   1,
   0, -1, -1, -2, -3,  -4,  -5,  -6,  -7,  -7,  -8,  -9, -10, -11, -12, -12,
  13, 12, 11, 10, 10,   9,   8,   7,   6,   5,   5,   4,   3,   2,   1,   0,
   0, -1, -2, -3, -4,  -4,  -5,  -6,  -7,  -8,  -9,  -9, -10, -11, -12, -13,
  13, 12, 11, 11, 10,   9,   8,   7,   6,   6,   5,   4,   3,   2,   2,   1,
   0, -1, -2, -2, -3,  -4,  -5,  -6,  -6,  -7,  -8,  -9, -10, -11, -11, -12,
  13, 12, 11, 11, 10,   9,   8,   7,   7,   6,   5,   4,   3,   3,   2,   1,
   0,  0, -1, -2, -3,  -4,  -4,  -5,  -6,  -7,  -8,  -8,  -9, -10, -11, -12,
  12, 11, 10, 10,  9,   8,   7,   7,   6,   5,   4,   3,   3,   2,   1,   0,
   0, -1, -2, -3, -3,  -4,  -5,  -6,  -7,  -7,  -8,  -9, -10, -10, -11, -12,
  12, 11, 10, 10,  9,   8,   7,   7,   6,   5,   4,   4,   3,   2,   1,   1,
   0, -1, -2, -2, -3,  -4,  -5,  -5,  -6,  -7,  -8,  -8,  -9, -10, -11, -11,
  12, 11, 10, 10,  9,   8,   8,   7,   6,   5,   5,   4,   3,   2,   2,   1,
   0, -1, -1, -2, -3,  -3,  -4,  -5,  -6,  -6,  -7,  -8,  -9,  -9, -10, -11,
  12, 11, 11, 10,  9,   8,   8,   7,   6,   6,   5,   4,   3,   3,   2,   1,
   0,  0, -1, -2, -2,  -3,  -4,  -5,  -5,  -6,  -7,  -7,  -8,  -9, -10, -10
};

static Bit16u rsqrt_table2H[1024] = {
  0x69F3, 0x69C6, 0x6998, 0x696B, 0x693E, 0x6911, 0x68E5, 0x68B8, 0x688B, 0x685E, 0x6832, 0x6805, 0x67D9, 0x67AC, 0x6780, 0x6754,
  0x6728, 0x66FB, 0x66CF, 0x66A3, 0x6677, 0x664B, 0x6620, 0x65F4, 0x65C8, 0x659C, 0x6571, 0x6545, 0x651A, 0x64EF, 0x64C3, 0x6498,
  0x646D, 0x6442, 0x6417, 0x63EB, 0x63C1, 0x6396, 0x636B, 0x6340, 0x6315, 0x62EB, 0x62C0, 0x6295, 0x626B, 0x6241, 0x6216, 0x61EC,
  0x61C2, 0x6197, 0x616D, 0x6143, 0x6119, 0x60EF, 0x60C5, 0x609B, 0x6072, 0x6048, 0x601E, 0x5FF5, 0x5FCB, 0x5FA2, 0x5F78, 0x5F4F,
  0x5F26, 0x5EFC, 0x5ED3, 0x5EAA, 0x5E81, 0x5E58, 0x5E2F, 0x5E06, 0x5DDD, 0x5DB4, 0x5D8C, 0x5D63, 0x5D3A, 0x5D12, 0x5CE9, 0x5CC1,
  0x5C98, 0x5C70, 0x5C47, 0x5C1F, 0x5BF7, 0x5BCF, 0x5BA7, 0x5B7F, 0x5B57, 0x5B2F, 0x5B07, 0x5ADF, 0x5AB7, 0x5A90, 0x5A68, 0x5A40,
  0x5A19, 0x59F1, 0x59CA, 0x59A3, 0x597B, 0x5954, 0x592D, 0x5905, 0x58DE, 0x58B7, 0x5890, 0x5869, 0x5842, 0x581B, 0x57F5, 0x57CE,
  0x57A7, 0x5780, 0x575A, 0x5733, 0x570D, 0x56E6, 0x56C0, 0x5699, 0x5673, 0x564D, 0x5626, 0x5600, 0x55DA, 0x55B4, 0x558E, 0x5568,
  0x5541, 0x551C, 0x54F6, 0x54D0, 0x54AA, 0x5485, 0x545F, 0x5439, 0x5414, 0x53EE, 0x53C9, 0x53A3, 0x537E, 0x5359, 0x5334, 0x530E,
  0x52E9, 0x52C4, 0x529F, 0x527A, 0x5255, 0x5230, 0x520B, 0x51E7, 0x51C2, 0x519D, 0x5178, 0x5154, 0x512F, 0x510B, 0x50E6, 0x50C2,
  0x509E, 0x5079, 0x5055, 0x5031, 0x500C, 0x4FE8, 0x4FC4, 0x4FA0, 0x4F7C, 0x4F58, 0x4F34, 0x4F10, 0x4EEC, 0x4EC9, 0x4EA5, 0x4E81,
  0x4E5D, 0x4E3A, 0x4E16, 0x4DF3, 0x4DCF, 0x4DAC, 0x4D88, 0x4D65, 0x4D42, 0x4D1E, 0x4CFB, 0x4CD8, 0x4CB5, 0x4C92, 0x4C6F, 0x4C4C,
  0x4C29, 0x4C06, 0x4BE3, 0x4BC0, 0x4B9D, 0x4B7B, 0x4B58, 0x4B35, 0x4B13, 0x4AF0, 0x4ACE, 0x4AAB, 0x4A89, 0x4A66, 0x4A44, 0x4A21,
  0x49FF, 0x49DD, 0x49BB, 0x4999, 0x4976, 0x4954, 0x4932, 0x4910, 0x48EE, 0x48CC, 0x48AB, 0x4889, 0x4867, 0x4845, 0x4823, 0x4802,
  0x47DF, 0x47BE, 0x479C, 0x477B, 0x4759, 0x4738, 0x4717, 0x46F5, 0x46D4, 0x46B3, 0x4691, 0x4670, 0x464F, 0x462E, 0x460D, 0x45EC,
  0x45CB, 0x45AA, 0x4589, 0x4568, 0x4547, 0x4526, 0x4506, 0x44E5, 0x44C4, 0x44A4, 0x4483, 0x4462, 0x4442, 0x4421, 0x4401, 0x43E0,
  0x43C0, 0x43A0, 0x4380, 0x4360, 0x433F, 0x431F, 0x42FF, 0x42DF, 0x42BF, 0x429F, 0x427F, 0x425F, 0x423F, 0x421F, 0x41FF, 0x41DF,
  0x41BF, 0x41A0, 0x4180, 0x4160, 0x4141, 0x4121, 0x4102, 0x40E2, 0x40C3, 0x40A3, 0x4084, 0x4064, 0x4045, 0x4026, 0x4006, 0x3FE7,
  0x3FC8, 0x3FA9, 0x3F8A, 0x3F6B, 0x3F4C, 0x3F2D, 0x3F0E, 0x3EEF, 0x3ED0, 0x3EB1, 0x3E92, 0x3E73, 0x3E55, 0x3E36, 0x3E17, 0x3DF9,
  0x3DDA, 0x3DBB, 0x3D9D, 0x3D7E, 0x3D60, 0x3D41, 0x3D23, 0x3D04, 0x3CE6, 0x3CC8, 0x3CA9, 0x3C8B, 0x3C6D, 0x3C4F, 0x3C30, 0x3C12,
  0x3BF4, 0x3BD6, 0x3BB8, 0x3B9A, 0x3B7C, 0x3B5E, 0x3B40, 0x3B22, 0x3B04, 0x3AE6, 0x3AC9, 0x3AAB, 0x3A8D, 0x3A70, 0x3A52, 0x3A34,
  0x3A17, 0x39F9, 0x39DC, 0x39BE, 0x39A1, 0x3983, 0x3966, 0x3949, 0x392B, 0x390E, 0x38F1, 0x38D4, 0x38B6, 0x3899, 0x387C, 0x385F,
  0x3843, 0x3826, 0x3809, 0x37EC, 0x37CF, 0x37B2, 0x3795, 0x3778, 0x375B, 0x373E, 0x3722, 0x3705, 0x36E8, 0x36CC, 0x36AF, 0x3693,
  0x3676, 0x3659, 0x363D, 0x3620, 0x3604, 0x35E8, 0x35CB, 0x35AF, 0x3593, 0x3576, 0x355A, 0x353E, 0x3522, 0x3506, 0x34E9, 0x34CD,
  0x34B1, 0x3495, 0x3479, 0x345D, 0x3441, 0x3425, 0x3409, 0x33ED, 0x33D1, 0x33B6, 0x339A, 0x337E, 0x3362, 0x3347, 0x332B, 0x330F,
  0x32F4, 0x32D8, 0x32BD, 0x32A1, 0x3286, 0x326A, 0x324F, 0x3233, 0x3218, 0x31FD, 0x31E1, 0x31C6, 0x31AB, 0x3190, 0x3174, 0x3159,
  0x313F, 0x3124, 0x3108, 0x30ED, 0x30D2, 0x30B7, 0x309C, 0x3081, 0x3067, 0x304C, 0x3031, 0x3016, 0x2FFB, 0x2FE0, 0x2FC6, 0x2FAB,
  0x2F90, 0x2F76, 0x2F5B, 0x2F40, 0x2F26, 0x2F0B, 0x2EF1, 0x2ED6, 0x2EBC, 0x2EA1, 0x2E87, 0x2E6C, 0x2E52, 0x2E38, 0x2E1D, 0x2E03,
  0x2DE8, 0x2DCE, 0x2DB4, 0x2D9A, 0x2D80, 0x2D66, 0x2D4C, 0x2D31, 0x2D17, 0x2CFD, 0x2CE3, 0x2CC9, 0x2CAF, 0x2C96, 0x2C7C, 0x2C62,
  0x2C48, 0x2C2E, 0x2C14, 0x2BFB, 0x2BE1, 0x2BC7, 0x2BAE, 0x2B94, 0x2B7A, 0x2B61, 0x2B47, 0x2B2D, 0x2B14, 0x2AFA, 0x2AE1, 0x2AC8,
  0x2AAE, 0x2A94, 0x2A7B, 0x2A62, 0x2A48, 0x2A2F, 0x2A16, 0x29FC, 0x29E3, 0x29CA, 0x29B1, 0x2998, 0x297F, 0x2966, 0x294C, 0x2933,
  0x291A, 0x2901, 0x28E8, 0x28CF, 0x28B7, 0x289E, 0x2885, 0x286C, 0x2853, 0x283A, 0x2821, 0x2809, 0x27F0, 0x27D7, 0x27BF, 0x27A6,
  0x278E, 0x2775, 0x275D, 0x2744, 0x272C, 0x2713, 0x26FB, 0x26E2, 0x26CA, 0x26B1, 0x2699, 0x2681, 0x2668, 0x2650, 0x2638, 0x261F,
  0x2607, 0x25EF, 0x25D7, 0x25BE, 0x25A6, 0x258E, 0x2576, 0x255E, 0x2546, 0x252E, 0x2516, 0x24FE, 0x24E6, 0x24CE, 0x24B6, 0x249E,
  0x2486, 0x246E, 0x2456, 0x243E, 0x2426, 0x240F, 0x23F7, 0x23DF, 0x23C8, 0x23B0, 0x2398, 0x2381, 0x2369, 0x2351, 0x233A, 0x2322,
  0x230B, 0x22F3, 0x22DC, 0x22C4, 0x22AD, 0x2295, 0x227E, 0x2267, 0x224F, 0x2238, 0x2221, 0x2209, 0x21F2, 0x21DB, 0x21C4, 0x21AD,
  0x2196, 0x217F, 0x2168, 0x2151, 0x213A, 0x2123, 0x210B, 0x20F4, 0x20DD, 0x20C6, 0x20B0, 0x2099, 0x2082, 0x206B, 0x2054, 0x203D,
  0x2026, 0x200F, 0x1FF9, 0x1FE2, 0x1FCB, 0x1FB4, 0x1F9E, 0x1F87, 0x1F70, 0x1F5A, 0x1F43, 0x1F2D, 0x1F16, 0x1EFF, 0x1EE9, 0x1ED2,
  0x1EBC, 0x1EA5, 0x1E8F, 0x1E78, 0x1E62, 0x1E4B, 0x1E35, 0x1E1F, 0x1E08, 0x1DF2, 0x1DDC, 0x1DC5, 0x1DAF, 0x1D99, 0x1D83, 0x1D6D,
  0x1D56, 0x1D40, 0x1D2A, 0x1D14, 0x1CFE, 0x1CE8, 0x1CD2, 0x1CBC, 0x1CA6, 0x1C90, 0x1C7A, 0x1C64, 0x1C4E, 0x1C38, 0x1C22, 0x1C0C,
  0x1BF6, 0x1BE0, 0x1BCB, 0x1BB5, 0x1B9F, 0x1B89, 0x1B74, 0x1B5E, 0x1B48, 0x1B32, 0x1B1D, 0x1B07, 0x1AF2, 0x1ADC, 0x1AC6, 0x1AB1,
  0x1A9B, 0x1A86, 0x1A70, 0x1A5B, 0x1A45, 0x1A30, 0x1A1B, 0x1A05, 0x19F0, 0x19DA, 0x19C5, 0x19B0, 0x199A, 0x1985, 0x1970, 0x195B,
  0x1945, 0x1930, 0x191B, 0x1905, 0x18F0, 0x18DB, 0x18C6, 0x18B1, 0x189C, 0x1887, 0x1872, 0x185D, 0x1848, 0x1833, 0x181E, 0x1809,
  0x17F4, 0x17DF, 0x17CA, 0x17B5, 0x17A0, 0x178C, 0x1777, 0x1762, 0x174D, 0x1738, 0x1724, 0x170F, 0x16FA, 0x16E5, 0x16D1, 0x16BC,
  0x16A8, 0x1694, 0x167F, 0x166A, 0x1656, 0x1641, 0x162D, 0x1618, 0x1604, 0x15EF, 0x15DB, 0x15C6, 0x15B2, 0x159D, 0x1589, 0x1575,
  0x1560, 0x154C, 0x1538, 0x1523, 0x150F, 0x14FB, 0x14E6, 0x14D2, 0x14BE, 0x14AA, 0x1496, 0x1481, 0x146D, 0x1459, 0x1445, 0x1431,
  0x141D, 0x1409, 0x13F4, 0x13E0, 0x13CC, 0x13B8, 0x13A4, 0x1390, 0x137D, 0x1369, 0x1355, 0x1341, 0x132D, 0x1319, 0x1305, 0x12F1,
  0x12DE, 0x12CA, 0x12B6, 0x12A2, 0x128E, 0x127B, 0x1267, 0x1253, 0x1240, 0x122C, 0x1218, 0x1205, 0x11F1, 0x11DE, 0x11CA, 0x11B6,
  0x11A3, 0x118F, 0x117C, 0x1168, 0x1155, 0x1141, 0x112E, 0x111A, 0x1107, 0x10F3, 0x10E0, 0x10CD, 0x10B9, 0x10A6, 0x1093, 0x107F,
  0x106C, 0x1059, 0x1045, 0x1032, 0x101F, 0x100C, 0x0FF9, 0x0FE5, 0x0FD2, 0x0FBF, 0x0FAC, 0x0F99, 0x0F86, 0x0F73, 0x0F60, 0x0F4D,
  0x0F3A, 0x0F27, 0x0F14, 0x0F01, 0x0EEE, 0x0EDB, 0x0EC8, 0x0EB5, 0x0EA3, 0x0E90, 0x0E7D, 0x0E6A, 0x0E57, 0x0E44, 0x0E31, 0x0E1F,
  0x0E0C, 0x0DF9, 0x0DE6, 0x0DD4, 0x0DC1, 0x0DAE, 0x0D9B, 0x0D89, 0x0D76, 0x0D63, 0x0D51, 0x0D3E, 0x0D2C, 0x0D19, 0x0D06, 0x0CF4,
  0x0CE1, 0x0CCF, 0x0CBC, 0x0CA9, 0x0C97, 0x0C85, 0x0C72, 0x0C60, 0x0C4D, 0x0C3B, 0x0C28, 0x0C16, 0x0C04, 0x0BF1, 0x0BDF, 0x0BCD,
  0x0BBA, 0x0BA8, 0x0B96, 0x0B83, 0x0B71, 0x0B5F, 0x0B4D, 0x0B3B, 0x0B28, 0x0B16, 0x0B04, 0x0AF2, 0x0AE0, 0x0ACE, 0x0ABC, 0x0AA9,
  0x0A97, 0x0A85, 0x0A73, 0x0A61, 0x0A4F, 0x0A3D, 0x0A2B, 0x0A19, 0x0A07, 0x09F5, 0x09E3, 0x09D1, 0x09BF, 0x09AD, 0x099C, 0x098A,
  0x0978, 0x0966, 0x0954, 0x0942, 0x0931, 0x091F, 0x090D, 0x08FB, 0x08EA, 0x08D8, 0x08C6, 0x08B4, 0x08A3, 0x0891, 0x087F, 0x086E,
  0x085C, 0x084A, 0x0839, 0x0827, 0x0816, 0x0804, 0x07F2, 0x07E1, 0x07CF, 0x07BE, 0x07AC, 0x079B, 0x078A, 0x0778, 0x0767, 0x0755,
  0x0744, 0x0732, 0x0721, 0x0710, 0x06FE, 0x06ED, 0x06DC, 0x06CA, 0x06B9, 0x06A8, 0x0696, 0x0685, 0x0674, 0x0663, 0x0651, 0x0640,
  0x062F, 0x061E, 0x060D, 0x05FB, 0x05EA, 0x05D9, 0x05C8, 0x05B7, 0x05A6, 0x0595, 0x0584, 0x0573, 0x0562, 0x0551, 0x0540, 0x052F,
  0x051E, 0x050D, 0x04FC, 0x04EB, 0x04DA, 0x04C9, 0x04B8, 0x04A7, 0x0496, 0x0485, 0x0474, 0x0464, 0x0453, 0x0442, 0x0431, 0x0420,
  0x0410, 0x0400, 0x03EF, 0x03DE, 0x03CD, 0x03BD, 0x03AC, 0x039B, 0x038B, 0x037A, 0x0369, 0x0359, 0x0348, 0x0337, 0x0327, 0x0316,
  0x0306, 0x02F5, 0x02E4, 0x02D4, 0x02C3, 0x02B3, 0x02A2, 0x0292, 0x0281, 0x0271, 0x0260, 0x0250, 0x0240, 0x022F, 0x021F, 0x020E,
  0x01FE, 0x01ED, 0x01DD, 0x01CD, 0x01BC, 0x01AC, 0x019C, 0x018B, 0x017B, 0x016B, 0x015B, 0x014A, 0x013A, 0x012A, 0x011A, 0x0109,
  0x00F9, 0x00E9, 0x00D9, 0x00C9, 0x00B9, 0x00A9, 0x0098, 0x0088, 0x0078, 0x0068, 0x0058, 0x0048, 0x0038, 0x0028, 0x0018, 0x0008
};

static Bit8s rsqrt_table2L[1024] = {
  22, 21, 19, 18, 16, 15, 14,  12,  11,  10,   8,   7,   5,   4,   3,   1,
   0, -2, -3, -4, -6, -7, -8, -10, -11, -13, -14, -15, -17, -18, -20, -21,
  21, 20, 18, 17, 16, 14, 13,  12,  10,   9,   8,   6,   5,   4,   2,   1,
   0, -2, -3, -4, -5, -7, -8,  -9, -11, -12, -13, -15, -16, -17, -19, -20,
  20, 19, 17, 16, 15, 14, 12,  11,  10,   9,   7,   6,   5,   4,   2,   1,
   0, -2, -3, -4, -5, -7, -8,  -9, -10, -12, -13, -14, -15, -17, -18, -19,
  19, 18, 17, 15, 14, 13, 12,  10,   9,   8,   7,   6,   4,   3,   2,   1,
   0, -2, -3, -4, -5, -6, -8,  -9, -10, -11, -13, -14, -15, -16, -17, -19,
  19, 18, 17, 15, 14, 13, 12,  11,  10,   8,   7,   6,   5,   4,   3,   2,
   0, -1, -2, -3, -4, -5, -7,  -8,  -9, -10, -11, -12, -14, -15, -16, -17,
  18, 17, 16, 15, 13, 12, 11,  10,   9,   8,   7,   6,   5,   3,   2,   1,
   0, -1, -2, -3, -4, -5, -7,  -8,  -9, -10, -11, -12, -13, -14, -16, -17,
  17, 16, 15, 14, 13, 12, 11,   9,   8,   7,   6,   5,   4,   3,   2,   1,
   0, -1, -2, -3, -4, -6, -7,  -8,  -9, -10, -11, -12, -13, -14, -15, -16,
  17, 16, 15, 14, 13, 12, 11,  10,   9,   8,   7,   6,   5,   4,   3,   1,
   0, -1, -2, -3, -4, -5, -6,  -7,  -8,  -9, -10, -11, -12, -13, -14, -15,
  16, 15, 14, 13, 12, 11, 10,   9,   8,   7,   6,   5,   4,   3,   2,   1,
   0, -1, -2, -3, -4, -5, -6,  -7,  -8,  -9, -10, -11, -12, -13, -14, -15,
  15, 14, 13, 12, 11, 10,  9,   8,   7,   6,   5,   4,   3,   2,   2,   1,
   0, -1, -2, -3, -4, -5, -6,  -7,  -8,  -9, -10, -11, -12, -13, -14, -15,
  15, 14, 13, 12, 11, 10,  9,   8,   8,   7,   6,   5,   4,   3,   2,   1,
   0, -1, -2, -3, -4, -4, -5,  -6,  -7,  -8,  -9, -10, -11, -12, -13, -14,
  14, 13, 12, 11, 10,  9,  9,   8,   7,   6,   5,   4,   3,   2,   1,   1,
   0, -1, -2, -3, -4, -5, -6,  -7,  -7,  -8,  -9, -10, -11, -12, -13, -14,
  14, 13, 12, 11, 10, 10,  9,   8,   7,   6,   5,   4,   4,   3,   2,   1,
   0, -1, -2, -2, -3, -4, -5,  -6,  -7,  -8,  -8,  -9, -10, -11, -12, -13,
  13, 12, 11, 10, 10,  9,  8,   7,   6,   5,   5,   4,   3,   2,   1,   0,
   0, -1, -2, -3, -4, -5, -5,  -6,  -7,  -8,  -9, -10, -10, -11, -12, -13,
  13, 12, 11, 11, 10,  9,  8,   7,   6,   6,   5,   4,   3,   2,   2,   1,
   0, -1, -2, -2, -3, -4, -5,  -6,  -6,  -7,  -8,  -9, -10, -10, -11, -12,
  13, 12, 11, 11, 10,  9,  8,   7,   7,   6,   5,   4,   4,   3,   2,   1,
   0,  0, -1, -2, -3, -3, -4,  -5,  -6,  -7,  -7,  -8,  -9, -10, -11, -11,
  12, 11, 10, 10,  9,  8,  7,   7,   6,   5,   4,   4,   3,   2,   1,   1,
   0, -1, -2, -2, -3, -4, -5,  -5,  -6,  -7,  -8,  -9,  -9, -10, -11, -12,
  12, 11, 10, 10,  9,  8,  8,   7,   6,   5,   5,   4,   3,   2,   2,   1,
   0, -1, -1, -2, -3, -3, -4,  -5,  -6,  -6,  -7,  -8,  -9,  -9, -10, -11,
  11, 10, 10,  9,  8,  7,  7,   6,   5,   5,   4,   3,   2,   2,   1,   0,
   0, -1, -2, -3, -3, -4, -5,  -5,  -6,  -7,  -8,  -8,  -9, -10, -10, -11,
  11, 10, 10,  9,  8,  7,  7,   6,   5,   5,   4,   3,   3,   2,   1,   1,
   0, -1, -2, -2, -3, -4, -4,  -5,  -6,  -6,  -7,  -8,  -8,  -9, -10, -11,
  11, 10, 10,  9,  8,  8,  7,   6,   6,   5,   4,   4,   3,   2,   2,   1,
   0,  0, -1, -2, -3, -3, -4,  -5,  -5,  -6,  -7,  -7,  -8,  -9,  -9, -10,
  11, 10, 10,  9,  8,  8,  7,   6,   6,   5,   4,   4,   3,   2,   2,   1,
   0,  0, -1, -1, -2, -3, -3,  -4,  -5,  -5,  -6,  -7,  -7,  -8,  -9,  -9,
  10,  9,  9,  8,  7,  7,  6,   5,   5,   4,   4,   3,   2,   2,   1,   0,
   0, -1, -2, -2, -3, -3, -4,  -5,  -5,  -6,  -7,  -7,  -8,  -9,  -9, -10,
  10,  9,  9,  8,  7,  7,  6,   6,   5,   4,   4,   3,   3,   2,   1,   1,
   0, -1, -1, -2, -2, -3, -4,  -4,  -5,  -6,  -6,  -7,  -7,  -8,  -9,  -9,
  10,  9,  9,  8,  8,  7,  6,   6,   5,   5,   4,   3,   3,   2,   2,   1,
   0,  0, -1, -2, -2, -3, -3,  -4,  -5,  -5,  -6,  -6,  -7,  -8,  -8,  -9,
   9,  8,  8,  7,  7,  6,  5,   5,   4,   4,   3,   2,   2,   1,   1,   0,
   0, -1, -2, -2, -3, -3, -4,  -5,  -5,  -6,  -6,  -7,  -7,  -8,  -9,  -9,
   9,  8,  8,  7,  7,  6,  6,   5,   4,   4,   3,   3,   2,   2,   1,   0,
   0, -1, -1, -2, -2, -3, -4,  -4,  -5,  -5,  -6,  -7,  -7,  -8,  -8,  -9,
   9,  8,  8,  7,  7,  6,  6,   5,   4,   4,   3,   3,   2,   2,   1,   1,
   0, -1, -1, -2, -2, -3, -3,  -4,  -4,  -5,  -6,  -6,  -7,  -7,  -8,  -8,
   9,  8,  8,  7,  7,  6,  6,   5,   5,   4,   4,   3,   2,   2,   1,   1,
   0,  0, -1, -1, -2, -2, -3,  -4,  -4,  -5,  -5,  -6,  -6,  -7,  -7,  -8,
   9,  8,  8,  7,  7,  6,  6,   5,   5,   4,   4,   3,   3,   2,   2,   1,
   0,  0, -1, -1, -2, -2, -3,  -3,  -4,  -4,  -5,  -5,  -6,  -6,  -7,  -8,
   8,  7,  7,  6,  6,  5,  5,   4,   4,   3,   3,   2,   2,   1,   1,   0,
   0, -1, -1, -2, -2, -3, -3,  -4,  -4,  -5,  -6,  -6,  -7,  -7,  -8,  -8,
   8,  7,  7,  6,  6,  5,  5,   4,   4,   3,   3,   2,   2,   1,   1,   0,
   0, -1, -1, -2, -2, -3, -3,  -4,  -4,  -5,  -5,  -6,  -6,  -7,  -7,  -8
};


// approximate reciprocal sqrt of scalar single precision FP with 14-bit precision
float32 approximate_rsqrt14_3dnow(float32 op)
{
  softfloat_class_t op_class = f32_class(op);
  bool sign = f32_sign(op);

  // Note that NaN handling is not documented in 3Dnow! manuals
  // The manual doesn't specify what result going to be if argument is NaN (undefined behavior)
  // This implementation choose IEEE-754 behavior which might not necessary match actual AMD's hardware
  switch(op_class) {
    case softfloat_zero:             // return maximum normal value with sign of the argument
    case softfloat_denormal:
      return packFloat32(sign, 0x7F, 0x7FFFFF);

    case softfloat_negative_inf:     // this is that actual AMD hardware returns
    case softfloat_positive_inf:
      return packFloat32(sign, 0x1F, 0x7FFF00);

    case softfloat_SNaN:
    case softfloat_QNaN:
      return convert_to_QNaN(op);

    case softfloat_normalized:
      break;
  };

  if (sign == 1)
    return float32_default_nan;

  Bit32u fraction = f32_fraction(op);
  Bit16s exp = f32_exp(op);

  const Bit16u *rsqrt_tableH = (exp & 1) ? rsqrt_table1H : rsqrt_table2H;
  const Bit8s  *rsqrt_tableL = (exp & 1) ? rsqrt_table1L : rsqrt_table2L;

  // Approximate 1/sqrt(1.b1b2b3..bn) with T1(b1..bk..bm) + T2(b1..bk,bm+1..bn)

  // Table 1 is indexed with bits[b1..b5][b6...b10]
  // Table 2 is indexed with bits[b1..b5][b11..b15]

  Bit32u fraction_b1_b5   = (fraction >> 18) & 0x01F;
  Bit32u fraction_b1_b10  = (fraction >> 13) & 0x3FF;
  Bit32u fraction_b11_b15 = (fraction >>  8) & 0x01F;

  exp = 0x7E - ((exp - 0x7F) >> 1);

  return packFloat32(sign, exp, Bit32u(int(rsqrt_tableH[fraction_b1_b10]) + int(rsqrt_tableL[(fraction_b1_b5 << 5) | fraction_b11_b15])) << 7);
}

void BX_CPP_AttrRegparmN(1) BX_CPU_C::PFRSQRT_PqQq(bxInstruction_c *i)
{
  BX_CPU_THIS_PTR FPU_check_pending_exceptions();

  BxPackedMmxRegister dst;
  float32 op;

  /* op is a register or memory reference */
  if (i->modC0()) {
    dst = BX_READ_MMX_REG(i->src());
    op = MMXUD0(dst);
  }
  else {
    bx_address eaddr = BX_CPU_RESOLVE_ADDR(i);
    /* pointer, segment address pair */
    op = read_virtual_dword(i->seg(), eaddr);
  }

  BX_CPU_THIS_PTR prepareFPU2MMX(); /* FPU2MMX transition */

  op = f32_denormal_to_zero(op);

  // Negative operands are treated as positive operands for purposes of reciprocal square
  // root computation, with the sign of the result the same as the sign of the source operand
  Bit32u sign = op & 0x80000000; // remember the sign

  op &= 0x7FFFFFFF; // reset the sign for now

  // for zero argument produce maximum normal value with the sign of source operand
  if (op == 0) {
    op = sign ^ packFloat32(0, 0x7F, 0x7FFFFF);
  }
  else {
    // Note that Inf/NaN handling is not documented in 3Dnow! manuals
    // This implementation choose IEEE-754 behavior which might not necessary match actual AMD's hardware
    op = sign ^ approximate_rsqrt14_3dnow(op);
  }

  MMXUD0(dst) = op;
  MMXUD1(dst) = op;

  /* now write result back to destination */
  BX_WRITE_MMX_REG(i->dst(), dst);

  BX_NEXT_INSTR(i);
}

void BX_CPP_AttrRegparmN(1) BX_CPU_C::PFRSQIT1_PqQq(bxInstruction_c *i)
{
  BX_CPU_THIS_PTR FPU_check_pending_exceptions();

  BxPackedMmxRegister op1 = BX_READ_MMX_REG(i->dst()), op2;

  /* op2 is a register or memory reference */
  if (i->modC0()) {
    op2 = BX_READ_MMX_REG(i->src());
  }
  else {
    bx_address eaddr = BX_CPU_RESOLVE_ADDR(i);
    /* pointer, segment address pair */
    op2 = read_virtual_qword(i->seg(), eaddr);
  }

  BX_CPU_THIS_PTR prepareFPU2MMX(); /* FPU2MMX transition */

  // Compute first step of Newton-Raphson Approximation iteration
  // One iteration of the sqrt(1/b) reciprocal approximation is:
  //       X1 = 0.5 * X0 * (3 - b * X0^2)

  // Usage for the instruction to compute a = 1/sqrt(b):

  //    X0 = PFRSQRT(b)                 initial approximation for 1/sqrt(b)
  //    X1 = PFMUL(X0, X0)              compute X0^2
  //    X2 = PFRSQIT1(b,X1)             this instruction compute 1st step of approximation 0.5 * (3 - b * X0^2)
  //                                    from arguments provided to the instruction are: X0^2 and b
  //     a = PFRCPIT2(X2,X0)            multiply by X0

  static softfloat_status_t status = prepare_softfloat_status_word_3dnow(softfloat_round_near_even);
  const float32 f32_three = 0x40400000, f32_half = 0x3f000000;

  // Use FMA call to compute (3 - b * X0^2) as the first step
  MMXUD0(op1) = f32_fnmadd(MMXUD0(op1), MMXUD0(op2), f32_three, &status);
  MMXUD1(op1) = f32_fnmadd(MMXUD1(op1), MMXUD1(op2), f32_three, &status);

  // Multiply result by 0.5
  MMXUD0(op1) = f32_mul(MMXUD0(op1), f32_half, &status);
  MMXUD1(op1) = f32_mul(MMXUD1(op1), f32_half, &status);

  /* now write result back to destination */
  BX_WRITE_MMX_REG(i->dst(), op1);

  BX_NEXT_INSTR(i);
}

#endif
