//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Detector/OffspecDetectorItem.cpp
//! @brief     Implements class OffspecDetectorItem
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Detector/OffspecDetectorItem.h"
#include "Base/Const/Units.h"
#include "Device/Detector/OffspecDetector.h"
#include "GUI/Support/XML/UtilXML.h"

namespace {
namespace Tag {

const QString AlphaAxis("AlphaAxis");
const QString PhiAxis("PhiAxis");
const QString BaseData("BaseData");

} // namespace Tag

} // namespace

OffspecDetectorItem::OffspecDetectorItem()
{
    m_phiAxis.initMin("Min", "Lower edge of first phi-bin", -1.0, Unit::degree,
                      RealLimits::limited(-90, 90));
    m_phiAxis.initMax("Max", "Upper edge of last phi-bin", 1.0, Unit::degree,
                      RealLimits::limited(-90, 90));

    m_alphaAxis.initMin("Min", "Lower edge of first alpha-bin", 0.0, Unit::degree,
                        RealLimits::limited(-90, 90));
    m_alphaAxis.initMax("Max", "Upper edge of last alpha-bin", 2.0, Unit::degree,
                        RealLimits::limited(-90, 90));
}

void OffspecDetectorItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // phi axis
    w->writeStartElement(Tag::PhiAxis);
    m_phiAxis.writeTo(w);
    w->writeEndElement();

    // alpha axis
    w->writeStartElement(Tag::AlphaAxis);
    m_alphaAxis.writeTo(w);
    w->writeEndElement();
}

void OffspecDetectorItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // phi axis
        if (tag == Tag::PhiAxis) {
            m_phiAxis.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // alpha axis
        } else if (tag == Tag::AlphaAxis) {
            m_alphaAxis.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}

std::unique_ptr<OffspecDetector> OffspecDetectorItem::createOffspecDetector() const
{
    const int n_x = m_phiAxis.nbins();
    const double x_min = Units::deg2rad(m_phiAxis.min());
    const double x_max = Units::deg2rad(m_phiAxis.max());

    const int n_y = m_alphaAxis.nbins();
    const double y_min = Units::deg2rad(m_alphaAxis.min());
    const double y_max = Units::deg2rad(m_alphaAxis.max());

    return std::make_unique<OffspecDetector>(n_x, x_min, x_max, n_y, y_min, y_max);
}

int OffspecDetectorItem::xSize() const
{
    return m_phiAxis.nbins();
}

int OffspecDetectorItem::ySize() const
{
    return m_alphaAxis.nbins();
}

void OffspecDetectorItem::setXSize(size_t nx)
{
    m_phiAxis.setNbins(nx);
}

void OffspecDetectorItem::setYSize(size_t ny)
{
    m_alphaAxis.setNbins(ny);
}

double OffspecDetectorItem::axesToCoreUnitsFactor() const
{
    return Units::deg;
}
