//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Sample/Multilayer/MultiLayer.cpp
//! @brief     Implements class MultiLayer.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "Sample/Multilayer/MultiLayer.h"
#include "Base/Util/Assert.h"
#include "Base/Util/StringUtil.h"
#include "Sample/Aggregate/ParticleLayout.h"
#include "Sample/Interface/LayerInterface.h"
#include "Sample/Interface/LayerRoughness.h"
#include "Sample/Material/MaterialUtil.h"
#include "Sample/Multilayer/Layer.h"
#include <utility>

MultiLayer::MultiLayer()
    : m_crossCorrLength(0)
{
    validateOrThrow();
}

MultiLayer::~MultiLayer() = default;

MultiLayer* MultiLayer::clone() const
{
    auto* result = new MultiLayer;
    result->setCrossCorrLength(crossCorrLength());
    result->setExternalField(externalField());
    result->setRoughnessModel(roughnessModel());
    for (size_t i = 0; i < numberOfLayers(); ++i) {
        const LayerInterface* interface = i > 0 ? m_interfaces[i - 1] : nullptr;
        if (i > 0 && interface->roughness())
            result->addLayerWithTopRoughness(std::as_const(*m_layers[i]), *interface->roughness());
        else
            result->addLayer(std::as_const(*m_layers[i]));
    }
    return result;
}

//! Adds layer with default (zero) roughness
void MultiLayer::addLayer(const Layer& layer)
{
    addLayerExec(layer, nullptr);
}

//! Adds layer with top roughness
void MultiLayer::addLayerWithTopRoughness(const Layer& layer, const LayerRoughness& roughness)
{
    addLayerExec(layer, &roughness);
}

//! Internal
void MultiLayer::addLayerExec(const Layer& layer, const LayerRoughness* roughness)
{
    const Layer* new_layer = layer.clone();

    if (numberOfLayers()) { // not the top layer
        const Layer* last_layer = m_layers.back();
        const LayerRoughness* new_roughness =
            roughness && roughness->sigma() != 0.0 ? roughness->clone() : nullptr;
        m_interfaces.emplace_back(
            LayerInterface::createInterface(last_layer, new_layer, new_roughness));

    } else { // the top layer
        if (new_layer->thickness() != 0.0)
            throw std::runtime_error("Invalid top layer; to indicate that it is semiinfinite,"
                                     " it must have a nominal thickness of 0");
        if (roughness)
            throw std::runtime_error("Invalid top layer with roughness");
    }

    m_layers.emplace_back(new_layer);
    m_validated = false;
}


const Layer* MultiLayer::layer(size_t i_layer) const
{
    return m_layers.at(i_layer);
}

const LayerInterface* MultiLayer::layerInterface(size_t i_interface) const
{
    return m_interfaces.at(i_interface);
}

void MultiLayer::setCrossCorrLength(double crossCorrLength)
{
    if (crossCorrLength < 0.0)
        throw std::runtime_error("MultiLayer::setCrossCorrLength called with negative argument");
    m_crossCorrLength = crossCorrLength;
    m_validated = false;
}

void MultiLayer::setExternalField(R3 ext_field)
{
    m_ext_field = ext_field;
    m_validated = false;
}

void MultiLayer::setRoughnessModel(RoughnessModel roughnessModel)
{
    m_roughness_model = roughnessModel;
    m_validated = false;
}

std::vector<const INode*> MultiLayer::nodeChildren() const
{
    std::vector<const INode*> result;
    const size_t N = m_layers.size();
    result.reserve(N + m_interfaces.size());

    for (size_t i = 0; i < N; ++i) {
        result.push_back(layer(i));
        if (i == N - 1)
            break;
        result.push_back(layerInterface(i));
    }
    return result;
}

double MultiLayer::hig(size_t i) const
{
    ASSERT(m_validated);
    ASSERT(i >= 1 && i < numberOfLayers());
    return ZInterfaces.at(i - 1);
}

double MultiLayer::low(size_t i) const
{
    ASSERT(m_validated);
    ASSERT(i < numberOfLayers() - 1);
    return ZInterfaces.at(i);
}

std::string MultiLayer::validate() const
{
    std::vector<std::string> errs;
    if (MaterialUtil::checkMaterialTypes(containedMaterials())
        == MATERIAL_TYPES::InvalidMaterialType)
        errs.push_back("Sample contains incompatible material definitions");

    for (size_t i = 0; i < m_layers.size(); ++i) {
        std::string err = m_layers[i]->validate();
        if (!err.empty())
            errs.push_back("{ layer " + std::to_string(i) + ": " + err + " }");
    }

    for (size_t i = 0; i < m_interfaces.size(); ++i) {
        std::string err = m_interfaces[i]->validate();
        if (!err.empty())
            errs.push_back("{ interface " + std::to_string(i) + ": " + err + " }");
    }

    requestGe0(errs, m_crossCorrLength, "CrossCorrelationLength");

    if (!errs.empty())
        return "[ " + Base::String::join(errs, ", ") + " ]";

    //! Precompute interface coordinates
    size_t N = numberOfLayers();
    if (N >= 1)
        ZInterfaces.resize(N - 1);
    if (N >= 2) {
        ZInterfaces[0] = 0;
        for (size_t i = 1; i < N - 1; ++i)
            ZInterfaces.at(i) = ZInterfaces.at(i - 1) - m_layers.at(i)->thickness();
    }

    m_validated = true;
    return "";
}
