#pragma once

#include <cstddef>
#include <iostream>
#include <limits>

namespace btllib::hashing_internals {

using NUM_HASHES_TYPE = uint8_t;
using K_TYPE = uint16_t;

template<typename T>
inline T
canonical(const T fwd, const T rev)
{
  return fwd + rev;
}

static_assert(std::numeric_limits<uint64_t>::max() + 1 == 0,
              "Integers don't overflow on this platform which is necessary for "
              "ntHash canonical hash computation.");

/**
 * Split a 64-bit word into 33 and 31-bit sub-words and left-rotate them
 * separately.
 * @param x A 64-bit unsigned integer
 * @return Split-rotation result
 */
inline uint64_t
srol(const uint64_t x)
{
  uint64_t m = ((x & 0x8000000000000000ULL) >> 30) | // NOLINT
               ((x & 0x100000000ULL) >> 32);         // NOLINT
  return ((x << 1) & 0xFFFFFFFDFFFFFFFFULL) | m;     // NOLINT
}

/**
 * Split a 64-bit word into 33 and 31-bit sub-words and left-rotate them
 * separately multiple times.
 * @param x A 64-bit unsigned integer
 * @param d Number of rotations
 * @return Split-rotation result
 */
inline uint64_t
srol(const uint64_t x, const unsigned d)
{
  uint64_t v = (x << d) | (x >> (64 - d));                         // NOLINT
  uint64_t y = (v ^ (v >> 33)) &                                   // NOLINT
               (std::numeric_limits<uint64_t>::max() >> (64 - d)); // NOLINT
  return v ^ (y | (y << 33));                                      // NOLINT
}

/**
 * Get the pre-computed result of srol for a specific base.
 * @param c The character as an unsigned char
 * @param d Number of rotations
 * @return Split-rotation result
 */
inline uint64_t
srol_table(unsigned char c, unsigned d);

/**
 * Split a 64-bit word into 33 and 31-bit sub-words and right-rotate them
 * separately.
 * @param x A 64-bit unsigned integer
 * @return Split-rotation result
 */
inline uint64_t
sror(const uint64_t x)
{
  uint64_t m = ((x & 0x200000000ULL) << 30) | ((x & 1ULL) << 32); // NOLINT
  return ((x >> 1) & 0xFFFFFFFEFFFFFFFFULL) | m;                  // NOLINT
}

// shift for generating multiple hash values
const int MULTISHIFT = 27;

// seed for generating multiple hash values
const uint64_t MULTISEED = 0x90b45d39fb6da1fa;

/**
 * Extend hash array using a base hash value.
 * @param fwd_hash Forward hash value
 * @param rev_hash Reverse hash value
 * @param k k-mer size
 * @param h Size of the resulting hash array (number of extra hashes minus one)
 * @param h_val Array of size h for storing the output hashes
 */
inline void
extend_hashes(uint64_t fwd_hash,
              uint64_t rev_hash,
              unsigned k,
              unsigned h,
              uint64_t* hash_array)
{
  uint64_t t_val;
  hash_array[0] = canonical(fwd_hash, rev_hash);
  for (unsigned i = 1; i < h; i++) {
    t_val = hash_array[0] * (i ^ k * MULTISEED);
    t_val ^= t_val >> MULTISHIFT;
    hash_array[i] = t_val;
  }
}

/**
 * Extend hash array using a base hash value.
 * @param base_hash Base hash value
 * @param k k-mer size
 * @param h Size of the resulting hash array (number of extra hashes minus one)
 * @param h_val Array of size h for storing the output hashes
 */
inline void
extend_hashes(uint64_t base_hash, unsigned k, unsigned h, uint64_t* hash_array)
{
  uint64_t t_val;
  hash_array[0] = base_hash;
  for (unsigned i = 1; i < h; i++) {
    t_val = hash_array[0] * (i ^ k * MULTISEED);
    t_val ^= t_val >> MULTISHIFT;
    hash_array[i] = t_val;
  }
}

// offset for the complement base in the random seeds table
const uint8_t CP_OFF = 0x07;

// 64-bit random seeds corresponding to bases and their complements
const uint64_t SEED_A = 0x3c8bfbb395c60474;
const uint64_t SEED_C = 0x3193c18562a02b4c;
const uint64_t SEED_G = 0x20323ed082572324;
const uint64_t SEED_T = 0x295549f54be24456;
const uint64_t SEED_N = 0x0000000000000000;

const int ASCII_SIZE = 256;

const uint64_t SEED_TAB[ASCII_SIZE] = {
  SEED_N, SEED_T, SEED_N, SEED_G, SEED_A, SEED_A, SEED_N, SEED_C, // 0..7
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 8..15
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 16..23
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 24..31
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 32..39
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 40..47
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 48..55
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 56..63
  SEED_N, SEED_A, SEED_N, SEED_C, SEED_N, SEED_N, SEED_N, SEED_G, // 64..71
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 72..79
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_T, SEED_T, SEED_N, SEED_N, // 80..87
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 88..95
  SEED_N, SEED_A, SEED_N, SEED_C, SEED_N, SEED_N, SEED_N, SEED_G, // 96..103
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 104..111
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_T, SEED_T, SEED_N, SEED_N, // 112..119
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 120..127
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 128..135
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 136..143
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 144..151
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 152..159
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 160..167
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 168..175
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 176..183
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 184..191
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 192..199
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 200..207
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 208..215
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 216..223
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 224..231
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 232..239
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, // 240..247
  SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N  // 248..255
};

const uint64_t A33R[33] = {
  0x195c60474, 0x12b8c08e9, 0x571811d3,  0xae3023a6,  0x15c60474c, 0xb8c08e99,
  0x171811d32, 0xe3023a65,  0x1c60474ca, 0x18c08e995, 0x11811d32b, 0x3023a657,
  0x60474cae,  0xc08e995c,  0x1811d32b8, 0x1023a6571, 0x474cae3,   0x8e995c6,
  0x11d32b8c,  0x23a65718,  0x474cae30,  0x8e995c60,  0x11d32b8c0, 0x3a657181,
  0x74cae302,  0xe995c604,  0x1d32b8c08, 0x1a6571811, 0x14cae3023, 0x995c6047,
  0x132b8c08e, 0x6571811d,  0xcae3023a
};

const uint64_t A31L[31] = {
  0x3c8bfbb200000000, 0x7917f76400000000, 0xf22feec800000000,
  0xe45fdd9200000000, 0xc8bfbb2600000000, 0x917f764e00000000,
  0x22feec9e00000000, 0x45fdd93c00000000, 0x8bfbb27800000000,
  0x17f764f200000000, 0x2feec9e400000000, 0x5fdd93c800000000,
  0xbfbb279000000000, 0x7f764f2200000000, 0xfeec9e4400000000,
  0xfdd93c8a00000000, 0xfbb2791600000000, 0xf764f22e00000000,
  0xeec9e45e00000000, 0xdd93c8be00000000, 0xbb27917e00000000,
  0x764f22fe00000000, 0xec9e45fc00000000, 0xd93c8bfa00000000,
  0xb27917f600000000, 0x64f22fee00000000, 0xc9e45fdc00000000,
  0x93c8bfba00000000, 0x27917f7600000000, 0x4f22feec00000000,
  0x9e45fdd800000000
};

const uint64_t C33R[33] = {
  0x162a02b4c, 0xc5405699,  0x18a80ad32, 0x115015a65, 0x2a02b4cb, 0x54056996,
  0xa80ad32c,  0x15015a658, 0xa02b4cb1,  0x140569962, 0x80ad32c5, 0x1015a658a,
  0x2b4cb15,   0x569962a,   0xad32c54,   0x15a658a8,  0x2b4cb150, 0x569962a0,
  0xad32c540,  0x15a658a80, 0xb4cb1501,  0x169962a02, 0xd32c5405, 0x1a658a80a,
  0x14cb15015, 0x9962a02b,  0x132c54056, 0x658a80ad,  0xcb15015a, 0x1962a02b4,
  0x12c540569, 0x58a80ad3,  0xb15015a6
};

const uint64_t C31L[31] = {
  0x3193c18400000000, 0x6327830800000000, 0xc64f061000000000,
  0x8c9e0c2200000000, 0x193c184600000000, 0x3278308c00000000,
  0x64f0611800000000, 0xc9e0c23000000000, 0x93c1846200000000,
  0x278308c600000000, 0x4f06118c00000000, 0x9e0c231800000000,
  0x3c18463200000000, 0x78308c6400000000, 0xf06118c800000000,
  0xe0c2319200000000, 0xc184632600000000, 0x8308c64e00000000,
  0x6118c9e00000000,  0xc23193c00000000,  0x1846327800000000,
  0x308c64f000000000, 0x6118c9e000000000, 0xc23193c000000000,
  0x8463278200000000, 0x8c64f0600000000,  0x118c9e0c00000000,
  0x23193c1800000000, 0x4632783000000000, 0x8c64f06000000000,
  0x18c9e0c200000000
};

const uint64_t G33R[33] = {
  0x82572324,  0x104ae4648, 0x95c8c91,   0x12b91922,  0x25723244,  0x4ae46488,
  0x95c8c910,  0x12b919220, 0x57232441,  0xae464882,  0x15c8c9104, 0xb9192209,
  0x172324412, 0xe4648825,  0x1c8c9104a, 0x191922095, 0x12324412b, 0x46488257,
  0x8c9104ae,  0x11922095c, 0x324412b9,  0x64882572,  0xc9104ae4,  0x1922095c8,
  0x124412b91, 0x48825723,  0x9104ae46,  0x122095c8c, 0x4412b919,  0x88257232,
  0x1104ae464, 0x2095c8c9,  0x412b9192
};

const uint64_t G31L[31] = {
  0x20323ed000000000, 0x40647da000000000, 0x80c8fb4000000000,
  0x191f68200000000,  0x323ed0400000000,  0x647da0800000000,
  0xc8fb41000000000,  0x191f682000000000, 0x323ed04000000000,
  0x647da08000000000, 0xc8fb410000000000, 0x91f6820200000000,
  0x23ed040600000000, 0x47da080c00000000, 0x8fb4101800000000,
  0x1f68203200000000, 0x3ed0406400000000, 0x7da080c800000000,
  0xfb41019000000000, 0xf682032200000000, 0xed04064600000000,
  0xda080c8e00000000, 0xb410191e00000000, 0x6820323e00000000,
  0xd040647c00000000, 0xa080c8fa00000000, 0x410191f600000000,
  0x820323ec00000000, 0x40647da00000000,  0x80c8fb400000000,
  0x10191f6800000000
};

const uint64_t T33R[33] = {
  0x14be24456, 0x97c488ad,  0x12f89115a, 0x5f1222b5,  0xbe24456a,  0x17c488ad4,
  0xf89115a9,  0x1f1222b52, 0x1e24456a5, 0x1c488ad4b, 0x189115a97, 0x11222b52f,
  0x24456a5f,  0x488ad4be,  0x9115a97c,  0x1222b52f8, 0x4456a5f1,  0x88ad4be2,
  0x1115a97c4, 0x22b52f89,  0x456a5f12,  0x8ad4be24,  0x115a97c48, 0x2b52f891,
  0x56a5f122,  0xad4be244,  0x15a97c488, 0xb52f8911,  0x16a5f1222, 0xd4be2445,
  0x1a97c488a, 0x152f89115, 0xa5f1222b
};

const uint64_t T31L[31] = {
  0x295549f400000000, 0x52aa93e800000000, 0xa55527d000000000,
  0x4aaa4fa200000000, 0x95549f4400000000, 0x2aa93e8a00000000,
  0x55527d1400000000, 0xaaa4fa2800000000, 0x5549f45200000000,
  0xaa93e8a400000000, 0x5527d14a00000000, 0xaa4fa29400000000,
  0x549f452a00000000, 0xa93e8a5400000000, 0x527d14aa00000000,
  0xa4fa295400000000, 0x49f452aa00000000, 0x93e8a55400000000,
  0x27d14aaa00000000, 0x4fa2955400000000, 0x9f452aa800000000,
  0x3e8a555200000000, 0x7d14aaa400000000, 0xfa29554800000000,
  0xf452aa9200000000, 0xe8a5552600000000, 0xd14aaa4e00000000,
  0xa295549e00000000, 0x452aa93e00000000, 0x8a55527c00000000,
  0x14aaa4fa00000000
};

const uint64_t N33R[33] = { SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N,
                            SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N,
                            SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N,
                            SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N,
                            SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N,
                            SEED_N, SEED_N, SEED_N };

const uint64_t N31L[31] = { SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N,
                            SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N,
                            SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N,
                            SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N,
                            SEED_N, SEED_N, SEED_N, SEED_N, SEED_N, SEED_N,
                            SEED_N };

const uint64_t* const MS_TAB_33R[ASCII_SIZE] = {
  N33R, T33R, N33R, G33R, A33R, A33R, N33R, C33R, // 0..7
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 8..15
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 16..23
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 24..31
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 32..39
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 40..47
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 48..55
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 56..63
  N33R, A33R, N33R, C33R, N33R, N33R, N33R, G33R, // 64..71
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 72..79
  N33R, N33R, N33R, N33R, T33R, T33R, N33R, N33R, // 80..87
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 88..95
  N33R, A33R, N33R, C33R, N33R, N33R, N33R, G33R, // 96..103
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 104..111
  N33R, N33R, N33R, N33R, T33R, T33R, N33R, N33R, // 112..119
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 120..127
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 128..135
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 136..143
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 144..151
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 152..159
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 160..167
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 168..175
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 176..183
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 184..191
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 192..199
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 200..207
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 208..215
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 216..223
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 224..231
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 232..239
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R, // 240..247
  N33R, N33R, N33R, N33R, N33R, N33R, N33R, N33R  // 248..255
};

const uint64_t* const MS_TAB_31L[ASCII_SIZE] = {
  N31L, T31L, N31L, G31L, A31L, A31L, N31L, C31L, // 0..7
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 8..15
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 16..23
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 24..31
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 32..39
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 40..47
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 48..55
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 56..63
  N31L, A31L, N31L, C31L, N31L, N31L, N31L, G31L, // 64..71
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 72..79
  N31L, N31L, N31L, N31L, T31L, T31L, N31L, N31L, // 80..87
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 88..95
  N31L, A31L, N31L, C31L, N31L, N31L, N31L, G31L, // 96..103
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 104..111
  N31L, N31L, N31L, N31L, T31L, T31L, N31L, N31L, // 112..119
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 120..127
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 128..135
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 136..143
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 144..151
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 152..159
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 160..167
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 168..175
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 176..183
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 184..191
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 192..199
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 200..207
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 208..215
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 216..223
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 224..231
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 232..239
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L, // 240..247
  N31L, N31L, N31L, N31L, N31L, N31L, N31L, N31L  // 248..255
};

inline uint64_t
srol_table(unsigned char c, unsigned d)
{
  return (MS_TAB_31L[c][d < 31 ? d : d % 31] | /* NOLINT */
          MS_TAB_33R[c][d < 33 ? d : d % 33]); /* NOLINT */
}

const uint8_t CONVERT_TAB[ASCII_SIZE] = {
  255, 255, 255, 255, 255, 255, 255, 255, // 0..7
  255, 255, 255, 255, 255, 255, 255, 255, // 8..15
  255, 255, 255, 255, 255, 255, 255, 255, // 16..23
  255, 255, 255, 255, 255, 255, 255, 255, // 24..31
  255, 255, 255, 255, 255, 255, 255, 255, // 32..39
  255, 255, 255, 255, 255, 255, 255, 255, // 40..47
  255, 255, 255, 255, 255, 255, 255, 255, // 48..55
  255, 255, 255, 255, 255, 255, 255, 255, // 56..63
  255, 0,   255, 1,   255, 255, 255, 2,   // 64..71
  255, 255, 255, 255, 255, 255, 255, 255, // 72..79
  255, 255, 255, 255, 3,   3,   255, 255, // 80..87
  255, 255, 255, 255, 255, 255, 255, 255, // 88..95
  255, 0,   255, 1,   255, 255, 255, 2,   // 96..103
  255, 255, 255, 255, 255, 255, 255, 255, // 104..111
  255, 255, 255, 255, 3,   3,   255, 255, // 112..119
  255, 255, 255, 255, 255, 255, 255, 255, // 120..127
  255, 255, 255, 255, 255, 255, 255, 255, // 128..135
  255, 255, 255, 255, 255, 255, 255, 255, // 136..143
  255, 255, 255, 255, 255, 255, 255, 255, // 144..151
  255, 255, 255, 255, 255, 255, 255, 255, // 152..159
  255, 255, 255, 255, 255, 255, 255, 255, // 160..167
  255, 255, 255, 255, 255, 255, 255, 255, // 168..175
  255, 255, 255, 255, 255, 255, 255, 255, // 176..183
  255, 255, 255, 255, 255, 255, 255, 255, // 184..191
  255, 255, 255, 255, 255, 255, 255, 255, // 192..199
  255, 255, 255, 255, 255, 255, 255, 255, // 200..207
  255, 255, 255, 255, 255, 255, 255, 255, // 208..215
  255, 255, 255, 255, 255, 255, 255, 255, // 216..223
  255, 255, 255, 255, 255, 255, 255, 255, // 224..231
  255, 255, 255, 255, 255, 255, 255, 255, // 232..239
  255, 255, 255, 255, 255, 255, 255, 255, // 240..247
  255, 255, 255, 255, 255, 255, 255, 255  // 248..255
};

const uint8_t RC_CONVERT_TAB[ASCII_SIZE] = {
  255, 255, 255, 255, 255, 255, 255, 255, // 0..7
  255, 255, 255, 255, 255, 255, 255, 255, // 8..15
  255, 255, 255, 255, 255, 255, 255, 255, // 16..23
  255, 255, 255, 255, 255, 255, 255, 255, // 24..31
  255, 255, 255, 255, 255, 255, 255, 255, // 32..39
  255, 255, 255, 255, 255, 255, 255, 255, // 40..47
  255, 255, 255, 255, 255, 255, 255, 255, // 48..55
  255, 255, 255, 255, 255, 255, 255, 255, // 56..63
  255, 3,   255, 2,   255, 255, 255, 1,   // 64..71
  255, 255, 255, 255, 255, 255, 255, 255, // 72..79
  255, 255, 255, 255, 0,   0,   255, 255, // 80..87
  255, 255, 255, 255, 255, 255, 255, 255, // 88..95
  255, 3,   255, 2,   255, 255, 255, 1,   // 96..103
  255, 255, 255, 255, 255, 255, 255, 255, // 104..111
  255, 255, 255, 255, 0,   0,   255, 255, // 112..119
  255, 255, 255, 255, 255, 255, 255, 255, // 120..127
  255, 255, 255, 255, 255, 255, 255, 255, // 128..135
  255, 255, 255, 255, 255, 255, 255, 255, // 136..143
  255, 255, 255, 255, 255, 255, 255, 255, // 144..151
  255, 255, 255, 255, 255, 255, 255, 255, // 152..159
  255, 255, 255, 255, 255, 255, 255, 255, // 160..167
  255, 255, 255, 255, 255, 255, 255, 255, // 168..175
  255, 255, 255, 255, 255, 255, 255, 255, // 176..183
  255, 255, 255, 255, 255, 255, 255, 255, // 184..191
  255, 255, 255, 255, 255, 255, 255, 255, // 192..199
  255, 255, 255, 255, 255, 255, 255, 255, // 200..207
  255, 255, 255, 255, 255, 255, 255, 255, // 208..215
  255, 255, 255, 255, 255, 255, 255, 255, // 216..223
  255, 255, 255, 255, 255, 255, 255, 255, // 224..231
  255, 255, 255, 255, 255, 255, 255, 255, // 232..239
  255, 255, 255, 255, 255, 255, 255, 255, // 240..247
  255, 255, 255, 255, 255, 255, 255, 255  // 248..255
};

const uint64_t DIMER_TAB[4 * 4] = {
  5015898201438948509U, 5225361804584821669U, 6423762225589857229U,
  5783394398799547583U, 6894017875502584557U, 5959461383092338133U,
  4833978511655400893U, 5364573296520205007U, 9002561594443973180U,
  8212239310050454788U, 6941810030513055084U, 7579897184553533982U,
  7935738758488558809U, 7149836515649299425U, 8257540373175577481U,
  8935100007508790523U
};

const uint64_t TRIMER_TAB[4 * 4 * 4] = {
  13237172352163388750U, 13451082378889146998U, 12324706752351386142U,
  11704099346423635308U, 12503002411303846718U, 11573033083854154758U,
  12770611021816489070U, 13284814289517544220U, 10286336837755622383U,
  9500434588327378135U,  10554658215321236671U, 11177611689138066381U,
  11245073286936829194U, 10454751004568891954U, 9274956656780491354U,
  9930495270120774952U,  9498947889754972591U,  10289371588586147479U,
  11487222103436658431U, 10812501148518244749U, 11088845979783725023U,
  10735249574334615783U, 9609199230360475791U,  10105458452942995453U,
  13447889238169808654U, 13238535845420384310U, 11968673763542288478U,
  12645600078955589420U, 12136759312206930411U, 11922809957208297171U,
  13031072242070652603U, 13668666814620918217U, 14219262150204358668U,
  14433136993975185204U, 15703263506252408668U, 15026899868095529006U,
  16097136083696541308U, 15167201938128040260U, 14113514427211577644U,
  14608043031429815902U, 18169629015343943341U, 17383691583363408277U,
  16185576633819064829U, 16859734366019948175U, 17215452794964541512U,
  16425095330967072624U, 17460550829194815256U, 18101973914136232042U,
  16197524846324948423U, 17136496960994620159U, 18190301010467109527U,
  17660752969549176293U, 18084590689685816247U, 17861669045228104847U,
  16591430392433501415U, 17233003275094786965U, 15689030113991676774U,
  15321980360070757470U, 14196301091602199606U, 14727918144983470916U,
  14660430141886012803U, 14297932370981794491U, 15550237822687034067U,
  16044915679164358049U
};

const uint64_t TETRAMER_TAB[4 * 4 * 4 * 4] = {
  6047278271377325800U,  6842100033257738704U,  5716751207778949560U,
  5058261232784932554U,  5322212292231585944U,  4955210659836481440U,
  6153481158060361672U,  6630136099103187130U,  7683058811908681801U,
  7460089081761259377U,  8513615477720831769U,  9169618076073996395U,
  8669810821731892908U,  8451393064794886548U,  7271235746105367036U,
  7894785163577458318U,  7461575445318369801U,  7680024275870068017U,
  8878022265940976985U,  8237757801848291883U,  9060296013225843833U,
  8116780716040188737U,  6991106539262573353U,  7521593563379047515U,
  6845292839028968616U,  6045914992845185936U,  4775672622745250808U,
  5413871935584767114U,  5490367161684853325U,  4695435745326017909U,
  5803018666222232861U,  6480400171096490607U,  2381043025085637546U,
  3175899973157948562U,  4445879008075678970U,  3807116472585741192U,
  4268108881087626714U,  3901072061426881250U,  2847008385469766282U,
  3379366782720458232U,  1763336001516006667U,  1540401457157816883U,
  342666797974407771U,   983493939256405289U,   771890739233563630U,
  553508169276984534U,   1589643033626739902U,  2263336780810576844U,
  330722743541775969U,   688712796851212633U,   1742668713148160305U,
  1245320973785726531U,  2208596672445898769U,  1422777727841816361U,
  152919646732699457U,   826464124477841459U,   4460107693596700864U,
  3530055095011467256U,  2403999925630162832U,  2899137386794791138U,
  3398970977768160805U,  2464498338584432925U,  3716128830812494197U,
  4248337413163712007U,  4264326372183459627U,  3906261395711551507U,
  2851952150714671227U,  3383149429014333193U,  2386233046276708699U,
  3172117876357805667U,  4441779805226941963U,  3801926588820052345U,
  170684860043692426U,   1100671402695403186U,  2226926226858061530U,
  1693589575942097320U,  1193606390847620975U,  2128144916583147607U,
  876319371625685055U,   382305650241144653U,   1102545060664966090U,
  168107437338776818U,   1437989166537956506U,  1915072878734195688U,
  1548519783094789562U,  1757891215679916674U,  703889661060612842U,
  46092416782165400U,    3908715595921208683U,  4262294307145226835U,
  3064498623987880507U,  2585134797421409609U,  2661735585529691022U,
  3019760716990469302U,  4055956603131813086U,  3543998858204232620U,
  5317339067591416425U,  4959238909506745681U,  6157334207435046201U,
  6635009461133220427U,  6051307208490845209U,  6837227221258447649U,
  5711490920986878793U,  5054232433096901691U,  8122648135453742280U,
  9052599496358476784U,  7782418148093113240U,  7307023562816214250U,
  7095314801322056237U,  8029818144085865749U,  9137340041034366333U,
  8622472983995947535U,  7806751516869674914U,  7011855109925922970U,
  8137690373747335410U,  8757695200062998400U,  8531879593853721042U,
  8898947385530005226U,  7700757522090507906U,  7186022138009770480U,
  6135219772853324035U,  6358123720871388731U,  5304510851123850835U,
  4682089562405882145U,  5182028715320330214U,  5400512630465816798U,
  6580751683450298550U,  5923625422568720324U,  13124074928584983660U,
  13491146941631638356U, 12293650504952193852U, 11816502978180760654U,
  12399079312662682140U, 11604187204414436644U, 12730450818222161228U,
  13388307479092468286U, 10327209524901530317U, 9388215691182564853U,
  10657868830410829213U, 11137168911054473967U, 11357920004770333736U,
  10414374197647485712U, 9306325182584103800U,  9818342344138146826U,
  9386341947321596045U,  10329786896059045813U, 11455812913355464669U,
  10924692575052363951U, 10984992149858150141U, 10766613702172592581U,
  9568826821541020077U,  10208598699842184927U, 13488692655530571308U,
  13126106942075820308U, 12072096584926548348U, 12605510244625659406U,
  12249677498819492041U, 11882645355480553457U, 13062230760632229785U,
  13556163143878539499U, 14178740190036597038U, 14545847390080448022U,
  15599559227675164286U, 15067834145139579148U, 16065876409530435422U,
  15270949115358734438U, 14000758968863088654U, 14640014089599289212U,
  18281953465151117199U, 17342994818563569847U, 16217267316526477535U,
  16746698532205467565U, 17255653680509032810U, 16312143059561297490U,
  17564497017566543418U, 18061360711745100104U, 16237972021990524133U,
  17023861349393640413U, 18293930539975648181U, 17619893477009409223U,
  18115916316835994261U, 17757855915011241389U, 16704251839199542725U,
  17200966263939144375U, 15576639675766950468U, 15362743113290245500U,
  14164544455910714644U, 14841019967217601126U, 14620295210399335585U,
  14410818688327658393U, 15446357621659116529U, 16085462927495578755U,
  18237799192036655099U, 17294270664133710019U, 16258109964509321387U,
  16773410497518403545U, 16657084189963477387U, 16875519862962278067U,
  18127020052323321563U, 17507580374969491881U, 14153168177888129370U,
  14515696771658964578U, 15624080140268688906U, 15110866744451150200U,
  15466708232756051903U, 15833797605570023559U, 14563810316809509103U,
  14085706539145691037U, 14517711175708869402U, 14150731501263563810U,
  15402451490950456394U, 15899948742203982648U, 15224753927964908906U,
  16019597712369578578U, 14983744703118572090U, 14310050713553640776U,
  17296865610423782843U, 18235907873078829699U, 17055988043521714923U,
  16561000163437350297U, 16340222631939670878U, 17283720110790814822U,
  18338064546595415054U, 17805706452459078524U, 10375933128878629561U,
  9432369415202180481U,  10612588863825479145U, 11105888166746317467U,
  10794790039591648457U, 11013260899437695985U, 9905396050428550041U,
  9228014311730625771U,  13154226096333843480U, 13516719503928509216U,
  12264699899470662472U, 11768891770841246778U, 11836546934201131773U,
  12203601119882644933U, 13328994472388527533U, 12798507759874630367U,
  12277767672444305266U, 12068343612890878026U, 13176021535246260258U,
  13816435502572994384U, 12705517425460601090U, 13640043170446921274U,
  12460006250421962322U, 11929369723008524576U, 10597232027372843475U,
  11387585128312430315U, 10351852510211364483U, 9713802769929286129U,
  9357917249443839798U,  10143859113470169102U, 11342251114164164710U,
  10664720106027613972U
};

#define AA_ROLL_TABLE(CHAR, LEVEL, ROT)                                        \
  (LEVEL_X_AA_SEED_LEFT_31BITS_ROLL_TABLE[LEVEL][CHAR]                         \
                                         [(ROT) < 31 ? (ROT) : (ROT) % 31] |   \
   LEVEL_X_AA_SEED_RIGHT_33BITS_ROLL_TABLE[LEVEL][CHAR]                        \
                                          [(ROT) < 33 ? (ROT) : (ROT) % 33])

// 64-bit random seeds corresponding to 20 amino acids
constexpr inline uint64_t AA_SEED_A = 0xf56d6192468323df;
constexpr inline uint64_t AA_SEED_C = 0x9b0b2fd724e1e1d2;
constexpr inline uint64_t AA_SEED_D = 0xe8c583296b03c7af;
constexpr inline uint64_t AA_SEED_E = 0x6d8186850ee2f67;
constexpr inline uint64_t AA_SEED_F = 0x921e1da156b717ad;
constexpr inline uint64_t AA_SEED_G = 0xa70dc450015e3ffe;
constexpr inline uint64_t AA_SEED_H = 0x2242263a9d5638ff;
constexpr inline uint64_t AA_SEED_I = 0x2469ca06d519cdef;
constexpr inline uint64_t AA_SEED_K = 0xd4e7f06ac0593d3b;
constexpr inline uint64_t AA_SEED_L = 0xa5e19c0b1b40a97f;
constexpr inline uint64_t AA_SEED_M = 0xfab3d6d4dd74c000;
constexpr inline uint64_t AA_SEED_N = 0x4b363f2cf7bc5200;
constexpr inline uint64_t AA_SEED_P = 0x21ac8af2adb65ce4;
constexpr inline uint64_t AA_SEED_Q = 0x1d3baae9ab7cd800;
constexpr inline uint64_t AA_SEED_R = 0x49015253a9dbedf;
constexpr inline uint64_t AA_SEED_S = 0x5bf1f1d7ae699000;
constexpr inline uint64_t AA_SEED_T = 0xdb0c63dd7282cf90;
constexpr inline uint64_t AA_SEED_V = 0x7df64ddf78874000;
constexpr inline uint64_t AA_SEED_W = 0xee9e700cae6aa279;
constexpr inline uint64_t AA_SEED_Y = 0x5852ffb781a97610;
constexpr inline uint64_t AA_SEED__ = 0x0000000000000000; // NOLINT

constexpr inline uint64_t AA_SEED_TABLE[ASCII_SIZE] = {
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 0..7
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 8..15
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 16..23
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 24..31
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 32..39
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 40..47
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 48..55
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 56..63
  AA_SEED__, AA_SEED_A, AA_SEED__, AA_SEED_C,
  AA_SEED_D, AA_SEED_E, AA_SEED_F, AA_SEED_G, // 64..71
  AA_SEED_H, AA_SEED_I, AA_SEED__, AA_SEED_K,
  AA_SEED_L, AA_SEED_M, AA_SEED_N, AA_SEED__, // 72..79
  AA_SEED_P, AA_SEED_Q, AA_SEED_R, AA_SEED_S,
  AA_SEED_T, AA_SEED__, AA_SEED_V, AA_SEED_W, // 80..87
  AA_SEED__, AA_SEED_Y, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 88..95
  AA_SEED__, AA_SEED_A, AA_SEED__, AA_SEED_C,
  AA_SEED_D, AA_SEED_E, AA_SEED_F, AA_SEED_G, // 96..103
  AA_SEED_H, AA_SEED_I, AA_SEED__, AA_SEED_K,
  AA_SEED_L, AA_SEED_M, AA_SEED_N, AA_SEED__, // 104..111
  AA_SEED_P, AA_SEED_Q, AA_SEED_R, AA_SEED_S,
  AA_SEED_T, AA_SEED__, AA_SEED_V, AA_SEED_W, // 112..119
  AA_SEED__, AA_SEED_Y, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 120..127
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 128..135
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 136..143
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 144..151
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 152..159
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 160..167
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 168..175
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 176..183
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 184..191
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 192..199
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 200..207
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 208..215
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 216..223
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 224..231
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 232..239
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, // 240..247
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__ // 248..255
};

constexpr inline uint64_t AA_SEED_A_RIGHT_33BITS_ROLL_TABLE[33] = {
  0x468323df,  0x8d0647be,  0x11a0c8f7c, 0x34191ef9,  0x68323df2,  0xd0647be4,
  0x1a0c8f7c8, 0x14191ef91, 0x8323df23,  0x10647be46, 0xc8f7c8d,   0x191ef91a,
  0x323df234,  0x647be468,  0xc8f7c8d0,  0x191ef91a0, 0x123df2341, 0x47be4683,
  0x8f7c8d06,  0x11ef91a0c, 0x3df23419,  0x7be46832,  0xf7c8d064,  0x1ef91a0c8,
  0x1df234191, 0x1be468323, 0x17c8d0647, 0xf91a0c8f,  0x1f234191e, 0x1e468323d,
  0x1c8d0647b, 0x191a0c8f7, 0x1234191ef
};
constexpr inline uint64_t AA_SEED_A_LEFT_31BITS_ROLL_TABLE[31] = {
  0xf56d619200000000, 0xeadac32600000000, 0xd5b5864e00000000,
  0xab6b0c9e00000000, 0x56d6193e00000000, 0xadac327c00000000,
  0x5b5864fa00000000, 0xb6b0c9f400000000, 0x6d6193ea00000000,
  0xdac327d400000000, 0xb5864faa00000000, 0x6b0c9f5600000000,
  0xd6193eac00000000, 0xac327d5a00000000, 0x5864fab600000000,
  0xb0c9f56c00000000, 0x6193eada00000000, 0xc327d5b400000000,
  0x864fab6a00000000, 0xc9f56d600000000,  0x193eadac00000000,
  0x327d5b5800000000, 0x64fab6b000000000, 0xc9f56d6000000000,
  0x93eadac200000000, 0x27d5b58600000000, 0x4fab6b0c00000000,
  0x9f56d61800000000, 0x3eadac3200000000, 0x7d5b586400000000,
  0xfab6b0c800000000
};
constexpr inline uint64_t AA_SEED_C_RIGHT_33BITS_ROLL_TABLE[33] = {
  0x124e1e1d2, 0x49c3c3a5,  0x9387874a,  0x1270f0e94, 0x4e1e1d29,  0x9c3c3a52,
  0x1387874a4, 0x70f0e949,  0xe1e1d292,  0x1c3c3a524, 0x187874a49, 0x10f0e9493,
  0x1e1d2927,  0x3c3a524e,  0x7874a49c,  0xf0e94938,  0x1e1d29270, 0x1c3a524e1,
  0x1874a49c3, 0x10e949387, 0x1d29270f,  0x3a524e1e,  0x74a49c3c,  0xe9493878,
  0x1d29270f0, 0x1a524e1e1, 0x14a49c3c3, 0x94938787,  0x129270f0e, 0x524e1e1d,
  0xa49c3c3a,  0x149387874, 0x9270f0e9
};
constexpr inline uint64_t AA_SEED_C_LEFT_31BITS_ROLL_TABLE[31] = {
  0x9b0b2fd600000000, 0x36165fae00000000, 0x6c2cbf5c00000000,
  0xd8597eb800000000, 0xb0b2fd7200000000, 0x6165fae600000000,
  0xc2cbf5cc00000000, 0x8597eb9a00000000, 0xb2fd73600000000,
  0x165fae6c00000000, 0x2cbf5cd800000000, 0x597eb9b000000000,
  0xb2fd736000000000, 0x65fae6c200000000, 0xcbf5cd8400000000,
  0x97eb9b0a00000000, 0x2fd7361600000000, 0x5fae6c2c00000000,
  0xbf5cd85800000000, 0x7eb9b0b200000000, 0xfd73616400000000,
  0xfae6c2ca00000000, 0xf5cd859600000000, 0xeb9b0b2e00000000,
  0xd736165e00000000, 0xae6c2cbe00000000, 0x5cd8597e00000000,
  0xb9b0b2fc00000000, 0x736165fa00000000, 0xe6c2cbf400000000,
  0xcd8597ea00000000
};
constexpr inline uint64_t AA_SEED_D_RIGHT_33BITS_ROLL_TABLE[33] = {
  0x16b03c7af, 0xd6078f5f,  0x1ac0f1ebe, 0x1581e3d7d, 0xb03c7afb,  0x16078f5f6,
  0xc0f1ebed,  0x181e3d7da, 0x103c7afb5, 0x78f5f6b,   0xf1ebed6,   0x1e3d7dac,
  0x3c7afb58,  0x78f5f6b0,  0xf1ebed60,  0x1e3d7dac0, 0x1c7afb581, 0x18f5f6b03,
  0x11ebed607, 0x3d7dac0f,  0x7afb581e,  0xf5f6b03c,  0x1ebed6078, 0x1d7dac0f1,
  0x1afb581e3, 0x15f6b03c7, 0xbed6078f,  0x17dac0f1e, 0xfb581e3d,  0x1f6b03c7a,
  0x1ed6078f5, 0x1dac0f1eb, 0x1b581e3d7
};
constexpr inline uint64_t AA_SEED_D_LEFT_31BITS_ROLL_TABLE[31] = {
  0xe8c5832800000000, 0xd18b065200000000, 0xa3160ca600000000,
  0x462c194e00000000, 0x8c58329c00000000, 0x18b0653a00000000,
  0x3160ca7400000000, 0x62c194e800000000, 0xc58329d000000000,
  0x8b0653a200000000, 0x160ca74600000000, 0x2c194e8c00000000,
  0x58329d1800000000, 0xb0653a3000000000, 0x60ca746200000000,
  0xc194e8c400000000, 0x8329d18a00000000, 0x653a31600000000,
  0xca7462c00000000,  0x194e8c5800000000, 0x329d18b000000000,
  0x653a316000000000, 0xca7462c000000000, 0x94e8c58200000000,
  0x29d18b0600000000, 0x53a3160c00000000, 0xa7462c1800000000,
  0x4e8c583200000000, 0x9d18b06400000000, 0x3a3160ca00000000,
  0x7462c19400000000
};
constexpr inline uint64_t AA_SEED_E_RIGHT_33BITS_ROLL_TABLE[33] = {
  0x50ee2f67,  0xa1dc5ece,  0x143b8bd9c, 0x87717b39,  0x10ee2f672, 0x1dc5ece5,
  0x3b8bd9ca,  0x7717b394,  0xee2f6728,  0x1dc5ece50, 0x1b8bd9ca1, 0x1717b3943,
  0xe2f67287,  0x1c5ece50e, 0x18bd9ca1d, 0x117b3943b, 0x2f672877,  0x5ece50ee,
  0xbd9ca1dc,  0x17b3943b8, 0xf6728771,  0x1ece50ee2, 0x1d9ca1dc5, 0x1b3943b8b,
  0x167287717, 0xce50ee2f,  0x19ca1dc5e, 0x13943b8bd, 0x7287717b,  0xe50ee2f6,
  0x1ca1dc5ec, 0x1943b8bd9, 0x1287717b3
};
constexpr inline uint64_t AA_SEED_E_LEFT_31BITS_ROLL_TABLE[31] = {
  0x6d8186800000000,  0xdb030d000000000,  0x1b6061a000000000,
  0x36c0c34000000000, 0x6d81868000000000, 0xdb030d0000000000,
  0xb6061a0200000000, 0x6c0c340600000000, 0xd818680c00000000,
  0xb030d01a00000000, 0x6061a03600000000, 0xc0c3406c00000000,
  0x818680da00000000, 0x30d01b600000000,  0x61a036c00000000,
  0xc3406d800000000,  0x18680db000000000, 0x30d01b6000000000,
  0x61a036c000000000, 0xc3406d8000000000, 0x8680db0200000000,
  0xd01b60600000000,  0x1a036c0c00000000, 0x3406d81800000000,
  0x680db03000000000, 0xd01b606000000000, 0xa036c0c200000000,
  0x406d818600000000, 0x80db030c00000000, 0x1b6061a00000000,
  0x36c0c3400000000
};
constexpr inline uint64_t AA_SEED_F_RIGHT_33BITS_ROLL_TABLE[33] = {
  0x156b717ad, 0xad6e2f5b,  0x15adc5eb6, 0xb5b8bd6d,  0x16b717ada, 0xd6e2f5b5,
  0x1adc5eb6a, 0x15b8bd6d5, 0xb717adab,  0x16e2f5b56, 0xdc5eb6ad,  0x1b8bd6d5a,
  0x1717adab5, 0xe2f5b56b,  0x1c5eb6ad6, 0x18bd6d5ad, 0x117adab5b, 0x2f5b56b7,
  0x5eb6ad6e,  0xbd6d5adc,  0x17adab5b8, 0xf5b56b71,  0x1eb6ad6e2, 0x1d6d5adc5,
  0x1adab5b8b, 0x15b56b717, 0xb6ad6e2f,  0x16d5adc5e, 0xdab5b8bd,  0x1b56b717a,
  0x16ad6e2f5, 0xd5adc5eb,  0x1ab5b8bd6
};
constexpr inline uint64_t AA_SEED_F_LEFT_31BITS_ROLL_TABLE[31] = {
  0x921e1da000000000, 0x243c3b4200000000, 0x4878768400000000,
  0x90f0ed0800000000, 0x21e1da1200000000, 0x43c3b42400000000,
  0x8787684800000000, 0xf0ed09200000000,  0x1e1da12400000000,
  0x3c3b424800000000, 0x7876849000000000, 0xf0ed092000000000,
  0xe1da124200000000, 0xc3b4248600000000, 0x8768490e00000000,
  0xed0921e00000000,  0x1da1243c00000000, 0x3b42487800000000,
  0x768490f000000000, 0xed0921e000000000, 0xda1243c200000000,
  0xb424878600000000, 0x68490f0e00000000, 0xd0921e1c00000000,
  0xa1243c3a00000000, 0x4248787600000000, 0x8490f0ec00000000,
  0x921e1da00000000,  0x1243c3b400000000, 0x2487876800000000,
  0x490f0ed000000000
};
constexpr inline uint64_t AA_SEED_G_RIGHT_33BITS_ROLL_TABLE[33] = {
  0x15e3ffe,   0x2bc7ffc,   0x578fff8,   0xaf1fff0,   0x15e3ffe0,  0x2bc7ffc0,
  0x578fff80,  0xaf1fff00,  0x15e3ffe00, 0xbc7ffc01,  0x178fff802, 0xf1fff005,
  0x1e3ffe00a, 0x1c7ffc015, 0x18fff802b, 0x11fff0057, 0x3ffe00af,  0x7ffc015e,
  0xfff802bc,  0x1fff00578, 0x1ffe00af1, 0x1ffc015e3, 0x1ff802bc7, 0x1ff00578f,
  0x1fe00af1f, 0x1fc015e3f, 0x1f802bc7f, 0x1f00578ff, 0x1e00af1ff, 0x1c015e3ff,
  0x1802bc7ff, 0x100578fff, 0xaf1fff
};
constexpr inline uint64_t AA_SEED_G_LEFT_31BITS_ROLL_TABLE[31] = {
  0xa70dc45000000000, 0x4e1b88a200000000, 0x9c37114400000000,
  0x386e228a00000000, 0x70dc451400000000, 0xe1b88a2800000000,
  0xc371145200000000, 0x86e228a600000000, 0xdc4514e00000000,
  0x1b88a29c00000000, 0x3711453800000000, 0x6e228a7000000000,
  0xdc4514e000000000, 0xb88a29c200000000, 0x7114538600000000,
  0xe228a70c00000000, 0xc4514e1a00000000, 0x88a29c3600000000,
  0x1145386e00000000, 0x228a70dc00000000, 0x4514e1b800000000,
  0x8a29c37000000000, 0x145386e200000000, 0x28a70dc400000000,
  0x514e1b8800000000, 0xa29c371000000000, 0x45386e2200000000,
  0x8a70dc4400000000, 0x14e1b88a00000000, 0x29c3711400000000,
  0x5386e22800000000
};
constexpr inline uint64_t AA_SEED_H_RIGHT_33BITS_ROLL_TABLE[33] = {
  0x9d5638ff,  0x13aac71fe, 0x7558e3fd,  0xeab1c7fa,  0x1d5638ff4, 0x1aac71fe9,
  0x1558e3fd3, 0xab1c7fa7,  0x15638ff4e, 0xac71fe9d,  0x158e3fd3a, 0xb1c7fa75,
  0x1638ff4ea, 0xc71fe9d5,  0x18e3fd3aa, 0x11c7fa755, 0x38ff4eab,  0x71fe9d56,
  0xe3fd3aac,  0x1c7fa7558, 0x18ff4eab1, 0x11fe9d563, 0x3fd3aac7,  0x7fa7558e,
  0xff4eab1c,  0x1fe9d5638, 0x1fd3aac71, 0x1fa7558e3, 0x1f4eab1c7, 0x1e9d5638f,
  0x1d3aac71f, 0x1a7558e3f, 0x14eab1c7f
};
constexpr inline uint64_t AA_SEED_H_LEFT_31BITS_ROLL_TABLE[31] = {
  0x2242263a00000000, 0x44844c7400000000, 0x890898e800000000,
  0x121131d200000000, 0x242263a400000000, 0x4844c74800000000,
  0x90898e9000000000, 0x21131d2200000000, 0x42263a4400000000,
  0x844c748800000000, 0x898e91200000000,  0x1131d22400000000,
  0x2263a44800000000, 0x44c7489000000000, 0x898e912000000000,
  0x131d224200000000, 0x263a448400000000, 0x4c74890800000000,
  0x98e9121000000000, 0x31d2242200000000, 0x63a4484400000000,
  0xc748908800000000, 0x8e91211200000000, 0x1d22422600000000,
  0x3a44844c00000000, 0x7489089800000000, 0xe912113000000000,
  0xd224226200000000, 0xa44844c600000000, 0x4890898e00000000,
  0x9121131c00000000
};
constexpr inline uint64_t AA_SEED_I_RIGHT_33BITS_ROLL_TABLE[33] = {
  0xd519cdef,  0x1aa339bde, 0x1546737bd, 0xa8ce6f7b,  0x1519cdef6, 0xa339bded,
  0x146737bda, 0x8ce6f7b5,  0x119cdef6a, 0x339bded5,  0x6737bdaa,  0xce6f7b54,
  0x19cdef6a8, 0x139bded51, 0x737bdaa3,  0xe6f7b546,  0x1cdef6a8c, 0x19bded519,
  0x137bdaa33, 0x6f7b5467,  0xdef6a8ce,  0x1bded519c, 0x17bdaa339, 0xf7b54673,
  0x1ef6a8ce6, 0x1ded519cd, 0x1bdaa339b, 0x17b546737, 0xf6a8ce6f,  0x1ed519cde,
  0x1daa339bd, 0x1b546737b, 0x16a8ce6f7
};
constexpr inline uint64_t AA_SEED_I_LEFT_31BITS_ROLL_TABLE[31] = {
  0x2469ca0600000000, 0x48d3940c00000000, 0x91a7281800000000,
  0x234e503200000000, 0x469ca06400000000, 0x8d3940c800000000,
  0x1a72819200000000, 0x34e5032400000000, 0x69ca064800000000,
  0xd3940c9000000000, 0xa728192200000000, 0x4e50324600000000,
  0x9ca0648c00000000, 0x3940c91a00000000, 0x7281923400000000,
  0xe503246800000000, 0xca0648d200000000, 0x940c91a600000000,
  0x2819234e00000000, 0x5032469c00000000, 0xa0648d3800000000,
  0x40c91a7200000000, 0x819234e400000000, 0x32469ca00000000,
  0x648d39400000000,  0xc91a72800000000,  0x19234e5000000000,
  0x32469ca000000000, 0x648d394000000000, 0xc91a728000000000,
  0x9234e50200000000
};
constexpr inline uint64_t AA_SEED_K_RIGHT_33BITS_ROLL_TABLE[33] = {
  0xc0593d3b,  0x180b27a76, 0x10164f4ed, 0x2c9e9db,   0x593d3b6,   0xb27a76c,
  0x164f4ed8,  0x2c9e9db0,  0x593d3b60,  0xb27a76c0,  0x164f4ed80, 0xc9e9db01,
  0x193d3b602, 0x127a76c05, 0x4f4ed80b,  0x9e9db016,  0x13d3b602c, 0x7a76c059,
  0xf4ed80b2,  0x1e9db0164, 0x1d3b602c9, 0x1a76c0593, 0x14ed80b27, 0x9db0164f,
  0x13b602c9e, 0x76c0593d,  0xed80b27a,  0x1db0164f4, 0x1b602c9e9, 0x16c0593d3,
  0xd80b27a7,  0x1b0164f4e, 0x1602c9e9d
};
constexpr inline uint64_t AA_SEED_K_LEFT_31BITS_ROLL_TABLE[31] = {
  0xd4e7f06a00000000, 0xa9cfe0d600000000, 0x539fc1ae00000000,
  0xa73f835c00000000, 0x4e7f06ba00000000, 0x9cfe0d7400000000,
  0x39fc1aea00000000, 0x73f835d400000000, 0xe7f06ba800000000,
  0xcfe0d75200000000, 0x9fc1aea600000000, 0x3f835d4e00000000,
  0x7f06ba9c00000000, 0xfe0d753800000000, 0xfc1aea7200000000,
  0xf835d4e600000000, 0xf06ba9ce00000000, 0xe0d7539e00000000,
  0xc1aea73e00000000, 0x835d4e7e00000000, 0x6ba9cfe00000000,
  0xd7539fc00000000,  0x1aea73f800000000, 0x35d4e7f000000000,
  0x6ba9cfe000000000, 0xd7539fc000000000, 0xaea73f8200000000,
  0x5d4e7f0600000000, 0xba9cfe0c00000000, 0x7539fc1a00000000,
  0xea73f83400000000
};
constexpr inline uint64_t AA_SEED_L_RIGHT_33BITS_ROLL_TABLE[33] = {
  0x11b40a97f, 0x368152ff,  0x6d02a5fe,  0xda054bfc,  0x1b40a97f8, 0x168152ff1,
  0xd02a5fe3,  0x1a054bfc6, 0x140a97f8d, 0x8152ff1b,  0x102a5fe36, 0x54bfc6d,
  0xa97f8da,   0x152ff1b4,  0x2a5fe368,  0x54bfc6d0,  0xa97f8da0,  0x152ff1b40,
  0xa5fe3681,  0x14bfc6d02, 0x97f8da05,  0x12ff1b40a, 0x5fe36815,  0xbfc6d02a,
  0x17f8da054, 0xff1b40a9,  0x1fe368152, 0x1fc6d02a5, 0x1f8da054b, 0x1f1b40a97,
  0x1e368152f, 0x1c6d02a5f, 0x18da054bf
};
constexpr inline uint64_t AA_SEED_L_LEFT_31BITS_ROLL_TABLE[31] = {
  0xa5e19c0a00000000, 0x4bc3381600000000, 0x9786702c00000000,
  0x2f0ce05a00000000, 0x5e19c0b400000000, 0xbc33816800000000,
  0x786702d200000000, 0xf0ce05a400000000, 0xe19c0b4a00000000,
  0xc338169600000000, 0x86702d2e00000000, 0xce05a5e00000000,
  0x19c0b4bc00000000, 0x3381697800000000, 0x6702d2f000000000,
  0xce05a5e000000000, 0x9c0b4bc200000000, 0x3816978600000000,
  0x702d2f0c00000000, 0xe05a5e1800000000, 0xc0b4bc3200000000,
  0x8169786600000000, 0x2d2f0ce00000000,  0x5a5e19c00000000,
  0xb4bc33800000000,  0x1697867000000000, 0x2d2f0ce000000000,
  0x5a5e19c000000000, 0xb4bc338000000000, 0x6978670200000000,
  0xd2f0ce0400000000
};
constexpr inline uint64_t AA_SEED_M_RIGHT_33BITS_ROLL_TABLE[33] = {
  0xdd74c000,  0x1bae98000, 0x175d30001, 0xeba60003, 0x1d74c0006, 0x1ae98000d,
  0x15d30001b, 0xba600037,  0x174c0006e, 0xe98000dd, 0x1d30001ba, 0x1a6000375,
  0x14c0006eb, 0x98000dd7,  0x130001bae, 0x6000375d, 0xc0006eba,  0x18000dd74,
  0x10001bae9, 0x375d3,     0x6eba6,     0xdd74c,    0x1bae98,    0x375d30,
  0x6eba60,    0xdd74c0,    0x1bae980,   0x375d300,  0x6eba600,   0xdd74c00,
  0x1bae9800,  0x375d3000,  0x6eba6000
};
constexpr inline uint64_t AA_SEED_M_LEFT_31BITS_ROLL_TABLE[31] = {
  0xfab3d6d400000000, 0xf567adaa00000000, 0xeacf5b5600000000,
  0xd59eb6ae00000000, 0xab3d6d5e00000000, 0x567adabe00000000,
  0xacf5b57c00000000, 0x59eb6afa00000000, 0xb3d6d5f400000000,
  0x67adabea00000000, 0xcf5b57d400000000, 0x9eb6afaa00000000,
  0x3d6d5f5600000000, 0x7adabeac00000000, 0xf5b57d5800000000,
  0xeb6afab200000000, 0xd6d5f56600000000, 0xadabeace00000000,
  0x5b57d59e00000000, 0xb6afab3c00000000, 0x6d5f567a00000000,
  0xdabeacf400000000, 0xb57d59ea00000000, 0x6afab3d600000000,
  0xd5f567ac00000000, 0xabeacf5a00000000, 0x57d59eb600000000,
  0xafab3d6c00000000, 0x5f567ada00000000, 0xbeacf5b400000000,
  0x7d59eb6a00000000
};
constexpr inline uint64_t AA_SEED_N_RIGHT_33BITS_ROLL_TABLE[33] = {
  0xf7bc5200,  0x1ef78a400, 0x1def14801, 0x1bde29003, 0x17bc52007, 0xf78a400f,
  0x1ef14801e, 0x1de29003d, 0x1bc52007b, 0x178a400f7, 0xf14801ef,  0x1e29003de,
  0x1c52007bd, 0x18a400f7b, 0x114801ef7, 0x29003def,  0x52007bde,  0xa400f7bc,
  0x14801ef78, 0x9003def1,  0x12007bde2, 0x400f7bc5,  0x801ef78a,  0x1003def14,
  0x7bde29,    0xf7bc52,    0x1ef78a4,   0x3def148,   0x7bde290,   0xf7bc520,
  0x1ef78a40,  0x3def1480,  0x7bde2900
};
constexpr inline uint64_t AA_SEED_N_LEFT_31BITS_ROLL_TABLE[31] = {
  0x4b363f2c00000000, 0x966c7e5800000000, 0x2cd8fcb200000000,
  0x59b1f96400000000, 0xb363f2c800000000, 0x66c7e59200000000,
  0xcd8fcb2400000000, 0x9b1f964a00000000, 0x363f2c9600000000,
  0x6c7e592c00000000, 0xd8fcb25800000000, 0xb1f964b200000000,
  0x63f2c96600000000, 0xc7e592cc00000000, 0x8fcb259a00000000,
  0x1f964b3600000000, 0x3f2c966c00000000, 0x7e592cd800000000,
  0xfcb259b000000000, 0xf964b36200000000, 0xf2c966c600000000,
  0xe592cd8e00000000, 0xcb259b1e00000000, 0x964b363e00000000,
  0x2c966c7e00000000, 0x592cd8fc00000000, 0xb259b1f800000000,
  0x64b363f200000000, 0xc966c7e400000000, 0x92cd8fca00000000,
  0x259b1f9600000000
};
constexpr inline uint64_t AA_SEED_P_RIGHT_33BITS_ROLL_TABLE[33] = {
  0xadb65ce4,  0x15b6cb9c8, 0xb6d97391,  0x16db2e722, 0xdb65ce45,  0x1b6cb9c8a,
  0x16d973915, 0xdb2e722b,  0x1b65ce456, 0x16cb9c8ad, 0xd973915b,  0x1b2e722b6,
  0x165ce456d, 0xcb9c8adb,  0x1973915b6, 0x12e722b6d, 0x5ce456db,  0xb9c8adb6,
  0x173915b6c, 0xe722b6d9,  0x1ce456db2, 0x19c8adb65, 0x13915b6cb, 0x722b6d97,
  0xe456db2e,  0x1c8adb65c, 0x1915b6cb9, 0x122b6d973, 0x456db2e7,  0x8adb65ce,
  0x115b6cb9c, 0x2b6d9739,  0x56db2e72
};
constexpr inline uint64_t AA_SEED_P_LEFT_31BITS_ROLL_TABLE[31] = {
  0x21ac8af200000000, 0x435915e400000000, 0x86b22bc800000000,
  0xd64579200000000,  0x1ac8af2400000000, 0x35915e4800000000,
  0x6b22bc9000000000, 0xd645792000000000, 0xac8af24200000000,
  0x5915e48600000000, 0xb22bc90c00000000, 0x6457921a00000000,
  0xc8af243400000000, 0x915e486a00000000, 0x22bc90d600000000,
  0x457921ac00000000, 0x8af2435800000000, 0x15e486b200000000,
  0x2bc90d6400000000, 0x57921ac800000000, 0xaf24359000000000,
  0x5e486b2200000000, 0xbc90d64400000000, 0x7921ac8a00000000,
  0xf243591400000000, 0xe486b22a00000000, 0xc90d645600000000,
  0x921ac8ae00000000, 0x2435915e00000000, 0x486b22bc00000000,
  0x90d6457800000000
};
constexpr inline uint64_t AA_SEED_Q_RIGHT_33BITS_ROLL_TABLE[33] = {
  0x1ab7cd800, 0x156f9b001, 0xadf36003,  0x15be6c006, 0xb7cd800d,  0x16f9b001a,
  0xdf360035,  0x1be6c006a, 0x17cd800d5, 0xf9b001ab,  0x1f3600356, 0x1e6c006ad,
  0x1cd800d5b, 0x19b001ab7, 0x13600356f, 0x6c006adf,  0xd800d5be,  0x1b001ab7c,
  0x1600356f9, 0xc006adf3,  0x1800d5be6, 0x1001ab7cd, 0x356f9b,    0x6adf36,
  0xd5be6c,    0x1ab7cd8,   0x356f9b0,   0x6adf360,   0xd5be6c0,   0x1ab7cd80,
  0x356f9b00,  0x6adf3600,  0xd5be6c00
};
constexpr inline uint64_t AA_SEED_Q_LEFT_31BITS_ROLL_TABLE[31] = {
  0x1d3baae800000000, 0x3a7755d000000000, 0x74eeaba000000000,
  0xe9dd574000000000, 0xd3baae8200000000, 0xa7755d0600000000,
  0x4eeaba0e00000000, 0x9dd5741c00000000, 0x3baae83a00000000,
  0x7755d07400000000, 0xeeaba0e800000000, 0xdd5741d200000000,
  0xbaae83a600000000, 0x755d074e00000000, 0xeaba0e9c00000000,
  0xd5741d3a00000000, 0xaae83a7600000000, 0x55d074ee00000000,
  0xaba0e9dc00000000, 0x5741d3ba00000000, 0xae83a77400000000,
  0x5d074eea00000000, 0xba0e9dd400000000, 0x741d3baa00000000,
  0xe83a775400000000, 0xd074eeaa00000000, 0xa0e9dd5600000000,
  0x41d3baae00000000, 0x83a7755c00000000, 0x74eeaba00000000,
  0xe9dd57400000000
};
constexpr inline uint64_t AA_SEED_R_RIGHT_33BITS_ROLL_TABLE[33] = {
  0x13a9dbedf, 0x753b7dbf,  0xea76fb7e,  0x1d4edf6fc, 0x1a9dbedf9, 0x153b7dbf3,
  0xa76fb7e7,  0x14edf6fce, 0x9dbedf9d,  0x13b7dbf3a, 0x76fb7e75,  0xedf6fcea,
  0x1dbedf9d4, 0x1b7dbf3a9, 0x16fb7e753, 0xdf6fcea7,  0x1bedf9d4e, 0x17dbf3a9d,
  0xfb7e753b,  0x1f6fcea76, 0x1edf9d4ed, 0x1dbf3a9db, 0x1b7e753b7, 0x16fcea76f,
  0xdf9d4edf,  0x1bf3a9dbe, 0x17e753b7d, 0xfcea76fb,  0x1f9d4edf6, 0x1f3a9dbed,
  0x1e753b7db, 0x1cea76fb7, 0x19d4edf6f
};
constexpr inline uint64_t AA_SEED_R_LEFT_31BITS_ROLL_TABLE[31] = {
  0x490152400000000,  0x9202a4800000000,  0x1240549000000000,
  0x2480a92000000000, 0x4901524000000000, 0x9202a48000000000,
  0x2405490200000000, 0x480a920400000000, 0x9015240800000000,
  0x202a481200000000, 0x4054902400000000, 0x80a9204800000000,
  0x152409200000000,  0x2a4812400000000,  0x549024800000000,
  0xa92049000000000,  0x1524092000000000, 0x2a48124000000000,
  0x5490248000000000, 0xa920490000000000, 0x5240920200000000,
  0xa481240400000000, 0x4902480a00000000, 0x9204901400000000,
  0x2409202a00000000, 0x4812405400000000, 0x902480a800000000,
  0x2049015200000000, 0x409202a400000000, 0x8124054800000000,
  0x2480a9200000000
};
constexpr inline uint64_t AA_SEED_S_RIGHT_33BITS_ROLL_TABLE[33] = {
  0x1ae699000, 0x15cd32001, 0xb9a64003,  0x1734c8006, 0xe699000d,  0x1cd32001a,
  0x19a640035, 0x134c8006b, 0x699000d7,  0xd32001ae,  0x1a640035c, 0x14c8006b9,
  0x99000d73,  0x132001ae6, 0x640035cd,  0xc8006b9a,  0x19000d734, 0x12001ae69,
  0x40035cd3,  0x8006b9a6,  0x1000d734c, 0x1ae699,    0x35cd32,    0x6b9a64,
  0xd734c8,    0x1ae6990,   0x35cd320,   0x6b9a640,   0xd734c80,   0x1ae69900,
  0x35cd3200,  0x6b9a6400,  0xd734c800
};
constexpr inline uint64_t AA_SEED_S_LEFT_31BITS_ROLL_TABLE[31] = {
  0x5bf1f1d600000000, 0xb7e3e3ac00000000, 0x6fc7c75a00000000,
  0xdf8f8eb400000000, 0xbf1f1d6a00000000, 0x7e3e3ad600000000,
  0xfc7c75ac00000000, 0xf8f8eb5a00000000, 0xf1f1d6b600000000,
  0xe3e3ad6e00000000, 0xc7c75ade00000000, 0x8f8eb5be00000000,
  0x1f1d6b7e00000000, 0x3e3ad6fc00000000, 0x7c75adf800000000,
  0xf8eb5bf000000000, 0xf1d6b7e200000000, 0xe3ad6fc600000000,
  0xc75adf8e00000000, 0x8eb5bf1e00000000, 0x1d6b7e3e00000000,
  0x3ad6fc7c00000000, 0x75adf8f800000000, 0xeb5bf1f000000000,
  0xd6b7e3e200000000, 0xad6fc7c600000000, 0x5adf8f8e00000000,
  0xb5bf1f1c00000000, 0x6b7e3e3a00000000, 0xd6fc7c7400000000,
  0xadf8f8ea00000000
};
constexpr inline uint64_t AA_SEED_T_RIGHT_33BITS_ROLL_TABLE[33] = {
  0x17282cf90, 0xe5059f21,  0x1ca0b3e42, 0x194167c85, 0x1282cf90b, 0x5059f217,
  0xa0b3e42e,  0x14167c85c, 0x82cf90b9,  0x1059f2172, 0xb3e42e5,   0x167c85ca,
  0x2cf90b94,  0x59f21728,  0xb3e42e50,  0x167c85ca0, 0xcf90b941,  0x19f217282,
  0x13e42e505, 0x7c85ca0b,  0xf90b9416,  0x1f217282c, 0x1e42e5059, 0x1c85ca0b3,
  0x190b94167, 0x1217282cf, 0x42e5059f,  0x85ca0b3e,  0x10b94167c, 0x17282cf9,
  0x2e5059f2,  0x5ca0b3e4,  0xb94167c8
};
constexpr inline uint64_t AA_SEED_T_LEFT_31BITS_ROLL_TABLE[31] = {
  0xdb0c63dc00000000, 0xb618c7ba00000000, 0x6c318f7600000000,
  0xd8631eec00000000, 0xb0c63dda00000000, 0x618c7bb600000000,
  0xc318f76c00000000, 0x8631eeda00000000, 0xc63ddb600000000,
  0x18c7bb6c00000000, 0x318f76d800000000, 0x631eedb000000000,
  0xc63ddb6000000000, 0x8c7bb6c200000000, 0x18f76d8600000000,
  0x31eedb0c00000000, 0x63ddb61800000000, 0xc7bb6c3000000000,
  0x8f76d86200000000, 0x1eedb0c600000000, 0x3ddb618c00000000,
  0x7bb6c31800000000, 0xf76d863000000000, 0xeedb0c6200000000,
  0xddb618c600000000, 0xbb6c318e00000000, 0x76d8631e00000000,
  0xedb0c63c00000000, 0xdb618c7a00000000, 0xb6c318f600000000,
  0x6d8631ee00000000
};
constexpr inline uint64_t AA_SEED_V_RIGHT_33BITS_ROLL_TABLE[33] = {
  0x178874000, 0xf10e8001, 0x1e21d0002, 0x1c43a0005, 0x18874000b, 0x110e80017,
  0x21d0002f,  0x43a0005e, 0x874000bc,  0x10e800178, 0x1d0002f1,  0x3a0005e2,
  0x74000bc4,  0xe8001788, 0x1d0002f10, 0x1a0005e21, 0x14000bc43, 0x80017887,
  0x10002f10e, 0x5e21d,    0xbc43a,     0x178874,    0x2f10e8,    0x5e21d0,
  0xbc43a0,    0x1788740,  0x2f10e80,   0x5e21d00,   0xbc43a00,   0x17887400,
  0x2f10e800,  0x5e21d000, 0xbc43a000
};
constexpr inline uint64_t AA_SEED_V_LEFT_31BITS_ROLL_TABLE[31] = {
  0x7df64dde00000000, 0xfbec9bbc00000000, 0xf7d9377a00000000,
  0xefb26ef600000000, 0xdf64ddee00000000, 0xbec9bbde00000000,
  0x7d9377be00000000, 0xfb26ef7c00000000, 0xf64ddefa00000000,
  0xec9bbdf600000000, 0xd9377bee00000000, 0xb26ef7de00000000,
  0x64ddefbe00000000, 0xc9bbdf7c00000000, 0x9377befa00000000,
  0x26ef7df600000000, 0x4ddefbec00000000, 0x9bbdf7d800000000,
  0x377befb200000000, 0x6ef7df6400000000, 0xddefbec800000000,
  0xbbdf7d9200000000, 0x77befb2600000000, 0xef7df64c00000000,
  0xdefbec9a00000000, 0xbdf7d93600000000, 0x7befb26e00000000,
  0xf7df64dc00000000, 0xefbec9ba00000000, 0xdf7d937600000000,
  0xbefb26ee00000000
};
constexpr inline uint64_t AA_SEED_W_RIGHT_33BITS_ROLL_TABLE[33] = {
  0xae6aa279,  0x15cd544f2, 0xb9aa89e5,  0x1735513ca, 0xe6aa2795,  0x1cd544f2a,
  0x19aa89e55, 0x135513cab, 0x6aa27957,  0xd544f2ae,  0x1aa89e55c, 0x15513cab9,
  0xaa279573,  0x1544f2ae6, 0xa89e55cd,  0x1513cab9a, 0xa2795735,  0x144f2ae6a,
  0x89e55cd5,  0x113cab9aa, 0x27957355,  0x4f2ae6aa,  0x9e55cd54,  0x13cab9aa8,
  0x79573551,  0xf2ae6aa2,  0x1e55cd544, 0x1cab9aa89, 0x195735513, 0x12ae6aa27,
  0x55cd544f,  0xab9aa89e,  0x15735513c
};
constexpr inline uint64_t AA_SEED_W_LEFT_31BITS_ROLL_TABLE[31] = {
  0xee9e700c00000000, 0xdd3ce01a00000000, 0xba79c03600000000,
  0x74f3806e00000000, 0xe9e700dc00000000, 0xd3ce01ba00000000,
  0xa79c037600000000, 0x4f3806ee00000000, 0x9e700ddc00000000,
  0x3ce01bba00000000, 0x79c0377400000000, 0xf3806ee800000000,
  0xe700ddd200000000, 0xce01bba600000000, 0x9c03774e00000000,
  0x3806ee9e00000000, 0x700ddd3c00000000, 0xe01bba7800000000,
  0xc03774f200000000, 0x806ee9e600000000, 0xddd3ce00000000,
  0x1bba79c00000000,  0x3774f3800000000,  0x6ee9e7000000000,
  0xddd3ce000000000,  0x1bba79c000000000, 0x3774f38000000000,
  0x6ee9e70000000000, 0xddd3ce0000000000, 0xbba79c0200000000,
  0x774f380600000000
};
constexpr inline uint64_t AA_SEED_Y_RIGHT_33BITS_ROLL_TABLE[33] = {
  0x181a97610, 0x10352ec21, 0x6a5d843,   0xd4bb086,   0x1a97610c,  0x352ec218,
  0x6a5d8430,  0xd4bb0860,  0x1a97610c0, 0x152ec2181, 0xa5d84303,  0x14bb08606,
  0x97610c0d,  0x12ec2181a, 0x5d843035,  0xbb08606a,  0x17610c0d4, 0xec2181a9,
  0x1d8430352, 0x1b08606a5, 0x1610c0d4b, 0xc2181a97,  0x18430352e, 0x108606a5d,
  0x10c0d4bb,  0x2181a976,  0x430352ec,  0x8606a5d8,  0x10c0d4bb0, 0x181a9761,
  0x30352ec2,  0x606a5d84,  0xc0d4bb08
};
constexpr inline uint64_t AA_SEED_Y_LEFT_31BITS_ROLL_TABLE[31] = {
  0x5852ffb600000000, 0xb0a5ff6c00000000, 0x614bfeda00000000,
  0xc297fdb400000000, 0x852ffb6a00000000, 0xa5ff6d600000000,
  0x14bfedac00000000, 0x297fdb5800000000, 0x52ffb6b000000000,
  0xa5ff6d6000000000, 0x4bfedac200000000, 0x97fdb58400000000,
  0x2ffb6b0a00000000, 0x5ff6d61400000000, 0xbfedac2800000000,
  0x7fdb585200000000, 0xffb6b0a400000000, 0xff6d614a00000000,
  0xfedac29600000000, 0xfdb5852e00000000, 0xfb6b0a5e00000000,
  0xf6d614be00000000, 0xedac297e00000000, 0xdb5852fe00000000,
  0xb6b0a5fe00000000, 0x6d614bfe00000000, 0xdac297fc00000000,
  0xb5852ffa00000000, 0x6b0a5ff600000000, 0xd614bfec00000000,
  0xac297fda00000000
};

// clang-format off
constexpr inline uint64_t AA_SEED___RIGHT_33BITS_ROLL_TABLE[33]{ // NOLINT
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__
};
constexpr inline uint64_t AA_SEED___LEFT_31BITS_ROLL_TABLE[31]{ // NOLINT
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__, AA_SEED__,
  AA_SEED__, AA_SEED__, AA_SEED__
};
// clang-format on

constexpr inline const uint64_t* AA_SEED_RIGHT_33BITS_ROLL_TABLE[ASCII_SIZE] = {
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 0..7
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 8..15
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 16..23
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 24..31
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 32..39
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 40..47
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 48..55
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 56..63
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED_A_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED_C_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED_D_RIGHT_33BITS_ROLL_TABLE, AA_SEED_E_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED_F_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED_G_RIGHT_33BITS_ROLL_TABLE, // 64..71
  AA_SEED_H_RIGHT_33BITS_ROLL_TABLE, AA_SEED_I_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED_K_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED_L_RIGHT_33BITS_ROLL_TABLE, AA_SEED_M_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED_N_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 72..79
  AA_SEED_P_RIGHT_33BITS_ROLL_TABLE, AA_SEED_Q_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED_R_RIGHT_33BITS_ROLL_TABLE, AA_SEED_S_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED_T_RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED_V_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED_W_RIGHT_33BITS_ROLL_TABLE, // 80..87
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED_Y_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 88..95
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED_A_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED_C_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED_D_RIGHT_33BITS_ROLL_TABLE, AA_SEED_E_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED_F_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED_G_RIGHT_33BITS_ROLL_TABLE, // 96..103
  AA_SEED_H_RIGHT_33BITS_ROLL_TABLE, AA_SEED_I_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED_K_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED_L_RIGHT_33BITS_ROLL_TABLE, AA_SEED_M_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED_N_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 104..111
  AA_SEED_P_RIGHT_33BITS_ROLL_TABLE, AA_SEED_Q_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED_R_RIGHT_33BITS_ROLL_TABLE, AA_SEED_S_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED_T_RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED_V_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED_W_RIGHT_33BITS_ROLL_TABLE, // 112..119
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED_Y_RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 120..127
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 128..135
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 136..143
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 144..151
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 152..159
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 160..167
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 168..175
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 176..183
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 184..191
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 192..199
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 200..207
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 208..215
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 216..223
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 224..231
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 232..239
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, // 240..247
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE, AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE,
  AA_SEED___RIGHT_33BITS_ROLL_TABLE // 248..255
};

constexpr inline const uint64_t* AA_SEED_LEFT_31BITS_ROLL_TABLE[ASCII_SIZE] = {
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 0..7
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 8..15
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 16..23
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 24..31
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 32..39
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 40..47
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 48..55
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 56..63
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED_A_LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED_C_LEFT_31BITS_ROLL_TABLE,
  AA_SEED_D_LEFT_31BITS_ROLL_TABLE, AA_SEED_E_LEFT_31BITS_ROLL_TABLE,
  AA_SEED_F_LEFT_31BITS_ROLL_TABLE,
  AA_SEED_G_LEFT_31BITS_ROLL_TABLE, // 64..71
  AA_SEED_H_LEFT_31BITS_ROLL_TABLE, AA_SEED_I_LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED_K_LEFT_31BITS_ROLL_TABLE,
  AA_SEED_L_LEFT_31BITS_ROLL_TABLE, AA_SEED_M_LEFT_31BITS_ROLL_TABLE,
  AA_SEED_N_LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 72..79
  AA_SEED_P_LEFT_31BITS_ROLL_TABLE, AA_SEED_Q_LEFT_31BITS_ROLL_TABLE,
  AA_SEED_R_LEFT_31BITS_ROLL_TABLE, AA_SEED_S_LEFT_31BITS_ROLL_TABLE,
  AA_SEED_T_LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED_V_LEFT_31BITS_ROLL_TABLE,
  AA_SEED_W_LEFT_31BITS_ROLL_TABLE, // 80..87
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED_Y_LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 88..95
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED_A_LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED_C_LEFT_31BITS_ROLL_TABLE,
  AA_SEED_D_LEFT_31BITS_ROLL_TABLE, AA_SEED_E_LEFT_31BITS_ROLL_TABLE,
  AA_SEED_F_LEFT_31BITS_ROLL_TABLE,
  AA_SEED_G_LEFT_31BITS_ROLL_TABLE, // 96..103
  AA_SEED_H_LEFT_31BITS_ROLL_TABLE, AA_SEED_I_LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED_K_LEFT_31BITS_ROLL_TABLE,
  AA_SEED_L_LEFT_31BITS_ROLL_TABLE, AA_SEED_M_LEFT_31BITS_ROLL_TABLE,
  AA_SEED_N_LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 104..111
  AA_SEED_P_LEFT_31BITS_ROLL_TABLE, AA_SEED_Q_LEFT_31BITS_ROLL_TABLE,
  AA_SEED_R_LEFT_31BITS_ROLL_TABLE, AA_SEED_S_LEFT_31BITS_ROLL_TABLE,
  AA_SEED_T_LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED_V_LEFT_31BITS_ROLL_TABLE,
  AA_SEED_W_LEFT_31BITS_ROLL_TABLE, // 112..119
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED_Y_LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 120..127
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 128..135
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 136..143
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 144..151
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 152..159
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 160..167
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 168..175
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 176..183
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 184..191
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 192..199
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 200..207
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 208..215
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 216..223
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 224..231
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 232..239
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, // 240..247
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE, AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE,
  AA_SEED___LEFT_31BITS_ROLL_TABLE // 248..255
};

constexpr inline uint64_t LEVEL_2_AA_SEED_C___ = 0x1d07fd644abe9962; // NOLINT
constexpr inline uint64_t LEVEL_2_AA_SEED_G___ = 0xf59c50929bdf4360; // NOLINT
constexpr inline uint64_t LEVEL_2_AA_SEED_A___ = 0x6f735c82fe9c6c03; // NOLINT
constexpr inline uint64_t LEVEL_2_AA_SEED_TS__ = 0xe7392f0ba1dbc3b0; // NOLINT
constexpr inline uint64_t LEVEL_2_AA_SEED_N___ = 0x956ddcfcd4b3961f; // NOLINT
constexpr inline uint64_t LEVEL_2_AA_SEED_DE__ = 0x4ec0ef1bac4f5efa; // NOLINT
constexpr inline uint64_t LEVEL_2_AA_SEED_QKR_ = 0x1cd6ca491872ed78; // NOLINT
constexpr inline uint64_t LEVEL_2_AA_SEED_VILM = 0x547ef17894921035; // NOLINT
constexpr inline uint64_t LEVEL_2_AA_SEED_WFY_ = 0x419722edb87bf79f; // NOLINT
constexpr inline uint64_t LEVEL_2_AA_SEED_H___ = 0xdd5cce5bfdc32de1; // NOLINT
constexpr inline uint64_t LEVEL_2_AA_SEED_P___ = 0x90e0c5e0c07d6598; // NOLINT
constexpr inline uint64_t LEVEL_2_AA_SEED_____ = AA_SEED__;          // NOLINT

constexpr inline uint64_t LEVEL_2_AA_SEED_TABLE[ASCII_SIZE] = {
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 0..7
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 8..15
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 16..23
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 24..31
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 32..39
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 40..47
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 48..55
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 56..63
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_A___, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_C___, LEVEL_2_AA_SEED_DE__, LEVEL_2_AA_SEED_DE__,
  LEVEL_2_AA_SEED_WFY_, LEVEL_2_AA_SEED_G___, // 64..71
  LEVEL_2_AA_SEED_H___, LEVEL_2_AA_SEED_VILM, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_QKR_, LEVEL_2_AA_SEED_VILM, LEVEL_2_AA_SEED_VILM,
  LEVEL_2_AA_SEED_N___, LEVEL_2_AA_SEED_____, // 72..79
  LEVEL_2_AA_SEED_P___, LEVEL_2_AA_SEED_QKR_, LEVEL_2_AA_SEED_QKR_,
  LEVEL_2_AA_SEED_TS__, LEVEL_2_AA_SEED_TS__, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_VILM, LEVEL_2_AA_SEED_WFY_, // 80..87
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_WFY_, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 88..95
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_A___, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_C___, LEVEL_2_AA_SEED_DE__, LEVEL_2_AA_SEED_DE__,
  LEVEL_2_AA_SEED_WFY_, LEVEL_2_AA_SEED_G___, // 96..103
  LEVEL_2_AA_SEED_H___, LEVEL_2_AA_SEED_VILM, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_QKR_, LEVEL_2_AA_SEED_VILM, LEVEL_2_AA_SEED_VILM,
  LEVEL_2_AA_SEED_N___, LEVEL_2_AA_SEED_____, // 104..111
  LEVEL_2_AA_SEED_P___, LEVEL_2_AA_SEED_QKR_, LEVEL_2_AA_SEED_QKR_,
  LEVEL_2_AA_SEED_TS__, LEVEL_2_AA_SEED_TS__, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_VILM, LEVEL_2_AA_SEED_WFY_, // 112..119
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_WFY_, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 120..127
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 128..135
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 136..143
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 144..151
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 152..159
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 160..167
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 168..175
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 176..183
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 184..191
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 192..199
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 200..207
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 208..215
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 216..223
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 224..231
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 232..239
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, // 240..247
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____,
  LEVEL_2_AA_SEED_____, LEVEL_2_AA_SEED_____ // 248..255
};

constexpr inline uint64_t
  LEVEL_2_AA_SEED_C____RIGHT_33BITS_ROLL_TABLE[33] = { // NOLINT
    0x4abe9962,  0x957d32c4,  0x12afa6588, 0x55f4cb11,  0xabe99622, 0x157d32c44,
    0xafa65889,  0x15f4cb112, 0xbe996225,  0x17d32c44a, 0xfa658895, 0x1f4cb112a,
    0x1e9962255, 0x1d32c44ab, 0x1a6588957, 0x14cb112af, 0x9962255f, 0x132c44abe,
    0x6588957d,  0xcb112afa,  0x1962255f4, 0x12c44abe9, 0x588957d3, 0xb112afa6,
    0x162255f4c, 0xc44abe99,  0x188957d32, 0x1112afa65, 0x2255f4cb, 0x44abe996,
    0x8957d32c,  0x112afa658, 0x255f4cb1
  };
constexpr inline uint64_t
  LEVEL_2_AA_SEED_C____LEFT_31BITS_ROLL_TABLE[31] = { // NOLINT
    0x1d07fd6400000000, 0x3a0ffac800000000, 0x741ff59000000000,
    0xe83feb2000000000, 0xd07fd64200000000, 0xa0ffac8600000000,
    0x41ff590e00000000, 0x83feb21c00000000, 0x7fd643a00000000,
    0xffac87400000000,  0x1ff590e800000000, 0x3feb21d000000000,
    0x7fd643a000000000, 0xffac874000000000, 0xff590e8200000000,
    0xfeb21d0600000000, 0xfd643a0e00000000, 0xfac8741e00000000,
    0xf590e83e00000000, 0xeb21d07e00000000, 0xd643a0fe00000000,
    0xac8741fe00000000, 0x590e83fe00000000, 0xb21d07fc00000000,
    0x643a0ffa00000000, 0xc8741ff400000000, 0x90e83fea00000000,
    0x21d07fd600000000, 0x43a0ffac00000000, 0x8741ff5800000000,
    0xe83feb200000000
  };
constexpr inline uint64_t
  LEVEL_2_AA_SEED_G____RIGHT_33BITS_ROLL_TABLE[33] = { // NOLINT
    0x9bdf4360,  0x137be86c0, 0x6f7d0d81,  0xdefa1b02,  0x1bdf43604,
    0x17be86c09, 0xf7d0d813,  0x1efa1b026, 0x1df43604d, 0x1be86c09b,
    0x17d0d8137, 0xfa1b026f,  0x1f43604de, 0x1e86c09bd, 0x1d0d8137b,
    0x1a1b026f7, 0x143604def, 0x86c09bdf,  0x10d8137be, 0x1b026f7d,
    0x3604defa,  0x6c09bdf4,  0xd8137be8,  0x1b026f7d0, 0x1604defa1,
    0xc09bdf43,  0x18137be86, 0x1026f7d0d, 0x4defa1b,   0x9bdf436,
    0x137be86c,  0x26f7d0d8,  0x4defa1b0
  };
constexpr inline uint64_t
  LEVEL_2_AA_SEED_G____LEFT_31BITS_ROLL_TABLE[31] = { // NOLINT
    0xf59c509200000000, 0xeb38a12600000000, 0xd671424e00000000,
    0xace2849e00000000, 0x59c5093e00000000, 0xb38a127c00000000,
    0x671424fa00000000, 0xce2849f400000000, 0x9c5093ea00000000,
    0x38a127d600000000, 0x71424fac00000000, 0xe2849f5800000000,
    0xc5093eb200000000, 0x8a127d6600000000, 0x1424face00000000,
    0x2849f59c00000000, 0x5093eb3800000000, 0xa127d67000000000,
    0x424face200000000, 0x849f59c400000000, 0x93eb38a00000000,
    0x127d671400000000, 0x24face2800000000, 0x49f59c5000000000,
    0x93eb38a000000000, 0x27d6714200000000, 0x4face28400000000,
    0x9f59c50800000000, 0x3eb38a1200000000, 0x7d67142400000000,
    0xface284800000000
  };
constexpr inline uint64_t
  LEVEL_2_AA_SEED_A____RIGHT_33BITS_ROLL_TABLE[33] = { // NOLINT
    0xfe9c6c03,  0x1fd38d806, 0x1fa71b00d, 0x1f4e3601b, 0x1e9c6c037,
    0x1d38d806f, 0x1a71b00df, 0x14e3601bf, 0x9c6c037f,  0x138d806fe,
    0x71b00dfd,  0xe3601bfa,  0x1c6c037f4, 0x18d806fe9, 0x11b00dfd3,
    0x3601bfa7,  0x6c037f4e,  0xd806fe9c,  0x1b00dfd38, 0x1601bfa71,
    0xc037f4e3,  0x1806fe9c6, 0x100dfd38d, 0x1bfa71b,   0x37f4e36,
    0x6fe9c6c,   0xdfd38d8,   0x1bfa71b0,  0x37f4e360,  0x6fe9c6c0,
    0xdfd38d80,  0x1bfa71b00, 0x17f4e3601
  };
constexpr inline uint64_t
  LEVEL_2_AA_SEED_A____LEFT_31BITS_ROLL_TABLE[31] = { // NOLINT
    0x6f735c8200000000, 0xdee6b90400000000, 0xbdcd720a00000000,
    0x7b9ae41600000000, 0xf735c82c00000000, 0xee6b905a00000000,
    0xdcd720b600000000, 0xb9ae416e00000000, 0x735c82de00000000,
    0xe6b905bc00000000, 0xcd720b7a00000000, 0x9ae416f600000000,
    0x35c82dee00000000, 0x6b905bdc00000000, 0xd720b7b800000000,
    0xae416f7200000000, 0x5c82dee600000000, 0xb905bdcc00000000,
    0x720b7b9a00000000, 0xe416f73400000000, 0xc82dee6a00000000,
    0x905bdcd600000000, 0x20b7b9ae00000000, 0x416f735c00000000,
    0x82dee6b800000000, 0x5bdcd7200000000,  0xb7b9ae400000000,
    0x16f735c800000000, 0x2dee6b9000000000, 0x5bdcd72000000000,
    0xb7b9ae4000000000
  };
constexpr inline uint64_t
  LEVEL_2_AA_SEED_TS___RIGHT_33BITS_ROLL_TABLE[33] = { // NOLINT
    0x1a1dbc3b0, 0x143b78761, 0x876f0ec3,  0x10ede1d86, 0x1dbc3b0d,
    0x3b78761a,  0x76f0ec34,  0xede1d868,  0x1dbc3b0d0, 0x1b78761a1,
    0x16f0ec343, 0xde1d8687,  0x1bc3b0d0e, 0x178761a1d, 0xf0ec343b,
    0x1e1d86876, 0x1c3b0d0ed, 0x18761a1db, 0x10ec343b7, 0x1d86876f,
    0x3b0d0ede,  0x761a1dbc,  0xec343b78,  0x1d86876f0, 0x1b0d0ede1,
    0x161a1dbc3, 0xc343b787,  0x186876f0e, 0x10d0ede1d, 0x1a1dbc3b,
    0x343b7876,  0x6876f0ec,  0xd0ede1d8
  };
constexpr inline uint64_t
  LEVEL_2_AA_SEED_TS___LEFT_31BITS_ROLL_TABLE[31] = { // NOLINT
    0xe7392f0a00000000, 0xce725e1600000000, 0x9ce4bc2e00000000,
    0x39c9785e00000000, 0x7392f0bc00000000, 0xe725e17800000000,
    0xce4bc2f200000000, 0x9c9785e600000000, 0x392f0bce00000000,
    0x725e179c00000000, 0xe4bc2f3800000000, 0xc9785e7200000000,
    0x92f0bce600000000, 0x25e179ce00000000, 0x4bc2f39c00000000,
    0x9785e73800000000, 0x2f0bce7200000000, 0x5e179ce400000000,
    0xbc2f39c800000000, 0x785e739200000000, 0xf0bce72400000000,
    0xe179ce4a00000000, 0xc2f39c9600000000, 0x85e7392e00000000,
    0xbce725e00000000,  0x179ce4bc00000000, 0x2f39c97800000000,
    0x5e7392f000000000, 0xbce725e000000000, 0x79ce4bc200000000,
    0xf39c978400000000
  };
constexpr inline uint64_t
  LEVEL_2_AA_SEED_N____RIGHT_33BITS_ROLL_TABLE[33] = { // NOLINT
    0xd4b3961f,  0x1a9672c3e, 0x152ce587d, 0xa59cb0fb,  0x14b3961f6,
    0x9672c3ed,  0x12ce587da, 0x59cb0fb5,  0xb3961f6a,  0x1672c3ed4,
    0xce587da9,  0x19cb0fb52, 0x13961f6a5, 0x72c3ed4b,  0xe587da96,
    0x1cb0fb52c, 0x1961f6a59, 0x12c3ed4b3, 0x587da967,  0xb0fb52ce,
    0x161f6a59c, 0xc3ed4b39,  0x187da9672, 0x10fb52ce5, 0x1f6a59cb,
    0x3ed4b396,  0x7da9672c,  0xfb52ce58,  0x1f6a59cb0, 0x1ed4b3961,
    0x1da9672c3, 0x1b52ce587, 0x16a59cb0f
  };
constexpr inline uint64_t
  LEVEL_2_AA_SEED_N____LEFT_31BITS_ROLL_TABLE[31] = { // NOLINT
    0x956ddcfc00000000, 0x2adbb9fa00000000, 0x55b773f400000000,
    0xab6ee7e800000000, 0x56ddcfd200000000, 0xadbb9fa400000000,
    0x5b773f4a00000000, 0xb6ee7e9400000000, 0x6ddcfd2a00000000,
    0xdbb9fa5400000000, 0xb773f4aa00000000, 0x6ee7e95600000000,
    0xddcfd2ac00000000, 0xbb9fa55a00000000, 0x773f4ab600000000,
    0xee7e956c00000000, 0xdcfd2ada00000000, 0xb9fa55b600000000,
    0x73f4ab6e00000000, 0xe7e956dc00000000, 0xcfd2adba00000000,
    0x9fa55b7600000000, 0x3f4ab6ee00000000, 0x7e956ddc00000000,
    0xfd2adbb800000000, 0xfa55b77200000000, 0xf4ab6ee600000000,
    0xe956ddce00000000, 0xd2adbb9e00000000, 0xa55b773e00000000,
    0x4ab6ee7e00000000
  };
constexpr inline uint64_t
  LEVEL_2_AA_SEED_DE___RIGHT_33BITS_ROLL_TABLE[33] = { // NOLINT
    0x1ac4f5efa, 0x1589ebdf5, 0xb13d7beb,  0x1627af7d6, 0xc4f5efad,
    0x189ebdf5a, 0x113d7beb5, 0x27af7d6b,  0x4f5efad6,  0x9ebdf5ac,
    0x13d7beb58, 0x7af7d6b1,  0xf5efad62,  0x1ebdf5ac4, 0x1d7beb589,
    0x1af7d6b13, 0x15efad627, 0xbdf5ac4f,  0x17beb589e, 0xf7d6b13d,
    0x1efad627a, 0x1df5ac4f5, 0x1beb589eb, 0x17d6b13d7, 0xfad627af,
    0x1f5ac4f5e, 0x1eb589ebd, 0x1d6b13d7b, 0x1ad627af7, 0x15ac4f5ef,
    0xb589ebdf,  0x16b13d7be, 0xd627af7d
  };
constexpr inline uint64_t
  LEVEL_2_AA_SEED_DE___LEFT_31BITS_ROLL_TABLE[31] = { // NOLINT
    0x4ec0ef1a00000000, 0x9d81de3400000000, 0x3b03bc6a00000000,
    0x760778d400000000, 0xec0ef1a800000000, 0xd81de35200000000,
    0xb03bc6a600000000, 0x60778d4e00000000, 0xc0ef1a9c00000000,
    0x81de353a00000000, 0x3bc6a7600000000,  0x778d4ec00000000,
    0xef1a9d800000000,  0x1de353b000000000, 0x3bc6a76000000000,
    0x778d4ec000000000, 0xef1a9d8000000000, 0xde353b0200000000,
    0xbc6a760600000000, 0x78d4ec0e00000000, 0xf1a9d81c00000000,
    0xe353b03a00000000, 0xc6a7607600000000, 0x8d4ec0ee00000000,
    0x1a9d81de00000000, 0x353b03bc00000000, 0x6a76077800000000,
    0xd4ec0ef000000000, 0xa9d81de200000000, 0x53b03bc600000000,
    0xa760778c00000000
  };
constexpr inline uint64_t
  LEVEL_2_AA_SEED_QKR__RIGHT_33BITS_ROLL_TABLE[33] = { // NOLINT
    0x11872ed78, 0x30e5daf1,  0x61cbb5e2,  0xc3976bc4,  0x1872ed788,
    0x10e5daf11, 0x1cbb5e23,  0x3976bc46,  0x72ed788c,  0xe5daf118,
    0x1cbb5e230, 0x1976bc461, 0x12ed788c3, 0x5daf1187,  0xbb5e230e,
    0x176bc461c, 0xed788c39,  0x1daf11872, 0x1b5e230e5, 0x16bc461cb,
    0xd788c397,  0x1af11872e, 0x15e230e5d, 0xbc461cbb,  0x1788c3976,
    0xf11872ed,  0x1e230e5da, 0x1c461cbb5, 0x188c3976b, 0x111872ed7,
    0x230e5daf,  0x461cbb5e,  0x8c3976bc
  };
constexpr inline uint64_t
  LEVEL_2_AA_SEED_QKR__LEFT_31BITS_ROLL_TABLE[31] = { // NOLINT
    0x1cd6ca4800000000, 0x39ad949000000000, 0x735b292000000000,
    0xe6b6524000000000, 0xcd6ca48200000000, 0x9ad9490600000000,
    0x35b2920e00000000, 0x6b65241c00000000, 0xd6ca483800000000,
    0xad94907200000000, 0x5b2920e600000000, 0xb65241cc00000000,
    0x6ca4839a00000000, 0xd949073400000000, 0xb2920e6a00000000,
    0x65241cd600000000, 0xca4839ac00000000, 0x9490735a00000000,
    0x2920e6b600000000, 0x5241cd6c00000000, 0xa4839ad800000000,
    0x490735b200000000, 0x920e6b6400000000, 0x241cd6ca00000000,
    0x4839ad9400000000, 0x90735b2800000000, 0x20e6b65200000000,
    0x41cd6ca400000000, 0x839ad94800000000, 0x735b29200000000,
    0xe6b652400000000
  };
constexpr inline uint64_t LEVEL_2_AA_SEED_VILM_RIGHT_33BITS_ROLL_TABLE[33] = {
  0x94921035,  0x12924206a, 0x524840d5,  0xa49081aa,  0x149210354, 0x924206a9,
  0x124840d52, 0x49081aa5,  0x9210354a,  0x124206a94, 0x4840d529,  0x9081aa52,
  0x1210354a4, 0x4206a949,  0x840d5292,  0x1081aa524, 0x10354a49,  0x206a9492,
  0x40d52924,  0x81aa5248,  0x10354a490, 0x6a94921,   0xd529242,   0x1aa52484,
  0x354a4908,  0x6a949210,  0xd5292420,  0x1aa524840, 0x154a49081, 0xa9492103,
  0x152924206, 0xa524840d,  0x14a49081a
};
constexpr inline uint64_t LEVEL_2_AA_SEED_VILM_LEFT_31BITS_ROLL_TABLE[31] = {
  0x547ef17800000000, 0xa8fde2f000000000, 0x51fbc5e200000000,
  0xa3f78bc400000000, 0x47ef178a00000000, 0x8fde2f1400000000,
  0x1fbc5e2a00000000, 0x3f78bc5400000000, 0x7ef178a800000000,
  0xfde2f15000000000, 0xfbc5e2a200000000, 0xf78bc54600000000,
  0xef178a8e00000000, 0xde2f151e00000000, 0xbc5e2a3e00000000,
  0x78bc547e00000000, 0xf178a8fc00000000, 0xe2f151fa00000000,
  0xc5e2a3f600000000, 0x8bc547ee00000000, 0x178a8fde00000000,
  0x2f151fbc00000000, 0x5e2a3f7800000000, 0xbc547ef000000000,
  0x78a8fde200000000, 0xf151fbc400000000, 0xe2a3f78a00000000,
  0xc547ef1600000000, 0x8a8fde2e00000000, 0x151fbc5e00000000,
  0x2a3f78bc00000000
};
constexpr inline uint64_t
  LEVEL_2_AA_SEED_WFY__RIGHT_33BITS_ROLL_TABLE[33] = { // NOLINT
    0x1b87bf79f, 0x170f7ef3f, 0xe1efde7f,  0x1c3dfbcfe, 0x187bf79fd,
    0x10f7ef3fb, 0x1efde7f7,  0x3dfbcfee,  0x7bf79fdc,  0xf7ef3fb8,
    0x1efde7f70, 0x1dfbcfee1, 0x1bf79fdc3, 0x17ef3fb87, 0xfde7f70f,
    0x1fbcfee1e, 0x1f79fdc3d, 0x1ef3fb87b, 0x1de7f70f7, 0x1bcfee1ef,
    0x179fdc3df, 0xf3fb87bf,  0x1e7f70f7e, 0x1cfee1efd, 0x19fdc3dfb,
    0x13fb87bf7, 0x7f70f7ef,  0xfee1efde,  0x1fdc3dfbc, 0x1fb87bf79,
    0x1f70f7ef3, 0x1ee1efde7, 0x1dc3dfbcf
  };
constexpr inline uint64_t
  LEVEL_2_AA_SEED_WFY__LEFT_31BITS_ROLL_TABLE[31] = { // NOLINT
    0x419722ec00000000, 0x832e45d800000000, 0x65c8bb200000000,
    0xcb9176400000000,  0x19722ec800000000, 0x32e45d9000000000,
    0x65c8bb2000000000, 0xcb91764000000000, 0x9722ec8200000000,
    0x2e45d90600000000, 0x5c8bb20c00000000, 0xb917641800000000,
    0x722ec83200000000, 0xe45d906400000000, 0xc8bb20ca00000000,
    0x9176419600000000, 0x22ec832e00000000, 0x45d9065c00000000,
    0x8bb20cb800000000, 0x1764197200000000, 0x2ec832e400000000,
    0x5d9065c800000000, 0xbb20cb9000000000, 0x7641972200000000,
    0xec832e4400000000, 0xd9065c8a00000000, 0xb20cb91600000000,
    0x6419722e00000000, 0xc832e45c00000000, 0x9065c8ba00000000,
    0x20cb917600000000
  };
constexpr inline uint64_t
  LEVEL_2_AA_SEED_H____RIGHT_33BITS_ROLL_TABLE[33] = { // NOLINT
    0x1fdc32de1, 0x1fb865bc3, 0x1f70cb787, 0x1ee196f0f, 0x1dc32de1f,
    0x1b865bc3f, 0x170cb787f, 0xe196f0ff,  0x1c32de1fe, 0x1865bc3fd,
    0x10cb787fb, 0x196f0ff7,  0x32de1fee,  0x65bc3fdc,  0xcb787fb8,
    0x196f0ff70, 0x12de1fee1, 0x5bc3fdc3,  0xb787fb86,  0x16f0ff70c,
    0xde1fee19,  0x1bc3fdc32, 0x1787fb865, 0xf0ff70cb,  0x1e1fee196,
    0x1c3fdc32d, 0x187fb865b, 0x10ff70cb7, 0x1fee196f,  0x3fdc32de,
    0x7fb865bc,  0xff70cb78,  0x1fee196f0
  };
constexpr inline uint64_t
  LEVEL_2_AA_SEED_H____LEFT_31BITS_ROLL_TABLE[31] = { // NOLINT
    0xdd5cce5a00000000, 0xbab99cb600000000, 0x7573396e00000000,
    0xeae672dc00000000, 0xd5cce5ba00000000, 0xab99cb7600000000,
    0x573396ee00000000, 0xae672ddc00000000, 0x5cce5bba00000000,
    0xb99cb77400000000, 0x73396eea00000000, 0xe672ddd400000000,
    0xcce5bbaa00000000, 0x99cb775600000000, 0x3396eeae00000000,
    0x672ddd5c00000000, 0xce5bbab800000000, 0x9cb7757200000000,
    0x396eeae600000000, 0x72ddd5cc00000000, 0xe5bbab9800000000,
    0xcb77573200000000, 0x96eeae6600000000, 0x2ddd5cce00000000,
    0x5bbab99c00000000, 0xb775733800000000, 0x6eeae67200000000,
    0xddd5cce400000000, 0xbbab99ca00000000, 0x7757339600000000,
    0xeeae672c00000000
  };
constexpr inline uint64_t
  LEVEL_2_AA_SEED_P____RIGHT_33BITS_ROLL_TABLE[33] = { // NOLINT
    0xc07d6598,  0x180facb30, 0x101f59661, 0x3eb2cc3,  0x7d65986,   0xfacb30c,
    0x1f596618,  0x3eb2cc30,  0x7d659860,  0xfacb30c0, 0x1f5966180, 0x1eb2cc301,
    0x1d6598603, 0x1acb30c07, 0x15966180f, 0xb2cc301f, 0x16598603e, 0xcb30c07d,
    0x1966180fa, 0x12cc301f5, 0x598603eb,  0xb30c07d6, 0x166180fac, 0xcc301f59,
    0x198603eb2, 0x130c07d65, 0x6180facb,  0xc301f596, 0x18603eb2c, 0x10c07d659,
    0x180facb3,  0x301f5966,  0x603eb2cc
  };
constexpr inline uint64_t
  LEVEL_2_AA_SEED_P____LEFT_31BITS_ROLL_TABLE[31] = { // NOLINT
    0x90e0c5e000000000, 0x21c18bc200000000, 0x4383178400000000,
    0x87062f0800000000, 0xe0c5e1200000000,  0x1c18bc2400000000,
    0x3831784800000000, 0x7062f09000000000, 0xe0c5e12000000000,
    0xc18bc24200000000, 0x8317848600000000, 0x62f090e00000000,
    0xc5e121c00000000,  0x18bc243800000000, 0x3178487000000000,
    0x62f090e000000000, 0xc5e121c000000000, 0x8bc2438200000000,
    0x1784870600000000, 0x2f090e0c00000000, 0x5e121c1800000000,
    0xbc24383000000000, 0x7848706200000000, 0xf090e0c400000000,
    0xe121c18a00000000, 0xc243831600000000, 0x8487062e00000000,
    0x90e0c5e00000000,  0x121c18bc00000000, 0x2438317800000000,
    0x487062f000000000
  };

constexpr inline const uint64_t*
  LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE = // NOLINT
  AA_SEED___RIGHT_33BITS_ROLL_TABLE;
constexpr inline const uint64_t*
  LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE = // NOLINT
  AA_SEED___LEFT_31BITS_ROLL_TABLE;

constexpr inline const uint64_t*
  LEVEL_2_AA_SEED_RIGHT_33BITS_ROLL_TABLE[ASCII_SIZE] = {
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 0..7
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 8..15
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 16..23
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 24..31
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 32..39
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 40..47
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 48..55
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 56..63
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_A____RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_C____RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_DE___RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_DE___RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_WFY__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_G____RIGHT_33BITS_ROLL_TABLE, // 64..71
    LEVEL_2_AA_SEED_H____RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_VILM_RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_QKR__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_VILM_RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_VILM_RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_N____RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 72..79
    LEVEL_2_AA_SEED_P____RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_QKR__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_QKR__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_TS___RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_TS___RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_VILM_RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_WFY__RIGHT_33BITS_ROLL_TABLE, // 80..87
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_WFY__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 88..95
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_A____RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_C____RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_DE___RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_DE___RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_WFY__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_G____RIGHT_33BITS_ROLL_TABLE, // 96..103
    LEVEL_2_AA_SEED_H____RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_VILM_RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_QKR__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_VILM_RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_VILM_RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_N____RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 104..111
    LEVEL_2_AA_SEED_P____RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_QKR__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_QKR__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_TS___RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_TS___RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_VILM_RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_WFY__RIGHT_33BITS_ROLL_TABLE, // 112..119
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_WFY__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 120..127
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 128..135
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 136..143
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 144..151
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 152..159
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 160..167
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 168..175
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 176..183
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 184..191
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 192..199
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 200..207
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 208..215
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 216..223
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 224..231
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 232..239
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 240..247
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______RIGHT_33BITS_ROLL_TABLE // 248..255
  };

constexpr inline const uint64_t*
  LEVEL_2_AA_SEED_LEFT_31BITS_ROLL_TABLE[ASCII_SIZE] = {
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 0..7
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 8..15
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 16..23
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 24..31
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 32..39
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 40..47
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 48..55
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 56..63
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_A____LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_C____LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_DE___LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_DE___LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_WFY__LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_G____LEFT_31BITS_ROLL_TABLE, // 64..71
    LEVEL_2_AA_SEED_H____LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_VILM_LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_QKR__LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_VILM_LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_VILM_LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_N____LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 72..79
    LEVEL_2_AA_SEED_P____LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_QKR__LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_QKR__LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_TS___LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_TS___LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_VILM_LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_WFY__LEFT_31BITS_ROLL_TABLE, // 80..87
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_WFY__LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 88..95
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_A____LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_C____LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_DE___LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_DE___LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_WFY__LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_G____LEFT_31BITS_ROLL_TABLE, // 96..103
    LEVEL_2_AA_SEED_H____LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_VILM_LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_QKR__LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_VILM_LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_VILM_LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_N____LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 104..111
    LEVEL_2_AA_SEED_P____LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_QKR__LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_QKR__LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_TS___LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_TS___LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_VILM_LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_WFY__LEFT_31BITS_ROLL_TABLE, // 112..119
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_WFY__LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 120..127
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 128..135
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 136..143
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 144..151
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 152..159
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 160..167
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 168..175
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 176..183
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 184..191
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 192..199
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 200..207
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 208..215
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 216..223
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 224..231
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 232..239
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 240..247
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED______LEFT_31BITS_ROLL_TABLE // 248..255
  };

constexpr inline uint64_t LEVEL_3_AA_SEED_C___ = 0x5713e4c10cebbfa3; // NOLINT
constexpr inline uint64_t LEVEL_3_AA_SEED_G___ = 0xbe084b869537379b; // NOLINT
constexpr inline uint64_t LEVEL_3_AA_SEED_ATS_ = 0x985fd9efa0fe5b82; // NOLINT
constexpr inline uint64_t LEVEL_3_AA_SEED_NDE_ = 0x9aca6c4f4ef69df0; // NOLINT
constexpr inline uint64_t LEVEL_3_AA_SEED_QKR_ = 0x917de473b721df0e; // NOLINT
constexpr inline uint64_t LEVEL_3_AA_SEED_VILM = 0x37cdd84aa07c5bd7;
constexpr inline uint64_t LEVEL_3_AA_SEED_WFY_ = 0x51a7955f1a67a896; // NOLINT
constexpr inline uint64_t LEVEL_3_AA_SEED_H___ = 0x1d2a0ba493708fbf; // NOLINT
constexpr inline uint64_t LEVEL_3_AA_SEED_P___ = 0xfe4c47da16611245; // NOLINT
constexpr inline uint64_t LEVEL_3_AA_SEED_____ = AA_SEED__;          // NOLINT

constexpr inline uint64_t LEVEL_3_AA_SEED_TABLE[ASCII_SIZE] = {
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 0..7
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 8..15
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 16..23
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 24..31
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 32..39
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 40..47
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 48..55
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 56..63
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_ATS_, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_C___, LEVEL_3_AA_SEED_NDE_, LEVEL_3_AA_SEED_NDE_,
  LEVEL_3_AA_SEED_WFY_, LEVEL_3_AA_SEED_G___, // 64..71
  LEVEL_3_AA_SEED_H___, LEVEL_3_AA_SEED_VILM, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_QKR_, LEVEL_3_AA_SEED_VILM, LEVEL_3_AA_SEED_VILM,
  LEVEL_3_AA_SEED_NDE_, LEVEL_3_AA_SEED_____, // 72..79
  LEVEL_3_AA_SEED_P___, LEVEL_3_AA_SEED_QKR_, LEVEL_3_AA_SEED_QKR_,
  LEVEL_3_AA_SEED_ATS_, LEVEL_3_AA_SEED_ATS_, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_VILM, LEVEL_3_AA_SEED_WFY_, // 80..87
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_WFY_, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 88..95
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_ATS_, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_C___, LEVEL_3_AA_SEED_NDE_, LEVEL_3_AA_SEED_NDE_,
  LEVEL_3_AA_SEED_WFY_, LEVEL_3_AA_SEED_G___, // 96..103
  LEVEL_3_AA_SEED_H___, LEVEL_3_AA_SEED_VILM, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_QKR_, LEVEL_3_AA_SEED_VILM, LEVEL_3_AA_SEED_VILM,
  LEVEL_3_AA_SEED_NDE_, LEVEL_3_AA_SEED_____, // 104..111
  LEVEL_3_AA_SEED_P___, LEVEL_3_AA_SEED_QKR_, LEVEL_3_AA_SEED_QKR_,
  LEVEL_3_AA_SEED_ATS_, LEVEL_3_AA_SEED_ATS_, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_VILM, LEVEL_3_AA_SEED_WFY_, // 112..119
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_WFY_, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 120..127
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 128..135
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 136..143
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 144..151
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 152..159
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 160..167
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 168..175
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 176..183
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 184..191
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 192..199
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 200..207
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 208..215
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 216..223
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 224..231
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 232..239
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, // 240..247
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____,
  LEVEL_3_AA_SEED_____, LEVEL_3_AA_SEED_____ // 248..255
};

constexpr inline uint64_t
  LEVEL_3_AA_SEED_C____RIGHT_33BITS_ROLL_TABLE[33] = { // NOLINT
    0x10cebbfa3, 0x19d77f47,  0x33aefe8e,  0x675dfd1c,  0xcebbfa38,
    0x19d77f470, 0x13aefe8e1, 0x75dfd1c3,  0xebbfa386,  0x1d77f470c,
    0x1aefe8e19, 0x15dfd1c33, 0xbbfa3867,  0x177f470ce, 0xefe8e19d,
    0x1dfd1c33a, 0x1bfa38675, 0x17f470ceb, 0xfe8e19d7,  0x1fd1c33ae,
    0x1fa38675d, 0x1f470cebb, 0x1e8e19d77, 0x1d1c33aef, 0x1a38675df,
    0x1470cebbf, 0x8e19d77f,  0x11c33aefe, 0x38675dfd,  0x70cebbfa,
    0xe19d77f4,  0x1c33aefe8, 0x18675dfd1
  };
constexpr inline uint64_t
  LEVEL_3_AA_SEED_C____LEFT_31BITS_ROLL_TABLE[31] = { // NOLINT
    0x5713e4c000000000, 0xae27c98000000000, 0x5c4f930200000000,
    0xb89f260400000000, 0x713e4c0a00000000, 0xe27c981400000000,
    0xc4f9302a00000000, 0x89f2605600000000, 0x13e4c0ae00000000,
    0x27c9815c00000000, 0x4f9302b800000000, 0x9f26057000000000,
    0x3e4c0ae200000000, 0x7c9815c400000000, 0xf9302b8800000000,
    0xf260571200000000, 0xe4c0ae2600000000, 0xc9815c4e00000000,
    0x9302b89e00000000, 0x2605713e00000000, 0x4c0ae27c00000000,
    0x9815c4f800000000, 0x302b89f200000000, 0x605713e400000000,
    0xc0ae27c800000000, 0x815c4f9200000000, 0x2b89f2600000000,
    0x5713e4c00000000,  0xae27c9800000000,  0x15c4f93000000000,
    0x2b89f26000000000
  };
constexpr inline uint64_t
  LEVEL_3_AA_SEED_G____RIGHT_33BITS_ROLL_TABLE[33] = { // NOLINT
    0x9537379b,  0x12a6e6f36, 0x54dcde6d,  0xa9b9bcda,  0x1537379b4,
    0xa6e6f369,  0x14dcde6d2, 0x9b9bcda5,  0x137379b4a, 0x6e6f3695,
    0xdcde6d2a,  0x1b9bcda54, 0x17379b4a9, 0xe6f36953,  0x1cde6d2a6,
    0x19bcda54d, 0x1379b4a9b, 0x6f369537,  0xde6d2a6e,  0x1bcda54dc,
    0x179b4a9b9, 0xf3695373,  0x1e6d2a6e6, 0x1cda54dcd, 0x19b4a9b9b,
    0x136953737, 0x6d2a6e6f,  0xda54dcde,  0x1b4a9b9bc, 0x169537379,
    0xd2a6e6f3,  0x1a54dcde6, 0x14a9b9bcd
  };
constexpr inline uint64_t
  LEVEL_3_AA_SEED_G____LEFT_31BITS_ROLL_TABLE[31] = { // NOLINT
    0xbe084b8600000000, 0x7c10970e00000000, 0xf8212e1c00000000,
    0xf0425c3a00000000, 0xe084b87600000000, 0xc10970ee00000000,
    0x8212e1de00000000, 0x425c3be00000000,  0x84b877c00000000,
    0x10970ef800000000, 0x212e1df000000000, 0x425c3be000000000,
    0x84b877c000000000, 0x970ef8200000000,  0x12e1df0400000000,
    0x25c3be0800000000, 0x4b877c1000000000, 0x970ef82000000000,
    0x2e1df04200000000, 0x5c3be08400000000, 0xb877c10800000000,
    0x70ef821200000000, 0xe1df042400000000, 0xc3be084a00000000,
    0x877c109600000000, 0xef8212e00000000,  0x1df0425c00000000,
    0x3be084b800000000, 0x77c1097000000000, 0xef8212e000000000,
    0xdf0425c200000000
  };
constexpr inline uint64_t
  LEVEL_3_AA_SEED_ATS__RIGHT_33BITS_ROLL_TABLE[33] = { // NOLINT
    0x1a0fe5b82, 0x141fcb705, 0x83f96e0b,  0x107f2dc16, 0xfe5b82d,
    0x1fcb705a,  0x3f96e0b4,  0x7f2dc168,  0xfe5b82d0,  0x1fcb705a0,
    0x1f96e0b41, 0x1f2dc1683, 0x1e5b82d07, 0x1cb705a0f, 0x196e0b41f,
    0x12dc1683f, 0x5b82d07f,  0xb705a0fe,  0x16e0b41fc, 0xdc1683f9,
    0x1b82d07f2, 0x1705a0fe5, 0xe0b41fcb,  0x1c1683f96, 0x182d07f2d,
    0x105a0fe5b, 0xb41fcb7,   0x1683f96e,  0x2d07f2dc,  0x5a0fe5b8,
    0xb41fcb70,  0x1683f96e0, 0xd07f2dc1
  };
constexpr inline uint64_t
  LEVEL_3_AA_SEED_ATS__LEFT_31BITS_ROLL_TABLE[31] = { // NOLINT
    0x985fd9ee00000000, 0x30bfb3de00000000, 0x617f67bc00000000,
    0xc2fecf7800000000, 0x85fd9ef200000000, 0xbfb3de600000000,
    0x17f67bcc00000000, 0x2fecf79800000000, 0x5fd9ef3000000000,
    0xbfb3de6000000000, 0x7f67bcc200000000, 0xfecf798400000000,
    0xfd9ef30a00000000, 0xfb3de61600000000, 0xf67bcc2e00000000,
    0xecf7985e00000000, 0xd9ef30be00000000, 0xb3de617e00000000,
    0x67bcc2fe00000000, 0xcf7985fc00000000, 0x9ef30bfa00000000,
    0x3de617f600000000, 0x7bcc2fec00000000, 0xf7985fd800000000,
    0xef30bfb200000000, 0xde617f6600000000, 0xbcc2fece00000000,
    0x7985fd9e00000000, 0xf30bfb3c00000000, 0xe617f67a00000000,
    0xcc2fecf600000000
  };
constexpr inline uint64_t
  LEVEL_3_AA_SEED_NDE__RIGHT_33BITS_ROLL_TABLE[33] = { // NOLINT
    0x14ef69df0, 0x9ded3be1,  0x13bda77c2, 0x77b4ef85,  0xef69df0a,
    0x1ded3be14, 0x1bda77c29, 0x17b4ef853, 0xf69df0a7,  0x1ed3be14e,
    0x1da77c29d, 0x1b4ef853b, 0x169df0a77, 0xd3be14ef,  0x1a77c29de,
    0x14ef853bd, 0x9df0a77b,  0x13be14ef6, 0x77c29ded,  0xef853bda,
    0x1df0a77b4, 0x1be14ef69, 0x17c29ded3, 0xf853bda7,  0x1f0a77b4e,
    0x1e14ef69d, 0x1c29ded3b, 0x1853bda77, 0x10a77b4ef, 0x14ef69df,
    0x29ded3be,  0x53bda77c,  0xa77b4ef8
  };
constexpr inline uint64_t
  LEVEL_3_AA_SEED_NDE__LEFT_31BITS_ROLL_TABLE[31] = { // NOLINT
    0x9aca6c4e00000000, 0x3594d89e00000000, 0x6b29b13c00000000,
    0xd653627800000000, 0xaca6c4f200000000, 0x594d89e600000000,
    0xb29b13cc00000000, 0x6536279a00000000, 0xca6c4f3400000000,
    0x94d89e6a00000000, 0x29b13cd600000000, 0x536279ac00000000,
    0xa6c4f35800000000, 0x4d89e6b200000000, 0x9b13cd6400000000,
    0x36279aca00000000, 0x6c4f359400000000, 0xd89e6b2800000000,
    0xb13cd65200000000, 0x6279aca600000000, 0xc4f3594c00000000,
    0x89e6b29a00000000, 0x13cd653600000000, 0x279aca6c00000000,
    0x4f3594d800000000, 0x9e6b29b000000000, 0x3cd6536200000000,
    0x79aca6c400000000, 0xf3594d8800000000, 0xe6b29b1200000000,
    0xcd65362600000000
  };
constexpr inline uint64_t
  LEVEL_3_AA_SEED_QKR__RIGHT_33BITS_ROLL_TABLE[33] = { // NOLINT
    0x1b721df0e, 0x16e43be1d, 0xdc877c3b,  0x1b90ef876, 0x1721df0ed,
    0xe43be1db,  0x1c877c3b6, 0x190ef876d, 0x121df0edb, 0x43be1db7,
    0x877c3b6e,  0x10ef876dc, 0x1df0edb9,  0x3be1db72,  0x77c3b6e4,
    0xef876dc8,  0x1df0edb90, 0x1be1db721, 0x17c3b6e43, 0xf876dc87,
    0x1f0edb90e, 0x1e1db721d, 0x1c3b6e43b, 0x1876dc877, 0x10edb90ef,
    0x1db721df,  0x3b6e43be,  0x76dc877c,  0xedb90ef8,  0x1db721df0,
    0x1b6e43be1, 0x16dc877c3, 0xdb90ef87
  };
constexpr inline uint64_t
  LEVEL_3_AA_SEED_QKR__LEFT_31BITS_ROLL_TABLE[31] = { // NOLINT
    0x917de47200000000, 0x22fbc8e600000000, 0x45f791cc00000000,
    0x8bef239800000000, 0x17de473200000000, 0x2fbc8e6400000000,
    0x5f791cc800000000, 0xbef2399000000000, 0x7de4732200000000,
    0xfbc8e64400000000, 0xf791cc8a00000000, 0xef23991600000000,
    0xde47322e00000000, 0xbc8e645e00000000, 0x791cc8be00000000,
    0xf239917c00000000, 0xe47322fa00000000, 0xc8e645f600000000,
    0x91cc8bee00000000, 0x239917de00000000, 0x47322fbc00000000,
    0x8e645f7800000000, 0x1cc8bef200000000, 0x39917de400000000,
    0x7322fbc800000000, 0xe645f79000000000, 0xcc8bef2200000000,
    0x9917de4600000000, 0x322fbc8e00000000, 0x645f791c00000000,
    0xc8bef23800000000
  };
constexpr inline uint64_t LEVEL_3_AA_SEED_VILM_RIGHT_33BITS_ROLL_TABLE[33] = {
  0xa07c5bd7,  0x140f8b7ae, 0x81f16f5d,  0x103e2deba, 0x7c5bd75,   0xf8b7aea,
  0x1f16f5d4,  0x3e2deba8,  0x7c5bd750,  0xf8b7aea0,  0x1f16f5d40, 0x1e2deba81,
  0x1c5bd7503, 0x18b7aea07, 0x116f5d40f, 0x2deba81f,  0x5bd7503e,  0xb7aea07c,
  0x16f5d40f8, 0xdeba81f1,  0x1bd7503e2, 0x17aea07c5, 0xf5d40f8b,  0x1eba81f16,
  0x1d7503e2d, 0x1aea07c5b, 0x15d40f8b7, 0xba81f16f,  0x17503e2de, 0xea07c5bd,
  0x1d40f8b7a, 0x1a81f16f5, 0x1503e2deb
};
constexpr inline uint64_t LEVEL_3_AA_SEED_VILM_LEFT_31BITS_ROLL_TABLE[31] = {
  0x37cdd84a00000000, 0x6f9bb09400000000, 0xdf37612800000000,
  0xbe6ec25200000000, 0x7cdd84a600000000, 0xf9bb094c00000000,
  0xf376129a00000000, 0xe6ec253600000000, 0xcdd84a6e00000000,
  0x9bb094de00000000, 0x376129be00000000, 0x6ec2537c00000000,
  0xdd84a6f800000000, 0xbb094df200000000, 0x76129be600000000,
  0xec2537cc00000000, 0xd84a6f9a00000000, 0xb094df3600000000,
  0x6129be6e00000000, 0xc2537cdc00000000, 0x84a6f9ba00000000,
  0x94df37600000000,  0x129be6ec00000000, 0x2537cdd800000000,
  0x4a6f9bb000000000, 0x94df376000000000, 0x29be6ec200000000,
  0x537cdd8400000000, 0xa6f9bb0800000000, 0x4df3761200000000,
  0x9be6ec2400000000
};
constexpr inline uint64_t
  LEVEL_3_AA_SEED_WFY__RIGHT_33BITS_ROLL_TABLE[33] = { // NOLINT
    0x11a67a896, 0x34cf512d,  0x699ea25a,  0xd33d44b4,  0x1a67a8968,
    0x14cf512d1, 0x99ea25a3,  0x133d44b46, 0x67a8968d,  0xcf512d1a,
    0x19ea25a34, 0x13d44b469, 0x7a8968d3,  0xf512d1a6,  0x1ea25a34c,
    0x1d44b4699, 0x1a8968d33, 0x1512d1a67, 0xa25a34cf,  0x144b4699e,
    0x8968d33d,  0x112d1a67a, 0x25a34cf5,  0x4b4699ea,  0x968d33d4,
    0x12d1a67a8, 0x5a34cf51,  0xb4699ea2,  0x168d33d44, 0xd1a67a89,
    0x1a34cf512, 0x14699ea25, 0x8d33d44b
  };
constexpr inline uint64_t
  LEVEL_3_AA_SEED_WFY__LEFT_31BITS_ROLL_TABLE[31] = { // NOLINT
    0x51a7955e00000000, 0xa34f2abc00000000, 0x469e557a00000000,
    0x8d3caaf400000000, 0x1a7955ea00000000, 0x34f2abd400000000,
    0x69e557a800000000, 0xd3caaf5000000000, 0xa7955ea200000000,
    0x4f2abd4600000000, 0x9e557a8c00000000, 0x3caaf51a00000000,
    0x7955ea3400000000, 0xf2abd46800000000, 0xe557a8d200000000,
    0xcaaf51a600000000, 0x955ea34e00000000, 0x2abd469e00000000,
    0x557a8d3c00000000, 0xaaf51a7800000000, 0x55ea34f200000000,
    0xabd469e400000000, 0x57a8d3ca00000000, 0xaf51a79400000000,
    0x5ea34f2a00000000, 0xbd469e5400000000, 0x7a8d3caa00000000,
    0xf51a795400000000, 0xea34f2aa00000000, 0xd469e55600000000,
    0xa8d3caae00000000
  };
constexpr inline uint64_t
  LEVEL_3_AA_SEED_H____RIGHT_33BITS_ROLL_TABLE[33] = { // NOLINT
    0x93708fbf,  0x126e11f7e, 0x4dc23efd,  0x9b847dfa,  0x13708fbf4,
    0x6e11f7e9,  0xdc23efd2,  0x1b847dfa4, 0x1708fbf49, 0xe11f7e93,
    0x1c23efd26, 0x1847dfa4d, 0x108fbf49b, 0x11f7e937,  0x23efd26e,
    0x47dfa4dc,  0x8fbf49b8,  0x11f7e9370, 0x3efd26e1,  0x7dfa4dc2,
    0xfbf49b84,  0x1f7e93708, 0x1efd26e11, 0x1dfa4dc23, 0x1bf49b847,
    0x17e93708f, 0xfd26e11f,  0x1fa4dc23e, 0x1f49b847d, 0x1e93708fb,
    0x1d26e11f7, 0x1a4dc23ef, 0x149b847df
  };
constexpr inline uint64_t
  LEVEL_3_AA_SEED_H____LEFT_31BITS_ROLL_TABLE[31] = { // NOLINT
    0x1d2a0ba400000000, 0x3a54174800000000, 0x74a82e9000000000,
    0xe9505d2000000000, 0xd2a0ba4200000000, 0xa541748600000000,
    0x4a82e90e00000000, 0x9505d21c00000000, 0x2a0ba43a00000000,
    0x5417487400000000, 0xa82e90e800000000, 0x505d21d200000000,
    0xa0ba43a400000000, 0x4174874a00000000, 0x82e90e9400000000,
    0x5d21d2a00000000,  0xba43a5400000000,  0x174874a800000000,
    0x2e90e95000000000, 0x5d21d2a000000000, 0xba43a54000000000,
    0x74874a8200000000, 0xe90e950400000000, 0xd21d2a0a00000000,
    0xa43a541600000000, 0x4874a82e00000000, 0x90e9505c00000000,
    0x21d2a0ba00000000, 0x43a5417400000000, 0x874a82e800000000,
    0xe9505d200000000
  };
constexpr inline uint64_t
  LEVEL_3_AA_SEED_P____RIGHT_33BITS_ROLL_TABLE[33] = { // NOLINT
    0x16611245,  0x2cc2248a,  0x59844914,  0xb3089228, 0x166112450, 0xcc2248a1,
    0x198449142, 0x130892285, 0x6112450b,  0xc2248a16, 0x18449142c, 0x108922859,
    0x112450b3,  0x2248a166,  0x449142cc,  0x89228598, 0x112450b30, 0x248a1661,
    0x49142cc2,  0x92285984,  0x12450b308, 0x48a16611, 0x9142cc22,  0x122859844,
    0x450b3089,  0x8a166112,  0x1142cc224, 0x28598449, 0x50b30892,  0xa1661124,
    0x142cc2248, 0x85984491,  0x10b308922
  };
constexpr inline uint64_t
  LEVEL_3_AA_SEED_P____LEFT_31BITS_ROLL_TABLE[31] = { // NOLINT
    0xfe4c47da00000000, 0xfc988fb600000000, 0xf9311f6e00000000,
    0xf2623ede00000000, 0xe4c47dbe00000000, 0xc988fb7e00000000,
    0x9311f6fe00000000, 0x2623edfe00000000, 0x4c47dbfc00000000,
    0x988fb7f800000000, 0x311f6ff200000000, 0x623edfe400000000,
    0xc47dbfc800000000, 0x88fb7f9200000000, 0x11f6ff2600000000,
    0x23edfe4c00000000, 0x47dbfc9800000000, 0x8fb7f93000000000,
    0x1f6ff26200000000, 0x3edfe4c400000000, 0x7dbfc98800000000,
    0xfb7f931000000000, 0xf6ff262200000000, 0xedfe4c4600000000,
    0xdbfc988e00000000, 0xb7f9311e00000000, 0x6ff2623e00000000,
    0xdfe4c47c00000000, 0xbfc988fa00000000, 0x7f9311f600000000,
    0xff2623ec00000000
  };

constexpr inline const uint64_t*
  LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE = // NOLINT
  AA_SEED___RIGHT_33BITS_ROLL_TABLE;
constexpr inline const uint64_t*
  LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE = // NOLINT
  AA_SEED___LEFT_31BITS_ROLL_TABLE;

constexpr inline const uint64_t*
  LEVEL_3_AA_SEED_RIGHT_33BITS_ROLL_TABLE[ASCII_SIZE] = {
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 0..7
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 8..15
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 16..23
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 24..31
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 32..39
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 40..47
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 48..55
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 56..63
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_ATS__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_C____RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_NDE__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_NDE__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_WFY__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_G____RIGHT_33BITS_ROLL_TABLE, // 64..71
    LEVEL_3_AA_SEED_H____RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_VILM_RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_QKR__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_VILM_RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_VILM_RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_NDE__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 72..79
    LEVEL_3_AA_SEED_P____RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_QKR__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_QKR__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_ATS__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_ATS__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_VILM_RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_WFY__RIGHT_33BITS_ROLL_TABLE, // 80..87
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_WFY__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 88..95
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_ATS__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_C____RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_NDE__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_NDE__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_WFY__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_G____RIGHT_33BITS_ROLL_TABLE, // 96..103
    LEVEL_3_AA_SEED_H____RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_VILM_RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_QKR__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_VILM_RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_VILM_RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_NDE__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 104..111
    LEVEL_3_AA_SEED_P____RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_QKR__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_QKR__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_ATS__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_ATS__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_VILM_RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_WFY__RIGHT_33BITS_ROLL_TABLE, // 112..119
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_WFY__RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 120..127
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 128..135
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 136..143
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 144..151
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 152..159
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 160..167
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 168..175
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 176..183
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 184..191
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 192..199
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 200..207
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 208..215
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 216..223
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 224..231
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 232..239
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE, // 240..247
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______RIGHT_33BITS_ROLL_TABLE // 248..255
  };

constexpr inline const uint64_t*
  LEVEL_3_AA_SEED_LEFT_31BITS_ROLL_TABLE[ASCII_SIZE] = {
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 0..7
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 8..15
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 16..23
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 24..31
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 32..39
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 40..47
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 48..55
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 56..63
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_ATS__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_C____LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_NDE__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_NDE__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_WFY__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_G____LEFT_31BITS_ROLL_TABLE, // 64..71
    LEVEL_3_AA_SEED_H____LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_VILM_LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_QKR__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_VILM_LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_VILM_LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_NDE__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 72..79
    LEVEL_3_AA_SEED_P____LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_QKR__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_QKR__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_ATS__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_ATS__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_VILM_LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_WFY__LEFT_31BITS_ROLL_TABLE, // 80..87
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_WFY__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 88..95
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_ATS__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_C____LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_NDE__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_NDE__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_WFY__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_G____LEFT_31BITS_ROLL_TABLE, // 96..103
    LEVEL_3_AA_SEED_H____LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_VILM_LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_QKR__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_VILM_LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_VILM_LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_NDE__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 104..111
    LEVEL_3_AA_SEED_P____LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_QKR__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_QKR__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_ATS__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_ATS__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_VILM_LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_WFY__LEFT_31BITS_ROLL_TABLE, // 112..119
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_WFY__LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 120..127
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 128..135
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 136..143
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 144..151
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 152..159
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 160..167
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 168..175
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 176..183
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 184..191
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 192..199
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 200..207
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 208..215
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 216..223
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 224..231
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 232..239
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE, // 240..247
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED______LEFT_31BITS_ROLL_TABLE // 248..255
  };

constexpr inline const uint64_t* const LEVEL_X_AA_SEED_TABLE[4] = {
  nullptr,
  AA_SEED_TABLE,
  LEVEL_2_AA_SEED_TABLE,
  LEVEL_3_AA_SEED_TABLE
};
constexpr inline const uint64_t* const*
  LEVEL_X_AA_SEED_RIGHT_33BITS_ROLL_TABLE[4] = {
    nullptr,
    AA_SEED_RIGHT_33BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_RIGHT_33BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_RIGHT_33BITS_ROLL_TABLE
  };
constexpr inline const uint64_t* const*
  LEVEL_X_AA_SEED_LEFT_31BITS_ROLL_TABLE[4] = {
    nullptr,
    AA_SEED_LEFT_31BITS_ROLL_TABLE,
    LEVEL_2_AA_SEED_LEFT_31BITS_ROLL_TABLE,
    LEVEL_3_AA_SEED_LEFT_31BITS_ROLL_TABLE
  };

} // namespace btllib::hashing_internals