/* 
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package javax.sql;

import java.util.EventListener;

/**
 * An interface used to receive events generated by a
 * <code>PooledConnection</code>.
 * <p>
 * This interface would typically be implemented by a component which implements
 * Connection Pooling (a Connection Pool Manager). A Connection will signal
 * events to a ConnectionEventListener either when the application closes a
 * Connection it has been using or when a significant error occurs while the
 * Connection is being used, where the Connection should not be used again.
 * <p>
 * The Connection Pool Manager can return closed Connections to the Pool for
 * later reuse. Connections experiencing an error should be discarded.
 * 
 */
public interface ConnectionEventListener extends EventListener {

    /**
     * Notifies the ConnectionEventListener that an application has called the
     * <code>close</code> method on a pooled Connection.
     * 
     * @param theEvent
     *            a ConnectionEvent containing detail about the source of the
     *            event.
     */
    public void connectionClosed(ConnectionEvent theEvent);

    /**
     * Notifies the ConnectionEventListener that an error has occurred while a
     * PooledConnection was being used and that the PooledConnection can no
     * longer be used for work. This notification is done just before the
     * SQLException passed in the event message is thrown to the application.
     * 
     * @param theEvent
     *            a ConnectionEvent containing detail about the source of the
     *            event and the SQLException that has occurred.
     */
    public void connectionErrorOccurred(ConnectionEvent theEvent);
}
