/*******************************************************************
 * libfaxophone                                                    *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file faxophone/faxophone.c
 * \brief CAPI routines and main faxophone functions
 */

#include <faxophone/faxophone.h>
#include <faxophone/fax.h>
#include <faxophone/phone.h>
#include <faxophone/isdn-convert.h>

/** The current active session */
static struct sSession *psSession = NULL;
/** Unique connection id */
static unsigned int nId = 0;
/** Faxophone debug level */
unsigned char nFaxophoneLogLevel = FOP_DEBUG;

/**
 * \brief Dump capi error (UNUSED)
 * \param nError capi error number
 */
static void capi_error( long nError ) {
	if ( nError != 0 ) {
		FopDebug( FOP_DEBUG, " -> Error: 0x%X\n", nError );
		if ( nError == 0x3301 ) {
			FopDebug( FOP_INFO, "Protocol Error Layer 1\n" );
		}
		if ( psSession ) {
			psSession -> psHandlers -> Status( NULL, nError );
		}
	}
}

/**
 * \brief Set connection type, transfer and cleanup routine, b3 informations
 * \param psConnection capi connection
 * \param nType connection type
 * \return error code: 0 on success, otherwise error
 */
static int connectionSetType( struct sCapiConnection *psConnection, int nType ) {
	int nResult = 0;

	/* Set type */
	psConnection -> nType = nType;

	/* Set informations depending on type */
	switch ( nType ) {
		case SESSION_PHONE:
			psConnection -> Data = phoneTransfer;
			psConnection -> Clean = NULL;
			psConnection -> nEarlyB3 = 1;
			break;
		case SESSION_FAX:
			psConnection -> Data = faxTransfer;
			psConnection -> Clean = faxClean;
			psConnection -> nEarlyB3 = 0;
			break;
		default:
			FopDebug( FOP_DEBUG, "Unhandled session type!!\n" );
			nResult = -1;
			break;
	}

	return nResult;
}

/**
 * \brief Return free capi connection index
 * \return free connection index or -1 on error
 */
struct sCapiConnection *capiGetFreeConnection( void ) {
	int nI;

	for ( nI = 0; nI < CAPI_CONNECTIONS; nI++ ) {
		if ( psSession -> asConnection[ nI ].nPlci == 0 && psSession -> asConnection[ nI ].nNcci == 0 ) {
			psSession -> asConnection[ nI ].nId = nId++;
			psSession -> asConnection[ nI ].nState = STATE_IDLE;
			return &psSession -> asConnection[ nI ];
		}
	}

	return NULL;
}

/**
 * \brief Free capi connection
 * \param psConnection capi connection
 * \return error code
 */
static int capiSetFree( struct sCapiConnection *psConnection ) {
	/* reset connection */
	if ( psConnection -> pPrivate != NULL ) {
		if ( psConnection -> Clean ) {
			psConnection -> Clean( psConnection );
		} else {
			FopDebug( FOP_DEBUG, "Warning: Private data but no clean function\n" );
		}
	}

	memset( psConnection, 0, sizeof( struct sCapiConnection ) );

	return 0;
}

/**
 * \brief Terminate selected connection
 * \param psConnection connection we want to terminate
 */
void capiHangup( struct sCapiConnection *psConnection ) {
	_cmsg sCmsg1;
	unsigned nInfo = 0;

	if ( psConnection == NULL ) {
		return;
	}

	switch ( psConnection -> nState ) {
		case STATE_CONNECT_WAIT:
		case STATE_CONNECT_ACTIVE:
		case STATE_DISCONNECT_B3_REQ:
		case STATE_DISCONNECT_B3_WAIT:
		case STATE_INCOMING_WAIT:
			FopDebug( FOP_DEBUG, "REQ: DISCONNECT - nPlci %d\n", psConnection -> nPlci );

			isdnLock();
			nInfo = DISCONNECT_REQ( &sCmsg1, psSession -> nApplId, 1, psConnection -> nPlci, NULL, NULL, NULL, NULL );
			isdnUnlock();

			if ( nInfo != 0 ) {
				psConnection -> nState = STATE_IDLE;
				psSession -> psHandlers -> Status( psConnection, nInfo );
			} else {
				psConnection -> nState = STATE_DISCONNECT_ACTIVE;
			}
			break;
		case STATE_CONNECT_B3_WAIT:
		case STATE_CONNECTED:
			FopDebug( FOP_DEBUG, "REQ: DISCONNECT_B3 - nNcci %d\n", psConnection -> nNcci );

			isdnLock();
			nInfo = DISCONNECT_B3_REQ( &sCmsg1, psSession -> nApplId, 1, psConnection -> nNcci, NULL );
			isdnUnlock();

			if ( nInfo != 0 ) {
				/* retry with disconnect on whole connection */
				isdnLock();
				nInfo = DISCONNECT_REQ( &sCmsg1, psSession -> nApplId, 1, psConnection -> nPlci, NULL, NULL, NULL, NULL );
				isdnUnlock();
				if ( nInfo != 0 ) {
					psConnection -> nState = STATE_IDLE;
					psSession -> psHandlers -> Status( psConnection, nInfo );
				} else {
					psConnection -> nState = STATE_DISCONNECT_ACTIVE;
				}
			} else {
				psConnection -> nState = STATE_DISCONNECT_B3_REQ;
			}
			break;
		case STATE_RINGING:
			/* reject the call */
			FopDebug( FOP_DEBUG, "RESP: CONNECT - nPlci %d\n", psConnection -> nPlci );

			isdnLock();
			nInfo = CONNECT_RESP( &sCmsg1, psSession -> nApplId, psSession -> nMessageNumber++, psConnection -> nPlci, 3, 0, 0, 0, 0, 0, 0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL );
			isdnUnlock();
			psConnection -> nState = STATE_IDLE;
			if ( nInfo != 0 ) {
				psSession -> psHandlers -> Status( psConnection, nInfo );
			}

			break;
		case STATE_IDLE:
			break;
		default:
			FopDebug( FOP_DEBUG, "Unexpected state 0x%x on disconnect\n", psConnection -> nState );
			break;
	}
}

/**
 * \brief Call number from target using CIP value
 * \param nController controller id
 * \param pnSrcNo source number
 * \param pnTrgNo target number
 * \param nCallAnonymous call anonymous flag
 * \param nType connection type
 * \param nCip caller id
 * \return error code
 */
struct sCapiConnection *capiCall( unsigned nController, const char *pnSrcNo, const char *pnTrgNo, unsigned nCallAnonymous, unsigned nType, unsigned nCip ) {
	_cmsg sCmsg;
	unsigned char anCalledPartyNumber[ 70 ];
	unsigned char anCallingPartyNumber[ 70 ];
	unsigned char anBc[ 4 ];
	unsigned char anLlc[ 3 ];
	unsigned char anHlc[ 3 ];
	struct sCapiConnection *psConnection = NULL;
	int nErr = 0;
	int nIntern = ( pnTrgNo[ 0 ] == '*' ) || ( pnTrgNo[ 0 ] == '#' );

	if ( pnSrcNo == NULL || strlen( pnSrcNo ) < 1 || pnTrgNo == NULL || strlen( pnTrgNo ) < 1 ) {
		FopDebug( FOP_DEBUG, "Wrong phone numbers!\n" );
		return psConnection;
	}

	/* Say hello */
	FopDebug( FOP_DEBUG, "REQ: CONNECT (%s -> %s)\n", pnSrcNo, pnTrgNo );

	/* get free connection */
	psConnection = capiGetFreeConnection();
	if ( psConnection == NULL ) {
		return psConnection;
	}

	/* set connection type */
	connectionSetType( psConnection, nType );

	/* TargetNo */
	anCalledPartyNumber[ 0 ] = 1 + strlen( pnTrgNo );
	anCalledPartyNumber[ 1 ] = 0x80;
	strncpy( ( char * ) &anCalledPartyNumber[ 2 ], pnTrgNo, sizeof( anCalledPartyNumber ) - 3 );

	/* MSN */
	anCallingPartyNumber[ 1 ] = 0x00;
	anCallingPartyNumber[ 2 ] = 0x80;

	if ( nCallAnonymous ) {
		anCallingPartyNumber[ 2 ] = 0xA0;
	}

	if ( nIntern ) {
		anCallingPartyNumber[ 0 ] = 2 + 5;
		strncpy( ( char * ) &anCallingPartyNumber[ 3 ], "**981", sizeof( anCallingPartyNumber ) - 4 );

		strncpy( ( char * ) anBc, "\x03\xE0\x90\xA3", sizeof( anBc ) );
	} else {
		anCallingPartyNumber[ 0 ] = 2 + strlen( pnSrcNo );
		strncpy( ( char * ) &anCallingPartyNumber[ 3 ], pnSrcNo, sizeof( anCallingPartyNumber ) - 4 );

		memset( anBc, 0, sizeof( anBc ) );
	}
	strncpy( ( char * ) anLlc, "\x02\x80\x90", sizeof( anLlc ) );

	if ( nCip == 0x04 ) {
		strncpy( ( char * ) anHlc, "\x02\x91\x81", sizeof( anHlc ) );
	} else if ( nCip == 0x11 ) {
		//strncpy( ( char * ) anHlc, "\x02\x91\x84", sizeof( anHlc ) );
		//strncpy( ( char * ) anBc, "\x03\x90\x90\xA3", sizeof( anBc ) );
		memset( anBc, 0, sizeof( anBc ) );
		memset( anLlc, 0, sizeof( anLlc ) );
		memset( anHlc, 0, sizeof( anHlc ) );
	}

	/* Request connect */
	isdnLock();
	nErr = CONNECT_REQ(
		&sCmsg,
		psSession -> nApplId,
		0,
		nController,
		nCip,
		( unsigned char * ) anCalledPartyNumber,
		( unsigned char * ) anCallingPartyNumber,
		( unsigned char * ) "",
		( unsigned char * ) "", 
		1,
		1,
		0,
		( unsigned char * ) "",
		( unsigned char * ) "",
		( unsigned char * ) "",
		( unsigned char * ) "",
		( unsigned char * ) anBc,
		( unsigned char * ) anLlc,
		( unsigned char * ) anHlc,
		( unsigned char * ) "",
		( unsigned char * ) "",
		( unsigned char * ) "",
		( unsigned char * ) "",
		( _cstruct ) "" );
	isdnUnlock();

	/* Error? */
	if ( nErr ) {
		FopDebug( FOP_DEBUG, "(%d) Unable to send CONNECT_REQ!\n", nErr );
		capi_error( nErr );
		capiSetFree( psConnection );
		psConnection = NULL;
		return psConnection;
	}

	psConnection -> pnTarget = strdup( pnTrgNo );
	psConnection -> pnSource = strdup( pnSrcNo );

	return psConnection;
}

/**
 * \brief Pickup an incoming call
 * \param psConnection incoming capi connection
 * \param nType handle connection as this type
 * \return error code: 0 on success, otherwise error
 */
int capiPickup( struct sCapiConnection *psConnection, int nType ) {
	_cmsg sMessage;
	unsigned char anLocalNum[ 4 ];
	struct sSession *psSession = faxophoneGetSession();

	connectionSetType( psConnection, nType );

	if ( psConnection -> nState != STATE_RINGING ) {
		FopDebug( FOP_DEBUG, "CAPI Pickup called, even if not ringing\n" );
		return -1;
	} else {
		anLocalNum[ 0 ] = 0x00;
		anLocalNum[ 1 ] = 0x00;
		anLocalNum[ 2 ] = 0x00;
		anLocalNum[ 3 ] = 0x00;

		isdnLock();
		FopDebug( FOP_DEBUG, "RESP: CAPI_CONNECT_RESP - nPlci %d\n", psConnection -> nPlci );
		CONNECT_RESP( &sMessage, psSession -> nApplId, psSession -> nMessageNumber++, psConnection -> nPlci, 0, 1, 1, 0, 0, 0, 0, &anLocalNum[ 0 ], NULL, NULL, NULL, NULL, NULL, NULL, NULL );
		isdnUnlock();

		/* connection initiated, wait for CONNECT_ACTIVE_IND */
		psConnection -> nState = STATE_INCOMING_WAIT;
	}

	return 0;
}

/**
 * \brief Get the calling party number on CAPI_CONNECT
 * \param psCmsg CAPI message
 * \param anNumber buffer to store number
 */
static void capiGetSourceNo(_cmsg *psCmsg, char anNumber[ 256 ] ) {
	unsigned char *pnX = CONNECT_IND_CALLINGPARTYNUMBER( psCmsg );
	unsigned int nLen = 0;

	memset( anNumber, 0, 256 );

	if ( pnX == NULL ) {
		pnX = INFO_IND_INFOELEMENT( psCmsg );

		if ( pnX != NULL ) {
			nLen = ( int ) pnX[ 0 ];
		}
	} else {
		nLen = *CONNECT_IND_CALLINGPARTYNUMBER( psCmsg );
	}

	if ( nLen <= 1 ) {
		strcpy( anNumber, "unknown" );
	} else {
		if ( nLen > 256 ) {
			nLen = 256 - 1;
		}

		/*switch ( pnX[ 1 ] & 112 ) {
			case 32:
				strcat( anNumber, getLineAccessCode() );
				break;
			case 64:
				strcat( anNumber, getLineAccessCode() );
				strcat( anNumber, getAreaCode() );
				break;
		}*/

		/* get number */
		if ( pnX[ 2 ] & 128 ) {
			anNumber[ strlen( anNumber ) + pnX[ 0 ] - 1 ] = 0;
			anNumber[ strlen( anNumber ) + pnX[ 0 ] - 2 ] = 0;

			memcpy( anNumber + strlen( anNumber ), pnX + 3, ( size_t )( pnX[ 0 ] - 2 ) );
		} else {
			anNumber[ strlen( anNumber ) + pnX[ 0 ] ] = 0;
			anNumber[ strlen( anNumber ) + pnX[ 0 ] - 1 ] = 0;
			memcpy( anNumber + strlen( anNumber ), pnX + 2, ( size_t )( pnX[ 0 ] - 1 ) );
		}
	}

	if ( !strlen( anNumber ) ) {
		strcpy( anNumber, "anonymous" );
	}
}

/**
 * \brief Get the called party anNumber on CAPI_CONNECT
 * \param psCmsg CAPI message
 * \param anNumber buffer to store number
 */
static void capiGetTargetNo( _cmsg *psCmsg, char anNumber[ 256 ] ) {
	unsigned char *pnX = CONNECT_IND_CALLEDPARTYNUMBER( psCmsg );
	unsigned int nLen = 0;

	memset( anNumber, 0, 256 );

	if ( pnX == NULL ) {
		pnX = INFO_IND_INFOELEMENT( psCmsg );
		if ( pnX != NULL ) {
			nLen = ( int ) pnX[ 0 ];
		}
	} else {
		nLen = *CONNECT_IND_CALLEDPARTYNUMBER( psCmsg );

		if ( CONNECT_IND_CALLEDPARTYNUMBER( psCmsg )[ 0 ] == 0 ) {
			nLen = 0;
		}
	}

	if ( nLen <= 1 ) {
		strcpy( anNumber, "unknown" );
	} else {
		if ( nLen > 256 ) {
			nLen = 256 - 1;
		}

		/* get number */
		/*if ( strncmp( ( char * ) pnX + 2, getCountryCode(), 2 ) == 0 ) {
			anNumber[ strlen( anNumber ) + ( size_t ) pnX[ 0 ] ] = 0;
			anNumber[ strlen( anNumber ) + ( size_t ) pnX[ 0 ] - 1 ] = 0;
			strcpy( anNumber, "0" );
			memcpy( anNumber + 1, pnX + 2 + 2, nLen - 3 );
		} else*/ {
			anNumber[ strlen( anNumber ) + ( size_t ) pnX[ 0 ] ] = 0;
			anNumber[ strlen( anNumber ) + ( size_t ) pnX[ 0 ] - 1 ] = 0;
			memcpy( anNumber + strlen( anNumber ), pnX + 2, ( size_t )( pnX[ 0 ] - 1 ) );
		}
	}

	if ( !strlen( anNumber ) ) {
		strcpy( anNumber, "anonymous" );
	}
}

/**
 * \brief Find capi connection by PLCI
 * \param nPlci plci
 * \return capi connection or NULL on error
 */
static struct sCapiConnection *capiFindPlci( int nPlci ) {
	int nIndex;

	for ( nIndex = 0; nIndex < CAPI_CONNECTIONS; nIndex++ ) {
		if ( psSession -> asConnection[ nIndex ].nPlci == nPlci ) {
			return &psSession -> asConnection[ nIndex ];
		}
	}

	return NULL;
}

/**
 * \brief Find newly created capi connection
 * \return capi connection or NULL on error
 */
static struct sCapiConnection *capiFindNew( void ) {
	int nIndex;

	for ( nIndex = 0; nIndex < CAPI_CONNECTIONS; nIndex++ ) {
		if ( psSession -> asConnection[ nIndex ].nPlci == 0 && psSession -> asConnection[ nIndex ].nType != 0 ) {
			return &psSession -> asConnection[ nIndex ];
		}
	}

	return NULL;
}

/**
 * \brief Find capi connection by NCCI
 * \param nNcci ncci
 * \return capi connection or NULL on error
 */
static struct sCapiConnection *capiFindNcci( int nNcci ) {
	int nIndex;

	for ( nIndex = 0; nIndex < CAPI_CONNECTIONS; nIndex++ ) {
		if ( psSession -> asConnection[ nIndex ].nNcci == nNcci ) {
			return &psSession -> asConnection[ nIndex ];
		}
	}

	return NULL;
}

/**
 * \brief Close capi
 * \return error code
 */
static int capiClose( void ) {
	int nIndex;

	if ( psSession != NULL && psSession -> nApplId != -1 ) {
		for ( nIndex = 0; nIndex < CAPI_CONNECTIONS; nIndex++ ) {
			if ( psSession -> asConnection[ nIndex ].nPlci != 0 || psSession -> asConnection[ nIndex ].nNcci != 0 ) {
				FopDebug( FOP_DEBUG, "Hangup %d\n", nIndex );
				capiHangup( &psSession -> asConnection[ nIndex ] );
				FopDebug( FOP_DEBUG, "Hangup %d done\n", nIndex );
				g_usleep( 25 );
			}
		}

		FopDebug( FOP_DEBUG, "Release\n" );
		CAPI20_RELEASE( psSession -> nApplId );
		FopDebug( FOP_DEBUG, "Release done\n" );
		psSession -> nApplId = -1;
	}

	return 0;
}

/**
 * \brief CAPI respond connection
 * \param nPlci plci
 * \param nIgnore ignore connection
 */
static void capiRespConnection( int nPlci, unsigned int nIgnore ) {
	_cmsg sCmsg1;

	if ( !nIgnore ) {
		/* *Ring* */
		FopDebug( FOP_DEBUG, "REQ: ALERT - nPlci %d\n", nPlci );
		isdnLock();
		ALERT_REQ( &sCmsg1, psSession -> nApplId, 0, nPlci, NULL, NULL, NULL, NULL, NULL );
		isdnUnlock();
	} else {
		/* ignore */
		isdnLock();
		CONNECT_RESP( &sCmsg1, psSession -> nApplId, psSession -> nMessageNumber++, nPlci, nIgnore, 1, 1, 0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL );
		isdnUnlock();
	}
}

/**
 * \brief Enable DTMF support.
 * \param psIsdn isdn device structure.
 * \param nNcci NCCI
 */
static void capiEnableDtmf( struct sCapiConnection *psConnection ) {
	_cmsg sMessage;
	_cbyte anFacility[ 11 ];

	/* Message length */
	anFacility[ 0 ] = 10;
	/* DTMF ON: 0x01, DTMF OFF: 0x02 */
	anFacility[ 1 ] = ( _cbyte ) 0x01;
	/* NULL */
	anFacility[ 2 ] = 0x00;
	/* DTMF Duration */
	anFacility[ 3 ] = 0x40;
	/* NULL */
	anFacility[ 4 ] = 0x00;
	/* DTMF Duration */
	anFacility[ 5 ] = 0x40;
	/* NULL */
	anFacility[ 6 ] = 0x00;
	/* NULL */
	anFacility[ 7 ] = 0x00;
	/* 2 */
	anFacility[ 8 ] = 0x02;
	/* NULL */
	anFacility[ 9 ] = 0x00;
	/* NULL */
	anFacility[ 10 ] = 0x00;

	FopDebug( FOP_DEBUG, "Enable DTMF for PLCI %d\n", psConnection -> nPlci );

	/* 0x01 = DTMF selector */
	isdnLock();
	FACILITY_REQ( &sMessage, psSession -> nApplId, 0/*psIsdn -> nMessageNumber++*/, psConnection -> nPlci, 0x01, ( unsigned char * ) anFacility );
	isdnUnlock();
}

/**
 * \brief Signal DTMF code to application
 * \param psConnection active capi connection
 * \param nDtmf DTMF code
 */
static void capiGetDtmfCode( struct sCapiConnection *psConnection, unsigned char nDtmf ) {
	if ( nDtmf == 0 ) {
		return;
	}

	if ( !isdigit( nDtmf ) ) {
		if ( nDtmf != '#' && nDtmf != '*' ) {
			return;
		}
	}

	psSession -> psHandlers -> Code( psConnection, nDtmf );
}

/**
 * \brief Send DTMF to remote
 * \param psConnection active capi connection
 * \param nDtmf DTMF code we want to send
 */
void capiSendDtmfCode( struct sCapiConnection *psConnection, unsigned char nDtmf ) {
	_cmsg sMessage;
	_cbyte anFacility[ 32 ];

	FopDebug( FOP_DEBUG, "nDtmf: %c\n", nDtmf );

	/* Message length */
	anFacility[ 0 ] = 0x08;
	/* Send DTMF 0x03 */
	anFacility[ 1 ] = ( _cbyte ) 0x03;
	/* NULL */
	anFacility[ 2 ] = 0x00;
	/* DTMF Duration */
	anFacility[ 3 ] = 0x30;
	/* NULL */
	anFacility[ 4 ] = 0x00;
	/* DTMF Duration */
	anFacility[ 5 ] = 0x30;
	/* NULL */
	anFacility[ 6 ] = 0x00;
	/* NULL */
	anFacility[ 7 ] = 0x01;
	/* NULL */
	anFacility[ 8 ] = nDtmf;

	FopDebug( FOP_DEBUG, "Sending DTMF code for NCCI %d\n", psConnection -> nNcci );

	/* 0x01 = DTMF selector */
	isdnLock();
	FACILITY_REQ( &sMessage, psSession -> nApplId, 0/*psIsdn -> nMessageNumber++*/, psConnection -> nNcci, 0x01, ( unsigned char * ) anFacility );
	isdnUnlock();
}

#if 0
/**
 * \brief Send display message to remote
 * \param psConnection active capi connection
 * \param pnDtmf text we want to send
 */
void capiSendDisplayMessage( struct sCapiConnection *psConnection, char *pnText ) {
	_cmsg sMessage;
	_cbyte anFacility[ 62 + 3 ];
	int nLen = 31;

	FopDebug( FOP_DEBUG, "Sending text: '%s'\n", pnText );
	memset( anFacility, 0, sizeof( anFacility ) );

	if ( strlen( pnText ) < 31 ) {
		nLen = strlen( pnText );
	}

	/* Message length */
	anFacility[ 0 ] = nLen + 2;
	/* Send DTMF 0x03 */
	anFacility[ 1 ] = ( _cbyte ) 0x28;

	strncpy( ( char * ) anFacility + 3, pnText, nLen );

	isdnLock();
	INFO_REQ( &sMessage, psSession -> nApplId, 0/*psIsdn -> nMessageNumber++*/, psConnection -> nPlci, ( unsigned char * ) "", ( unsigned char * ) "", ( unsigned char * ) "", ( unsigned char * ) "", ( unsigned char * ) anFacility, NULL );
	//FACILITY_REQ( &sMessage, psSession -> nApplId, 0/*psIsdn -> nMessageNumber++*/, psConnection -> nPlci, 0x01, ( unsigned char * ) anFacility );
	isdnUnlock();
}
#endif

/**
 * \brief CAPI indication
 * \param sCapiMessage capi message structure
 * \return error code
 */
static int capiIndication( _cmsg sCapiMessage ) {
	_cmsg sCmsg1;
	int nPlci = -1;
	int nNcci = -1;
	char anSourcePhoneNumber[ 256 ];
	char anTargetPhoneNumber[ 256 ];
	int nCip = -1;
	struct sCapiConnection *psConnection = NULL;
	int nReject = 0;
	int nInfo;
	char anInfoElement[ 128 ];
	int nIndex;
	int nTmp;

	switch ( sCapiMessage.Command ) {
		case CAPI_CONNECT:
			/* CAPI_CONNECT - Connect indication when called from remote phone */
			nPlci = CONNECT_IND_PLCI( &sCapiMessage );
			nCip = CONNECT_IND_CIPVALUE( &sCapiMessage );

			capiGetSourceNo( &sCapiMessage, anSourcePhoneNumber );
			capiGetTargetNo( &sCapiMessage, anTargetPhoneNumber );

			FopDebug( FOP_DEBUG, "IND: CAPI_CONNECT - nPlci %d, source %s, target %s, nCip: %d\n", nPlci, anSourcePhoneNumber, anTargetPhoneNumber, nCip );

			nReject = 0;

			if ( nCip != 16 && nCip != 1 && nCip != 4 && nCip != 17 ) {
				/* not telephony nor fax, ignore */
				nReject = 1;
			}

			if ( nReject  ) {
				/* Ignore */
				FopDebug( FOP_DEBUG, "IND: CAPI_CONNECT - nPlci: %d, nNcci: %d - IGNORING (%s <- %s)\n", nPlci, 0, anTargetPhoneNumber, anSourcePhoneNumber );
				capiRespConnection( nPlci, 1 );
			} else {
				psConnection = capiGetFreeConnection();

				psConnection -> nType = SESSION_NONE;
				psConnection -> nState = STATE_RINGING;
				psConnection -> nPlci = nPlci;
				psConnection -> pnSource = g_strdup( anSourcePhoneNumber );
				psConnection -> pnTarget = g_strdup( anTargetPhoneNumber );

				capiRespConnection( nPlci, 0 );

			}
			FopDebug( FOP_DEBUG, "IND: CAPI_CONNECT done\n" );

			break;

		/* CAPI_CONNECT_ACTIVE - Active */
		case CAPI_CONNECT_ACTIVE:
			nPlci = CONNECT_ACTIVE_IND_PLCI( &sCapiMessage );

			FopDebug( FOP_DEBUG, "IND: CAPI_CONNECT_ACTIVE - nPlci %d\n", nPlci );

			FopDebug( FOP_DEBUG, "RESP: CAPI_CONNECT_ACTIVE - nPlci %d\n", nPlci );
			isdnLock();
			CONNECT_ACTIVE_RESP( &sCmsg1, psSession -> nApplId, psSession -> nMessageNumber++, nPlci );
			isdnUnlock();

			psConnection = capiFindPlci( nPlci );
			if ( psConnection == NULL ) {
				FopDebug( FOP_DEBUG, "Wrong PLCI 0x%x\n", nPlci );
				break;
			}
			FopDebug( FOP_DEBUG, "IND: CAPI_CONNECT_ACTIVE - nConnection: %d, nPlci: %d\n", psConnection -> nId, psConnection -> nPlci );

			/* Request B3 when sending... */
			if ( psConnection -> nState == STATE_INCOMING_WAIT ) {
				psConnection -> nConnectTime = time( NULL );

				if ( psConnection -> nType == SESSION_PHONE ) {
					psSession -> psHandlers -> AudioOpen();
				}
				psConnection -> nState = STATE_CONNECT_ACTIVE;
			} else if ( psConnection -> nEarlyB3 == 0 ) {
				FopDebug( FOP_DEBUG, "REQ: CONNECT_B3 - nplci %d\n", nPlci );
				isdnLock();
				nInfo = CONNECT_B3_REQ( &sCmsg1, psSession -> nApplId, 0, nPlci, 0 );
				isdnUnlock();

				if ( nInfo != 0 ) {
					psSession -> psHandlers -> Status( psConnection, nInfo );
					/* initiate hangup on PLCI */
					capiHangup( psConnection );
				} else {
					/* wait for CONNECT_B3, then announce result to application via callback */
					psConnection -> nConnectTime = time( NULL );

					if ( psConnection -> nType == SESSION_PHONE ) {
						psSession -> psHandlers -> AudioOpen();
					}
					psConnection -> nState = STATE_CONNECT_ACTIVE;
				}
			}

			break;

		/* CAPI_CONNECT_B3 - Data connect */
		case CAPI_CONNECT_B3:
			FopDebug( FOP_DEBUG, "IND: CAPI_CONNECT_B3\n" );
			nNcci = CONNECT_B3_IND_NCCI( &sCapiMessage );
			nPlci = nNcci & 0x0000ffff;

			psConnection = capiFindPlci( nPlci );
			if ( psConnection == NULL ) {
				break;
			}

			/* Answer the info message */
			isdnLock();
			CONNECT_B3_RESP( &sCmsg1, psSession -> nApplId, psSession -> nMessageNumber++, nNcci, 0, ( _cstruct ) NULL );
			isdnUnlock();

			if ( psConnection -> nState == STATE_CONNECT_ACTIVE ) {
				psConnection -> nNcci = nNcci;
				psConnection -> nState = STATE_CONNECT_B3_WAIT;
			} else {
				/* Wrong connection state for B3 connect, trigger disconnect */
				capiHangup( psConnection );
			}
			break;

		/* CAPI_CONNECT_B3_ACTIVE - Data active */
		case CAPI_CONNECT_B3_ACTIVE:
			FopDebug( FOP_DEBUG, "IND: CAPI_CONNECT_B3_ACTIVE\n" );
			nNcci = CONNECT_B3_ACTIVE_IND_NCCI( &sCapiMessage );
			nPlci = nNcci & 0x0000ffff;

			psConnection = capiFindPlci( nPlci );
			if ( psConnection == NULL ) {
				FopDebug( FOP_DEBUG, "Wrong NCCI, got 0x%x\n", nNcci );
				break;
			}
			psConnection -> nNcci = nNcci;

			isdnLock();
			CONNECT_B3_ACTIVE_RESP( &sCmsg1, psSession -> nApplId, psSession -> nMessageNumber++, nNcci );
			isdnUnlock();

			psConnection -> nState = STATE_CONNECTED;

			capiEnableDtmf( psConnection );

			/* notify application about successful call establishment */
			psSession -> psHandlers -> Connected( psConnection );
			break;

		/* CAPI_DATA_B3 - Data - receive/send */
		case CAPI_DATA_B3:
			FopDebug( FOP_TRANSFER, "IND: CAPI_DATA_B3\n" );
			nNcci = CONNECT_B3_IND_NCCI( &sCapiMessage );
			nPlci = nNcci & 0x0000ffff;

			psConnection = capiFindNcci( nNcci );
			if ( psConnection == NULL ) {
				break;
			}

			FopDebug( FOP_TRANSFER, "IND: CAPI_DATA_B3 - nConnection: %d, nPlci: %d, nNcci: %d\n", psConnection -> nId, psConnection -> nPlci, psConnection -> nNcci );
			if ( psConnection -> Data ) {
				psConnection -> Data( psConnection, sCapiMessage );
			}
			break;

		/* CAPI_FACILITY - Facility (DTMF) */
		case CAPI_FACILITY:
			FopDebug( FOP_DEBUG, "IND: CAPI_FACILITY\n" );
			nNcci = CONNECT_B3_IND_NCCI( &sCapiMessage );
			nPlci = nNcci & 0x0000ffff;

			isdnLock();
			FACILITY_RESP( &sCmsg1, psSession -> nApplId, psSession -> nMessageNumber++, nPlci, FACILITY_IND_FACILITYSELECTOR( &sCapiMessage ), FACILITY_IND_FACILITYINDICATIONPARAMETER( &sCapiMessage ) );
			isdnUnlock();

			psConnection = capiFindPlci( nPlci );
			if ( psConnection == NULL ) {
				break;
			}

			FopDebug( FOP_DEBUG, "IND: CAPI_FACILITY %d\n", FACILITY_IND_FACILITYSELECTOR( &sCapiMessage ) );
			switch ( FACILITY_IND_FACILITYSELECTOR( &sCapiMessage ) ) {
				case 0x0001:
					/* DTMF */
					capiGetDtmfCode( psConnection, ( unsigned char ) FACILITY_IND_FACILITYINDICATIONPARAMETER( &sCapiMessage )[ 1 ] );
					break;
				case 0x0003:
					/* Supplementary Services */
					nTmp = ( unsigned int )( ( ( unsigned int ) FACILITY_IND_FACILITYINDICATIONPARAMETER( &sCapiMessage )[ 1 ] ) | ( ( unsigned int ) FACILITY_IND_FACILITYINDICATIONPARAMETER( &sCapiMessage )[ 3 ] << 8 ) );

					FopDebug( FOP_DEBUG, "%x %x %x %x %x %x\n", FACILITY_IND_FACILITYINDICATIONPARAMETER( &sCapiMessage )[ 0 ],
						FACILITY_IND_FACILITYINDICATIONPARAMETER( &sCapiMessage )[ 1 ],
						FACILITY_IND_FACILITYINDICATIONPARAMETER( &sCapiMessage )[ 2 ],
						FACILITY_IND_FACILITYINDICATIONPARAMETER( &sCapiMessage )[ 3 ],
						FACILITY_IND_FACILITYINDICATIONPARAMETER( &sCapiMessage )[ 4 ],
						FACILITY_IND_FACILITYINDICATIONPARAMETER( &sCapiMessage )[ 5 ] );
					if ( nTmp == 0x0203 ) {
						/* Retrieve */
						FopDebug( FOP_DEBUG, "FACILITY: RETRIEVE\n" );
						isdnLock();
						nInfo = CONNECT_B3_REQ( &sCmsg1, psSession -> nApplId, 0, nPlci, 0 );
						isdnUnlock();

						if ( nInfo != 0 ) {
							psSession -> psHandlers -> Status( psConnection, nInfo );
							/* initiate hangup on PLCI */
							capiHangup( psConnection );
						} else {
							/* wait for CONNECT_B3, then announce result to application via callback */
							psConnection -> nState = STATE_CONNECT_ACTIVE;
						}
					} else if ( nTmp == 0x0202 ) {
						/* Hold */
						FopDebug( FOP_DEBUG, "FACILITY: HOLD\n" );
					} else {
						FopDebug( FOP_DEBUG, "FACILITY: Unknown %x\n", nTmp );
					}
					break;
				default:
					FopDebug( FOP_DEBUG, "Unhandled facility selector!!\n", FACILITY_IND_FACILITYSELECTOR( &sCapiMessage ) );
					break;
			}
			break;

		/* CAPI_INFO */
		case CAPI_INFO:
			nPlci = INFO_IND_PLCI( &sCapiMessage );
			nInfo = INFO_IND_INFONUMBER( &sCapiMessage );

			/* Respond to INFO */
			isdnLock();
			INFO_RESP( &sCmsg1, psSession -> nApplId, psSession -> nMessageNumber++, nPlci );
			isdnUnlock();

			memset( anInfoElement, 0, sizeof( anInfoElement ) );
			for ( nIndex = 0; nIndex < sizeof( anInfoElement ); nIndex++ ) {
				anInfoElement[ nIndex ] = INFO_IND_INFOELEMENT( &sCapiMessage )[ nIndex ];
			}

			switch ( nInfo ) {
				case 0x0008:
					/* Cause */
					FopDebug( FOP_DEBUG, "CAPI_INFO - CAUSE\n" );
					FopDebug( FOP_DEBUG, "Hangup cause: 0x%x\n", anInfoElement[ 2 ] & 0x7F );
					break;
				case 0x00014:
					/* Call state */
					FopDebug( FOP_DEBUG, "CAPI_INFO - CALL STATE (0x%02x)\n", anInfoElement[ 0 ] );
					break;
				case 0x0018:
					/* Channel identification */
					FopDebug( FOP_DEBUG, "CAPI_INFO - CHANNEL IDENTIFICATION (0x%02x)\n", anInfoElement[ 0 ] );
					break;
				case 0x001C:
					/* Facility Q.932 */
					FopDebug( FOP_DEBUG, "CAPI_INFO - FACILITY Q.932\n" );
					break;
				case 0x001E:
					/* Progress Indicator */
					FopDebug( FOP_DEBUG, "CAPI_INFO - PROGRESS INDICATOR (0x%02x)\n", anInfoElement[ 0 ] );
					if ( anInfoElement[ 0 ] < 2 ) {
						FopDebug( FOP_DEBUG, "CAPI_INFO - Progress description missing\n" );
					} else {
						switch ( anInfoElement[ 2 ] & 0x7F ) {
							case 0x01:
								FopDebug( FOP_DEBUG, "CAPI_INFO - Not end-to-end ISDN\n" );
								break;
							case 0x02:
								FopDebug( FOP_DEBUG, "CAPI_INFO - Destination is non ISDN\n" );
								break;
							case 0x03:
								FopDebug( FOP_DEBUG, "CAPI_INFO - Origination is non ISDN\n" );
								break;
							case 0x04:
								FopDebug( FOP_DEBUG, "CAPI_INFO - Call returned to ISDN\n" );
								break;
							case 0x05:
								FopDebug( FOP_DEBUG, "CAPI_INFO - Interworking occurred\n" );
								break;
							case 0x08:
								FopDebug( FOP_DEBUG, "CAPI_INFO - In-band information available\n" );
								break;
							default:
								FopDebug( FOP_DEBUG, "CAPI_INFO - Unknown progress description 0x%02x\n", anInfoElement[ 2 ] );
								break;
						}
					}
					break;
				case 0x0027:
					/* Notification Indicator */
					switch ( ( unsigned int ) anInfoElement[ 0 ] ) {
						case 0:
							FopDebug( FOP_DEBUG, "CAPI_INFO - NI - CALL SUSPENDED (%d)\n", anInfoElement[ 0 ] );
							break;
						case 1:
							FopDebug( FOP_DEBUG, "CAPI_INFO - NI - CALL RESUMED (%d)\n", anInfoElement[ 0 ] );
							break;
						case 2:
							FopDebug( FOP_DEBUG, "CAPI_INFO - NI - BEARER SERVICE CHANGED (%d)\n", anInfoElement[ 0 ] );
							break;
						case 0xF9:
							FopDebug( FOP_DEBUG, "CAPI_INFO - NI - PUT ON HOLD (%d)\n", anInfoElement[ 0 ] );
							break;
						case 0xFA:
							FopDebug( FOP_DEBUG, "CAPI_INFO - NI - RETRIEVED FROM HOLD (%d)\n", anInfoElement[ 0 ] );
							break;
						default:
							FopDebug( FOP_DEBUG, "CAPI_INFO - NI - UNKNOWN (%d)\n", anInfoElement[ 0 ] );
							break;
					}
					break;
				case 0x0028:
					/* Display */
					FopDebug( FOP_DEBUG, "CAPI_INFO - DISPLAY\n" );
					break;
				case 0x0029:
					/* DateTime */
					FopDebug( FOP_DEBUG, "CAPI_INFO - DATE/TIME (%02d/%02d/%02d %02d:%02d)\n",
						anInfoElement[ 0 ], anInfoElement[ 1 ], anInfoElement[ 2 ], anInfoElement[ 3 ], anInfoElement[ 4 ] );
					break;
				case 0x002C:
					/* Keypad facility */
					FopDebug( FOP_DEBUG, "CAPI_INFO - KEYPAD FACILITY\n" );
					break;
				case 0x006C: {
					/* Caller party number */
					//int nTmp;

					//FopDebug( FOP_DEBUG, "CAPI_INFO - CALLER PARTY NUMBER (%.%s)\n", anInfoElement[ 0 ], &anInfoElement[ 1 ] );
					FopDebug( FOP_DEBUG, "CAPI_INFO - CALLER PARTY NUMBER\n" );

					/*for ( nTmp = 0; nTmp < sizeof( anInfoElement ); nTmp++ ) {
						FopDebug( FOP_DEBUG, "InfoElement (%d): %x (%c)\n", nTmp, anInfoElement[ nTmp ], anInfoElement[ nTmp ] );
					}*/
					break;
				}
				case 0x0070:
					/* Called Party Number */
					FopDebug( FOP_DEBUG, "CAPI_INFO - CALLED PARTY NUMBER\n" );
					break;
				case 0x0074:
					/* Redirecting Number */
					FopDebug( FOP_DEBUG, "CAPI_INFO - REDIRECTING NUMBER\n" );
					break;
				case 0x00A1:
					/* Sending complete */
					FopDebug( FOP_DEBUG, "CAPI_INFO - SENDING COMPLETE\n" );
					break;
				case 0x4000:
					/* Charge in Units */
					FopDebug( FOP_DEBUG, "CAPI_INFO - CHARGE IN UNITS\n" );
					break;
				case 0x4001:
					/* Charge in Currency */
					FopDebug( FOP_DEBUG, "CAPI_INFO - CHARGE IN CURRENCY\n" );
					break;
				case 0x8001:
					/* Alerting */
					FopDebug( FOP_DEBUG, "CAPI_INFO - ALERTING (Setup early...)\n" );
					break;
				case 0x8002:
					/* Call Proceeding */
					FopDebug( FOP_DEBUG, "CAPI_INFO - CALL PROCEEDING\n" );
					break;
				case 0x8003:
					/* Progress */
					FopDebug( FOP_DEBUG, "CAPI_INFO - PROGRESS (Setup early...)\n" );
					break;
				case 0x8005:
					/* Setup */
					FopDebug( FOP_DEBUG, "CAPI_INFO - SETUP\n" );
					break;
				case 0x8007:
					/* Connect */
					FopDebug( FOP_DEBUG, "CAPI_INFO - CONNECT\n" );
					break;
				case 0x800D:
					/* Setup ACK */
					FopDebug( FOP_DEBUG, "CAPI_INFO - SETUP ACK\n" );
					break;
				case 0x800F:
					/* Connect ACK */
					FopDebug( FOP_DEBUG, "CAPI_INFO - CONNECT ACK\n" );
					break;
				case 0x8045:
					/* Disconnect */
					psConnection = capiFindPlci( nPlci );

					if ( psConnection == NULL ) {
						break;
					}
					FopDebug( FOP_DEBUG, "CAPI_INFO information indicated disconnect, so terminate connection\n" );

					capiHangup( psConnection );
					break;
				case 0x804D:
					/* Release */
					FopDebug( FOP_DEBUG, "CAPI_INFO - RELEASE\n" );
					break;
				case 0x805A:
					/* Release Complete */
					FopDebug( FOP_DEBUG, "CAPI_INFO - RELEASE COMPLETE\n" );
					break;
				case 0x8062:
					/* Facility */
					FopDebug( FOP_DEBUG, "CAPI_INFO - FACILITY\n" );
					break;
				case 0x806E:
					/* Notify */
					FopDebug( FOP_DEBUG, "CAPI_INFO - NOTIFY\n" );
					break;
				case 0x807B:
					/* Information */
					FopDebug( FOP_DEBUG, "CAPI_INFO - INFORMATION\n" );
					break;
				case 0x807D:
					/* Status */
					FopDebug( FOP_DEBUG, "CAPI_INFO - STATUS\n" );
					break;
				case 0xC000:/* {
					int nTmp;

					for ( nTmp = 0; nTmp < sizeof( anInfoElement ); nTmp++ ) {
						FopDebug( FOP_DEBUG, "InfoElement (%d): %x (%c)\n", nTmp, anInfoElement[ nTmp ], anInfoElement[ nTmp ] );
					}
				}*/
				default:
						/* Unknown */
					FopDebug( FOP_DEBUG, "CAPI_INFO - UNKNOWN INFO (0x%02x)\n", nInfo );
					break;
			}

			psConnection = capiFindPlci( nPlci );
			if ( psConnection != NULL ) {
				if ( psConnection -> nEarlyB3 != 0 && psConnection -> nState == STATE_CONNECT_WAIT && nInfo == 0x001E ) {
					FopDebug( FOP_DEBUG, "REQ: CONNECT_B3 - Early-B3\n" );

					isdnLock();
					CONNECT_B3_REQ( &sCmsg1, psSession -> nApplId, 0, nPlci, 0 );
					isdnUnlock();

					psConnection -> nConnectTime = time( NULL );
					if ( psConnection -> nType == SESSION_PHONE ) {
						psSession -> psHandlers -> AudioOpen();
					}
					psConnection -> nState = STATE_CONNECT_ACTIVE;
				}
			}
			break;

		/* CAPI_DISCONNECT_B3 - Disconnect data */
		case CAPI_DISCONNECT_B3:
			FopDebug( FOP_DEBUG, "IND: DISCONNECT_B3\n" );
			nNcci = CONNECT_B3_IND_NCCI( &sCapiMessage );
			nPlci = nNcci & 0x0000ffff;

			isdnLock();
			DISCONNECT_B3_RESP( &sCmsg1, psSession -> nApplId, psSession -> nMessageNumber++, nNcci );
			isdnUnlock();

			psConnection = capiFindNcci( nNcci );
			if ( psConnection == NULL ) {
				break;
			}

			psConnection -> nNcci = 0;
			if ( psConnection -> nState == STATE_CONNECTED || psConnection -> nState == STATE_CONNECT_B3_WAIT ) {
				/* passive disconnect, DISCONNECT_IND comes later */
				psConnection -> nState = STATE_DISCONNECT_ACTIVE;
			} else {
				/* active disconnect, needs to send DISCONNECT_REQ */
				capiHangup( psConnection );
			}

			FopDebug( FOP_DEBUG, "IND: CAPI_DISCONNECT_B3 - nConnection: %d, nPlci: %d, nNcci: %d\n", psConnection -> nId, psConnection -> nPlci, psConnection -> nNcci );
			break;

		/* CAPI_DISCONNECT - Disconnect */
		case CAPI_DISCONNECT:
			nPlci = DISCONNECT_IND_PLCI( &sCapiMessage );
			nInfo = DISCONNECT_IND_REASON( &sCapiMessage );

			FopDebug( FOP_DEBUG, "IND: DISCONNECT - nPlci %d\n", nPlci );

			FopDebug( FOP_DEBUG, "RESP: DISCONNECT - nPlci %d\n", nPlci );
			isdnLock();
			DISCONNECT_RESP( &sCmsg1, psSession -> nApplId, psSession -> nMessageNumber++, nPlci );
			isdnUnlock();

			psConnection = capiFindPlci( nPlci );
			if ( psConnection == NULL ) {
				FopDebug( FOP_DEBUG, "Connection not found, IGNORING\n" );
				break;
			}

			/* CAPI-Error code */
			psConnection -> nCapiCode = DISCONNECT_IND_REASON( &sCapiMessage );
			psConnection -> nState = STATE_IDLE;
			psConnection -> nNcci = 0;
			psConnection -> nPlci = 0;

			if ( psConnection -> nType == SESSION_PHONE ) {
				if ( psSession -> nInputThreadState == 1 ) {
					psSession -> nInputThreadState++;
					do {
						g_usleep( 10 );
					} while ( psSession -> nInputThreadState != 0 );
				}
				psSession -> psHandlers -> AudioClose();
			}

			psSession -> psHandlers -> Disconnected( psConnection );

			capiSetFree( psConnection );
			break;
		default:
			FopDebug( FOP_DEBUG, "Unhandled command 0x%x\n", sCapiMessage.Command );
			break;
	}

	return 0;
}

/**
 * \brief CAPI confirmation
 * \param sCapiMessage capi message structure
 */
static void capiConfirmation( _cmsg sCapiMessage ) {
	struct sCapiConnection *psConnection = NULL;
	unsigned int nInfo;
	unsigned int nPlci;
	int nController;

	switch ( sCapiMessage.Command ) {
		case CAPI_FACILITY:
			/* Facility */
			FopDebug( FOP_DEBUG, "CNF: CAPI_FACILITY; Info: %d\n", sCapiMessage.Info );
			break;
		case CAPI_LISTEN:
			/* Listen confirmation */
			nController = LISTEN_CONF_CONTROLLER( &sCapiMessage );
			FopDebug( FOP_DEBUG, "CNF: CAPI_LISTEN: controller %d, info %d\n", nController, sCapiMessage.Info );
			break;
		case CAPI_ALERT:
			/* Alert message */
			FopDebug( FOP_DEBUG, "CNF: CAPI_ALERT\n" );
			nInfo = ALERT_CONF_INFO( &sCapiMessage );
			nPlci = ALERT_CONF_PLCI( &sCapiMessage );

			FopDebug( FOP_DEBUG, "CNF: CAPI_ALERT: nInfo %d, nPlci %d\n", nInfo, nPlci );

			psConnection = capiFindPlci( nPlci );

			if ( nInfo != 0 && nInfo != 3 ) {
				if ( psConnection != NULL ) {
					psConnection -> nState = STATE_IDLE;
				}
			} else {
				psSession -> psHandlers -> Ring( psConnection );
			}
			break;
		case CAPI_DATA_B3:
			/* Sent data acknowledge, NOP */
			break;
		case CAPI_INFO:
			/* Info, NOP */
			FopDebug( FOP_DEBUG, "CNF: CAPI_INFO: info %d\n", sCapiMessage.Info );
			break;
		case CAPI_CONNECT:
			/* Physical channel connection is being established */
			nPlci = CONNECT_CONF_PLCI( &sCapiMessage );
			nInfo = CONNECT_CONF_INFO( &sCapiMessage );

			FopDebug( FOP_DEBUG, "CNF: CAPI_CONNECT - (nPlci: %d)\n", nPlci );
			/* .. or new outgoing call? get nPlci. */
			psConnection = capiFindNew();
			if ( psConnection == NULL ) {
				FopDebug( FOP_DEBUG, "CND: CAPI_CONNECT - Warning! Received confirmation but we didn't requested a connect!!!\n" );
				break;
			}

			if ( nInfo != 0 ) {
				/* Connection error */
				psConnection -> nState = STATE_IDLE;

				psSession -> psHandlers -> Status( psConnection, nInfo );

				capiSetFree( psConnection );
			} else {
				/* CONNECT_ACTIVE_IND comes later, when connection actually established */
				psConnection -> nPlci = nPlci;
				psConnection -> nState = STATE_CONNECT_WAIT;
			}
			break;
		case CAPI_CONNECT_B3:
			nPlci = CONNECT_CONF_PLCI( &sCapiMessage );

			FopDebug( FOP_DEBUG, "CNF: CAPI_CONNECT_B3\n" );
			capi_error( sCapiMessage.Info );
			break;
		case CAPI_DISCONNECT:
			FopDebug( FOP_DEBUG, "CNF: CAPI_DISCONNECT\n" );
			break;
		case CAPI_DISCONNECT_B3:
			FopDebug( FOP_DEBUG, "CNF: CAPI_DISCONNECT_B3\n" );
			break;
		default:
			FopDebug( FOP_DEBUG, "Unhandled confirmation, command 0x%x\n", sCapiMessage.Command );
			break;
	}
}

/**
 * \brief Main capi loop function
 * \param pUnused unused pointer
 * \return NULL
 */
static gpointer capiLoop( void *pUnused ) {
	struct timeval sTimeVal;
	unsigned int nInfo;
	unsigned int nRet;
	_cmsg sCapiMessage;

	while ( 1 ) {
		sTimeVal.tv_sec = 1;
		sTimeVal.tv_usec = 0;

		nRet = CAPI20_WaitforMessage( psSession -> nApplId, &sTimeVal );
		if ( nRet == CapiNoError ) {
			isdnLock();
			nInfo = capi_get_cmsg( &sCapiMessage, psSession -> nApplId );
			isdnUnlock();

			switch ( nInfo ) {
				case CapiNoError:
					switch ( sCapiMessage.Subcommand ) {
						/* Indication */
						case CAPI_IND:
							capiIndication( sCapiMessage );
							break;
						/* Confirmation */
						case CAPI_CONF:
							capiConfirmation( sCapiMessage );
							break;
					}
					break;
				case CapiReceiveQueueEmpty:
					FopDebug( FOP_INFO, "Empty queue, even if message pending.. sleeping 1 second\n" );
					g_usleep( 1 * G_USEC_PER_SEC );
					break;
				default:
					g_usleep( 1 );
					return NULL;
			}
		} else {
			if ( psSession == NULL || psSession -> nApplId == -1 ) {
				g_usleep( 1 * G_USEC_PER_SEC );
			} else {
				g_usleep( 10 );
			}
		}
	}

	return NULL;
}

/**
 * byteswap functions. We do not use byteswap.h 
 * as it is not partable
 */

static inline unsigned short bswap_16(unsigned short x) {
  return (x>>8) | (x<<8);
}

static inline unsigned int bswap_32(unsigned int x) {
  return (bswap_16(x&0xffff)<<16) | (bswap_16(x>>16));
}

/**
 * \brief get capi profile
 * Convert capi_profile data from wire format to host format
 * \param Controller capi controller
 * \param host host formated capi profile pointer
 * \return error code
 */
static int getCapiProfile( unsigned nController, struct capi_profile *psHost ) {
	int nRetVal = CAPI20_GET_PROFILE( nController, (unsigned char *) psHost );

	if ( nRetVal == 0 ) {
	}

	return nRetVal;
}

/**
 * \brief Initialize CAPI controller
 * \param nController controller id
 * \return error code
 */
static int capiInit( int nController ) {
	CAPI_REGISTER_ERROR nErrorCode = 0;
	_cmsg sCapiMessage;
	unsigned int nApplId = -1;
	unsigned char anBuffer[ 64 ];
	int nIndex;
	int nStart = 0;
	int nEnd = 0;
	int nSkip = -1;
	int nNumControllers = 0;
	struct capi_profile sProfile;
	int nDtmf;
	int nSuppServ;
	int nChannels = 0;
	int nTransp;
	int nFax;
	int nFaxExt;
	int nEcho;
	int nIntern;
	int nExtern;

	/* Check if capi is installed */
	nErrorCode = CAPI20_ISINSTALLED();
	if ( nErrorCode != 0 ) {
		FopDebug( FOP_ERROR, "CAPI 2.0: not installed, RC=0x%x\n", nErrorCode );
		return -1;
	}

	/* Fetch controller/bchannel count */
	nErrorCode = getCapiProfile ( 0, &sProfile );
	if ( nErrorCode != 0 ) {
		FopDebug( FOP_ERROR, "CAPI 2.0: Error getting profile, RC=0x%x\n", nErrorCode );
		return -1;
	}

	/* If there are no available controllers something went wrong, abort */
	nNumControllers = sProfile.ncontroller;
	if ( nNumControllers == 0 ) {
		FopDebug( FOP_ERROR, "CAPI 2.0: No ISDN controllers installed\n" );
		return -1;
	}

	/* Read manufacturer and version from device (entry 0) */
	FopDebug( FOP_INFO, "CAPI 2.0: Controllers found: %d\n", nNumControllers );
	if ( capi20_get_manufacturer( 0, anBuffer ) ) {
		FopDebug( FOP_INFO, "CAPI 2.0: Manufacturer: %s\n", anBuffer );
	}
	if ( capi20_get_version( 0, anBuffer ) ) {
		FopDebug( FOP_INFO, "CAPI 2.0: Version %d.%d/%d.%d\n",
			anBuffer[ 0 ], anBuffer[ 1 ], anBuffer[ 2 ], anBuffer[ 3 ] );
	}

	/* Now dump some informations about the controllers */
	for ( nIndex = 1; nIndex <= nNumControllers; nIndex++ ) {
		struct capi_profile sController;
		unsigned char *pnManu;

		FopDebug( FOP_INFO, "CAPI 2.0: Checking controller %d\n", nIndex );
		if ( capi20_get_manufacturer( nIndex, anBuffer ) ) {
			FopDebug( FOP_INFO, "CAPI 2.0: Controller: %d, Manufacturer: %s\n", nIndex, anBuffer );
		}
		if ( capi20_get_version( nIndex, anBuffer ) ) {
			FopDebug( FOP_INFO, "CAPI 2.0: Controller: %d, Version %d.%d/%d.%d\n",
				nIndex, anBuffer[ 0 ], anBuffer[ 1 ], anBuffer[ 2 ], anBuffer[ 3 ] );
		}

		nErrorCode = getCapiProfile( nIndex, &sController );
		if ( nErrorCode != 0 ) {
			FopDebug( FOP_ERROR, "CAPI 2.0: error getting controller %d profile, RC=0x%x\n", nIndex, nErrorCode );
			return -1;
		}
		FopDebug( FOP_INFO, "CAPI 2.0: Controller: %d, Options: 0x%x\n", nIndex, sController.goptions );

		nChannels = sController.nbchannel;
		nDtmf = sController.goptions & 0x08 ? 1 : 0;
		nSuppServ = sController.goptions & 0x10;
		nEcho = sController.goptions & 0x200;
		nIntern = sController.goptions & 0x01;
		nExtern = sController.goptions & 0x02;

		if ( sController.support1 & 0x02 && sController.support2 & 0x02 && sController.support3 & 0x01 ) {
			nTransp = 1;
		} else {
			nTransp = 0;
		}

		if ( sController.support1 & 0x10 && sController.support2 & 0x10 && sController.support3 & 0x10 ) {
			nFax = 1;
		} else {
			nFax = 0;
		}

		if ( sController.support1 & 0x10 && sController.support2 & 0x10 && sController.support3 & 0x20 ) {
			nFaxExt = 1;
		} else {
			nFaxExt = 0;
		}

		FopDebug( FOP_INFO, "CAPI 2.0: B-Channels %d, DTMF %d, FAX %d/%d, Transp %d, SuppServ %d\n",
			nChannels, nDtmf, nFax, nFaxExt, nTransp, nSuppServ );
		FopDebug( FOP_INFO, "CAPI 2.0: Echo: %d, Intern: %d, Extern: %d\n", nEcho, nIntern, nExtern );

		pnManu = ( unsigned char * ) sController.manu;

		FopDebug( FOP_DEBUG, "manufactor profile: 0x%x, 0x%x, 0x%x, 0x%x\n", pnManu[0], pnManu[1], pnManu[2], pnManu[3]);
		FopDebug( FOP_DEBUG, "Found nController #%d with %d B-channel(s)\n", nIndex, sController.nbchannel );
		if ( sController.nbchannel == 2 && pnManu[ 3 ] == 0x00 ) {
			FopDebug( FOP_DEBUG, "-> skipping internal nController %d\n", nIndex );
			nSkip = nIndex;
		}
	}

	/* Listen to all (<=0) or single controller (>=1) */
	if ( nController <= 0 ) {
		nStart = 1;
		nEnd = nNumControllers;
	} else {
		nStart = nController;
		nEnd = nController;
	}

	/* Register with CAPI */
	if ( nApplId == -1 ) {
		nErrorCode = CAPI20_REGISTER( CAPI_BCHANNELS, CAPI_BUFFERCNT, CAPI_PACKETS, &nApplId );
		if ( nErrorCode != 0 || nApplId == 0 ) {
			FopDebug( FOP_DEBUG, "Error while registering application, RC=0x%x\n", nErrorCode );
			/* registration error! */
			return -2;
		}
	}

	/* Listen to CAPI controller(s) */
	for ( nIndex = nStart; nIndex <= nEnd; nIndex++ ) {
		if ( nController <= 0 && nSkip == nIndex ) {
			/* if we are using auto channels, skip internal controller */
			continue;
		}
		nErrorCode = LISTEN_REQ( &sCapiMessage, nApplId, 0, nIndex, 0x3FF, 0x1FFF03FF, 0, NULL, NULL );
		if ( nErrorCode != 0 ) {
			FopDebug( FOP_DEBUG, "LISTEN_REQ failed, RC=0x%x\n", nErrorCode );
			return -3;
		}
		FopDebug( FOP_DEBUG, "Listen to nController #%d ...\n", nIndex );
	}

	/* ok! */
	return nApplId;
}

extern void setHostName( const char *pnHost );

/**
 * \brief Initialize faxophone structure
 * \param psHandlers session handlers
 * \param pnHost host name of router
 * \return session pointer or NULL on error
 */
struct sSession *faxophoneInit( struct sSessionHandlers *psHandlers, const char *pnHost ) {
	int nController = -1;
	int nApplId = -1;

	if ( psSession == NULL ) {
		if ( pnHost != NULL ) {
			setHostName( pnHost );
		}

		nApplId = capiInit( nController );
		if ( nApplId <= 0 ) {
			FopDebug( FOP_DEBUG, "Initialization failed! Error %d!\n", nApplId );

			return NULL;
		} else {
			psSession = g_malloc0( sizeof( struct sSession ) );

			psSession -> psIsdnMutex = g_mutex_new();

			psSession -> psHandlers = psHandlers;

			psSession -> nApplId = nApplId;

			createTableBuffer();

			/* start capi transmission loop */
			CREATE_THREAD( "capi", capiLoop, NULL );
		}

		FopDebug( FOP_DEBUG, "Successfully initialized!\n" );
	}

	return psSession;
}

/**
 * \brief Destroy faxophone
 * \return error code 0
 */
int faxophoneClose( void ) {
	/* Close capi connection */
	capiClose();

	FopDebug( FOP_DEBUG, "closed!\n" );

	return 0;
}

/**
 * \brief Get active faxophone session
 * \return session pointer or NULL on error
 */
struct sSession *faxophoneGetSession( void ) {
	return psSession;
}
