/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright (C) 1991-2010 OpenCFD Ltd.
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::probes

Description
    Set of locations to sample.

    Call write() to sample and write files.

SourceFiles
    probes.C

\*---------------------------------------------------------------------------*/

#ifndef probes_H
#define probes_H

#include <OpenFOAM/HashPtrTable.H>
#include <OpenFOAM/OFstream.H>
#include <OpenFOAM/polyMesh.H>
#include <OpenFOAM/pointField.H>
#include <finiteVolume/volFieldsFwd.H>

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

// Forward declaration of classes
class objectRegistry;
class dictionary;
class fvMesh;
class mapPolyMesh;

/*---------------------------------------------------------------------------*\
                          Class probes Declaration
\*---------------------------------------------------------------------------*/

class probes
{
protected:

    // Protected classes

        //- Class used for grouping field types
        template<class Type>
        class fieldGroup
        :
            public wordList
        {
        public:
            //- Construct null
            fieldGroup()
            :
                wordList()
            {}

            //- Construct for a list of field names
            fieldGroup(const wordList& fieldNames)
            :
                wordList(fieldNames)
            {}
        };


    // Protected data

        //- Name of this set of probes,
        //  Also used as the name of the probes directory.
        word name_;

        //- Const reference to objectRegistry
        const objectRegistry& obr_;

        //- Load fields from files (not from objectRegistry)
        bool loadFromFiles_;


        // Read from dictonary

            //- Names of fields to probe
            wordList fieldNames_;

            //- Locations to probe
            vectorField probeLocations_;


        // Calculated

            //- Categorized scalar/vector/tensor fields
            fieldGroup<scalar> scalarFields_;
            fieldGroup<vector> vectorFields_;
            fieldGroup<sphericalTensor> sphericalTensorFields_;
            fieldGroup<symmTensor> symmTensorFields_;
            fieldGroup<tensor> tensorFields_;

            // Cells to be probed (obtained from the locations)
            labelList elementList_;

            //- Current open files
            HashPtrTable<OFstream> probeFilePtrs_;


    // Private Member Functions

        //- Find element containing probes
        virtual void findElements(const fvMesh&);

        //- classify field types, return true if nFields > 0
        bool checkFieldTypes();

        //- Find the fields in the list of the given type, return count
        template<class Type>
        label countFields
        (
            fieldGroup<Type>& fieldList,
            const wordList& fieldTypes
        ) const;


private:

        //- Sample and write a particular volume field
        template<class Type>
        void sampleAndWrite
        (
            const GeometricField<Type, fvPatchField, volMesh>&
        );

        //- Sample and write all the fields of the given type
        template <class Type>
        void sampleAndWrite(const fieldGroup<Type>&);

        //- Disallow default bitwise copy construct
        probes(const probes&);

        //- Disallow default bitwise assignment
        void operator=(const probes&);


public:

    //- Runtime type information
    TypeName("probes");


    // Constructors

        //- Construct for given objectRegistry and dictionary.
        //  Allow the possibility to load fields from files
        probes
        (
            const word& name,
            const objectRegistry&,
            const dictionary&,
            const bool loadFromFiles = false
        );


    //- Destructor
    virtual ~probes();


    // Member Functions

        //- Return name of the set of probes
        virtual const word& name() const
        {
            return name_;
        }

        //- Return names of fields to probe
        virtual const wordList& fieldNames() const
        {
            return fieldNames_;
        }

        //- Return locations to probe
        virtual const vectorField& probeLocations() const
        {
            return probeLocations_;
        }

        //- Cells to be probed (obtained from the locations)
        const labelList& elements() const
        {
            return elementList_;
        }

        //- Execute, currently does nothing
        virtual void execute();

        //- Execute at the final time-loop, currently does nothing
        virtual void end();

        //- Sample and write
        virtual void write();

        //- Read the probes
        virtual void read(const dictionary&);

        //- Update for changes of mesh
        virtual void updateMesh(const mapPolyMesh&)
        {}

        //- Update for changes of mesh
        virtual void movePoints(const pointField&)
        {}

        //- Update for changes of mesh due to readUpdate
        virtual void readUpdate(const polyMesh::readUpdateState state)
        {}

        //- Sample a volume field at all locations
        template<class Type>
        tmp<Field<Type> > sample
        (
            const GeometricField<Type, fvPatchField, volMesh>&
        ) const;

        //- Sample a single field on all sample locations
        template <class Type>
        tmp<Field<Type> > sample(const word& fieldName) const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#ifdef NoRepository
#   include <sampling/probesTemplates.C>
#endif

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************ vim: set sw=4 sts=4 et: ************************ //
