/*
 * GnomeSplit.java
 * 
 * Copyright (c) 2009-2011 Guillaume Mazoyer
 * 
 * This file is part of GNOME Split.
 * 
 * GNOME Split is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GNOME Split is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNOME Split.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.gnome.split;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;

import org.freedesktop.bindings.Internationalization;
import org.gnome.glib.Glib;
import org.gnome.gtk.Gtk;
import org.gnome.notify.Notify;
import org.gnome.split.config.Configuration;
import org.gnome.split.config.Constants;
import org.gnome.split.core.EngineListener;
import org.gnome.split.core.utils.Algorithm;
import org.gnome.split.core.utils.ShutdownHandler;
import org.gnome.split.core.utils.UncaughtExceptionLogger;
import org.gnome.split.getopt.GetOptions;
import org.gnome.split.getopt.LongOption;
import org.gnome.split.gtk.DefaultEngineListener;
import org.gnome.split.gtk.MainWindow;
import org.gnome.split.gtk.action.ActionManager;
import org.gnome.split.gtk.action.ActionManager.ActionId;
import org.gnome.split.gtk.dialog.ErrorDialog;
import org.gnome.split.gtk.dialog.QuestionDialog;
import org.gnome.unique.Application;
import org.gnome.unique.Command;
import org.gnome.unique.MessageData;
import org.gnome.unique.Response;

import static org.freedesktop.bindings.Internationalization._;

/**
 * This class contains the GNOME Split application entry point.
 * 
 * @author Guillaume Mazoyer
 */
public final class GnomeSplit
{
    /**
     * Configuration for the application.
     */
    private Configuration config;

    /**
     * Application actions manager.
     */
    private ActionManager actions;

    /**
     * Application main window.
     */
    private MainWindow window;

    /**
     * Engine listener to update the view.
     */
    private EngineListener engine;

    /**
     * Create an instance of the application.
     */
    private GnomeSplit(String[] args) {
        // Initialize uncaught exception handler
        new UncaughtExceptionLogger();

        // Start kill signals handler
        Runtime.getRuntime().addShutdownHook(new ShutdownHandler());

        // Load program name
        Glib.setProgramName(Constants.PROGRAM_NAME);

        // Load GTK
        Gtk.init(args);

        // Load config
        this.loadConfig();

        // Check if an instance is running
        this.checkRunning();

        // Load logo
        Gtk.setDefaultIcon(Constants.PROGRAM_LOGO);

        // Load translations
        Internationalization.init("gnome-split", "share/locale/");

        // Load libnotify
        if (config.USE_NOTIFICATION) {
            Notify.init(Constants.PROGRAM_NAME);
        }

        // Build the user interface
        this.buildUserInterface();

        // If there are some arguments
        if (args.length > 0) {
            this.parseCommandLine(args);
        } else {
            if (config.ASSISTANT_ON_START) {
                // Show the assistant on start if requested
                actions.activateAction(ActionId.ASSISTANT);
            }
        }

        // Start GTK main loop (blocker method)
        Gtk.main();
    }

    /**
     * Load the configuration and preferences.
     */
    private void loadConfig() {
        try {
            // Load constants and preferences
            Constants.load();
            config = new Configuration();
        } catch (IOException e) {
            e.printStackTrace();
            System.exit(1);
        }
    }

    /**
     * Check if an instance of GNOME Split is already running.
     */
    private void checkRunning() {
        // Initialize unique application check
        Application application = new Application("org.gnome.GnomeSplit", null);

        // Signal to handle message from other instances
        application.connect(new Application.MessageReceived() {
            @Override
            public Response onMessageReceived(Application source, Command cmd, MessageData data, int time) {
                ErrorDialog dialog = new ErrorDialog(
                        getMainWindow(),
                        _("More than one instance."),
                        _("Only one instance of GNOME Split can be executed at a time. If you want to run multiple instances, edit the preferences. Remember that it is never safe to run more than one instance of GNOME Split."));
                dialog.run();
                dialog.hide();
                return Response.OK;
            }
        });

        // Already running, quit this application
        if (application.isRunning() && !config.MULTIPLE_INSTANCES) {
            // Send the message
            application.sendMessage(Command.CLOSE, new MessageData());

            // Quit the current app
            System.exit(1);
        }
    }

    /**
     * Build the GTK+ user interface.
     */
    private void buildUserInterface() {
        // Load actions manager
        actions = new ActionManager(this);

        // Start the user interface
        window = new MainWindow(this);
        window.selectDefaultView();
        window.show();

        // Load engine listener
        engine = new DefaultEngineListener(this);
    }

    /**
     * Change the view of the window, and update it using a filename. If no
     * filename is given (filename == null), the view will not be updated. 0
     * means split view and other means merge view.
     */
    private void selectView(byte view, String filename) {
        // Choose split
        if (view == 0) {
            if (filename != null) {
                // Update the merge widget
                window.getSplitWidget().setFile(filename);
            }

            // Show the merge widget
            actions.activateRadioAction(ActionId.SPLIT);
        } else {
            if (filename != null) {
                // Load the file to split
                window.getMergeWidget().setFile(filename);
            }

            // Show the split widget
            actions.activateRadioAction(ActionId.MERGE);
        }
    }

    /**
     * Parse the command line using the GNU getopt.
     */
    private void parseCommandLine(String[] args) {
        int character;

        // Long options - use them make the command line more human readable
        LongOption[] options = new LongOption[2];
        options[0] = new LongOption("merge", LongOption.REQUIRED_ARGUMENT, 'm');
        options[1] = new LongOption("split", LongOption.REQUIRED_ARGUMENT, 's');

        // Create the getopt object to parse the arguments
        GetOptions getopt = new GetOptions("gnome-split", args, "m:s:", options);

        // While there is something to parse
        while ((character = getopt.getOption()) != -1) {
            switch (character) {
            case 'm':
                this.selectView((byte) 1, getopt.getArgument());

                break;

            case 's':
                this.selectView((byte) 0, getopt.getArgument());

                break;

            default:
                break;
            }
        }

        // No options have been used, this is just a file that has been given
        if (!args[0].startsWith("-")) {
            // Update the view, checking the file extension
            byte select = (byte) (Algorithm.isValidExtension(args[0]) ? 1 : 0);
            this.selectView(select, args[0]);
        }
    }

    /**
     * Return the configuration object of the app.
     */
    public Configuration getConfig() {
        return config;
    }

    /**
     * Return the actions manager of the app.
     */
    public ActionManager getActionManager() {
        return actions;
    }

    /**
     * Return the main window of the app.
     */
    public MainWindow getMainWindow() {
        return window;
    }

    /**
     * Return the engine listener of the app.
     */
    public EngineListener getEngineListener() {
        return engine;
    }

    /**
     * Open the the URI with the default program.
     */
    public void openURI(String uri) {
        try {
            Gtk.showURI(new URI(uri));
        } catch (URISyntaxException e) {
            // Should *never* happen
            e.printStackTrace();
        }
    }

    /**
     * This will cause the program to be ended.
     */
    public void quit() {
        boolean quit = true;

        // An action is running
        if (!config.DO_NOT_ASK_QUIT && (engine.getEngine() != null)) {
            // Show a question to the user
            QuestionDialog dialog = new QuestionDialog(this, window, _("Quit GNOME Split."),
                    _("An action is currently being perfomed. Do you really want to quit GNOME Split?"));

            // Get his response and hide the dialog
            quit = dialog.response();
            dialog.hide();
        }

        // The user really wants to quit
        if (quit) {
            // Hide the window immediately
            window.hide();

            // Uninitialize the libnotify
            if (Notify.isInitialized()) {
                Notify.uninit();
            }

            // Quit the GTK main loop (cause the app end)
            Gtk.mainQuit();

            // Ending program
            System.exit(0);
        }
    }

    /**
     * Application entry point.
     */
    public static void main(String[] args) {
        new GnomeSplit(args);
    }
}
