// Copyright (c) Microsoft Corporation.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception

#ifndef FLOAT_GENERAL_PRECISION_TO_CHARS_TEST_CASES_HPP
#define FLOAT_GENERAL_PRECISION_TO_CHARS_TEST_CASES_HPP

#include <charconv>

#include "test.hpp"
using namespace std;

// C11 7.21.6.1 "The fprintf function"/8:

// "Then, if a conversion with style E would have an exponent of X:
// - if P > X >= -4, the conversion is with style f (or F) and precision P - (X + 1).
// - otherwise, the conversion is with style e (or E) and precision P - 1."

// "Finally, [...] any trailing zeros are removed from the fractional portion of the result
// and the decimal-point character is removed if there is no fractional portion remaining."

inline constexpr FloatPrecisionToCharsTestCase float_general_precision_to_chars_test_cases[] = {
    // Test special cases (zero, inf, nan) and an ordinary case. Also test negative signs.
    {0.0f, chars_format::general, 4, "0"},
    {-0.0f, chars_format::general, 4, "-0"},
    {float_inf, chars_format::general, 4, "inf"},
    {-float_inf, chars_format::general, 4, "-inf"},
    {float_nan, chars_format::general, 4, "nan"},
    {-float_nan, chars_format::general, 4, "-nan(ind)"},
    {float_nan_payload, chars_format::general, 4, "nan"},
    {-float_nan_payload, chars_format::general, 4, "-nan"},
    {1.729f, chars_format::general, 4, "1.729"},
    {-1.729f, chars_format::general, 4, "-1.729"},

    // Test corner cases.
    {0x0.000002p-126f, chars_format::general, 1000,
        "1.40129846432481707092372958328991613128026194187651577175706828388979108268586060148663818836212158203125"
        "e-45"}, // min subnormal
    {0x0.fffffep-126f, chars_format::general, 1000,
        "1."
        "17549421069244107548702944484928734882705242874589333385717453057158887047561890426550235133618116378784179687"
        "5e-38"}, // max subnormal
    {0x1p-126f, chars_format::general, 1000,
        "1.1754943508222875079687365372222456778186655567720875215087517062784172594547271728515625e-38"}, // min normal
    {0x1.fffffep+127f, chars_format::general, 1000, "340282346638528859811704183484516925440"}, // max normal

    {0x0.000002p-126f, chars_format::general, 6, "1.4013e-45"}, // min subnormal
    {0x0.fffffep-126f, chars_format::general, 6, "1.17549e-38"}, // max subnormal
    {0x1p-126f, chars_format::general, 6, "1.17549e-38"}, // min normal
    {0x1.fffffep+127f, chars_format::general, 6, "3.40282e+38"}, // max normal

    // Test maximum-length output (excluding minus signs).
    {0x1.fffffep-126f, chars_format::general, 1000,
        "2."
        "35098856151472858345576598207153302664571798551798085536592623685000612993034607711706485133618116378784179687"
        "5e-38"}, // scientific, happens to be the same length as max subnormal
    {0x1.fffffep-14f, chars_format::general, 1000, "0.0001220703052240423858165740966796875"}, // fixed

    // Test varying precision. Negative precision requests P == 6. Zero precision requests P == 1.
    // Here, the scientific exponent X is 0.
    // Therefore, fixed notation is always chosen with precision P - (X + 1) == P - 1.
    {0x1.b04p0f, chars_format::general, -2, "1.68848"},
    {0x1.b04p0f, chars_format::general, -1, "1.68848"},
    {0x1.b04p0f, chars_format::general, 0, "2"},
    {0x1.b04p0f, chars_format::general, 1, "2"}, // fixed notation trims decimal point
    {0x1.b04p0f, chars_format::general, 2, "1.7"},
    {0x1.b04p0f, chars_format::general, 3, "1.69"},
    {0x1.b04p0f, chars_format::general, 4, "1.688"},
    {0x1.b04p0f, chars_format::general, 5, "1.6885"},
    {0x1.b04p0f, chars_format::general, 6, "1.68848"},
    {0x1.b04p0f, chars_format::general, 7, "1.688477"},
    {0x1.b04p0f, chars_format::general, 8, "1.6884766"},
    {0x1.b04p0f, chars_format::general, 9, "1.68847656"},
    {0x1.b04p0f, chars_format::general, 10, "1.688476562"}, // round to even
    {0x1.b04p0f, chars_format::general, 11, "1.6884765625"}, // exact
    {0x1.b04p0f, chars_format::general, 12, "1.6884765625"}, // trim trailing zeros
    {0x1.b04p0f, chars_format::general, 13, "1.6884765625"},

    // Here, the scientific exponent X is -5.
    // Therefore, scientific notation is always chosen with precision P - 1.
    {0x1.8p-15f, chars_format::general, -2, "4.57764e-05"},
    {0x1.8p-15f, chars_format::general, -1, "4.57764e-05"},
    {0x1.8p-15f, chars_format::general, 0, "5e-05"},
    {0x1.8p-15f, chars_format::general, 1, "5e-05"}, // scientific notation trims decimal point
    {0x1.8p-15f, chars_format::general, 2, "4.6e-05"},
    {0x1.8p-15f, chars_format::general, 3, "4.58e-05"},
    {0x1.8p-15f, chars_format::general, 4, "4.578e-05"},
    {0x1.8p-15f, chars_format::general, 5, "4.5776e-05"},
    {0x1.8p-15f, chars_format::general, 6, "4.57764e-05"},
    {0x1.8p-15f, chars_format::general, 7, "4.577637e-05"},
    {0x1.8p-15f, chars_format::general, 8, "4.5776367e-05"},
    {0x1.8p-15f, chars_format::general, 9, "4.57763672e-05"},
    {0x1.8p-15f, chars_format::general, 10, "4.577636719e-05"},
    {0x1.8p-15f, chars_format::general, 11, "4.5776367188e-05"}, // round to even
    {0x1.8p-15f, chars_format::general, 12, "4.57763671875e-05"}, // exact
    {0x1.8p-15f, chars_format::general, 13, "4.57763671875e-05"}, // trim trailing zeros
    {0x1.8p-15f, chars_format::general, 14, "4.57763671875e-05"},

    // Trim trailing zeros.
    {0x1.80015p0f, chars_format::general, 1, "2"}, // fixed notation trims decimal point
    {0x1.80015p0f, chars_format::general, 2, "1.5"},
    {0x1.80015p0f, chars_format::general, 3, "1.5"}, // general trims trailing zeros
    {0x1.80015p0f, chars_format::general, 4, "1.5"},
    {0x1.80015p0f, chars_format::general, 5, "1.5"},
    {0x1.80015p0f, chars_format::general, 6, "1.50002"},
    {0x1.80015p0f, chars_format::general, 7, "1.50002"},
    {0x1.80015p0f, chars_format::general, 8, "1.50002"},
    {0x1.80015p0f, chars_format::general, 9, "1.50002003"},
    {0x1.80015p0f, chars_format::general, 10, "1.500020027"},
    {0x1.80015p0f, chars_format::general, 11, "1.5000200272"},
    {0x1.80015p0f, chars_format::general, 12, "1.50002002716"},
    {0x1.80015p0f, chars_format::general, 13, "1.500020027161"},
    {0x1.80015p0f, chars_format::general, 14, "1.5000200271606"},
    {0x1.80015p0f, chars_format::general, 15, "1.50002002716064"},
    {0x1.80015p0f, chars_format::general, 16, "1.500020027160645"},
    {0x1.80015p0f, chars_format::general, 17, "1.5000200271606445"},
    {0x1.80015p0f, chars_format::general, 18, "1.50002002716064453"},
    {0x1.80015p0f, chars_format::general, 19, "1.500020027160644531"},
    {0x1.80015p0f, chars_format::general, 20, "1.5000200271606445312"}, // round to even
    {0x1.80015p0f, chars_format::general, 21, "1.50002002716064453125"}, // exact

    // Trim trailing zeros and decimal point.
    {0x1.00015p0f, chars_format::general, 1, "1"}, // fixed notation trims decimal point
    {0x1.00015p0f, chars_format::general, 2, "1"}, // general trims decimal point and trailing zeros
    {0x1.00015p0f, chars_format::general, 3, "1"},
    {0x1.00015p0f, chars_format::general, 4, "1"},
    {0x1.00015p0f, chars_format::general, 5, "1"},
    {0x1.00015p0f, chars_format::general, 6, "1.00002"},
    {0x1.00015p0f, chars_format::general, 7, "1.00002"},
    {0x1.00015p0f, chars_format::general, 8, "1.00002"},
    {0x1.00015p0f, chars_format::general, 9, "1.00002003"},
    {0x1.00015p0f, chars_format::general, 10, "1.000020027"},
    {0x1.00015p0f, chars_format::general, 11, "1.0000200272"},
    {0x1.00015p0f, chars_format::general, 12, "1.00002002716"},
    {0x1.00015p0f, chars_format::general, 13, "1.000020027161"},
    {0x1.00015p0f, chars_format::general, 14, "1.0000200271606"},
    {0x1.00015p0f, chars_format::general, 15, "1.00002002716064"},
    {0x1.00015p0f, chars_format::general, 16, "1.000020027160645"},
    {0x1.00015p0f, chars_format::general, 17, "1.0000200271606445"},
    {0x1.00015p0f, chars_format::general, 18, "1.00002002716064453"},
    {0x1.00015p0f, chars_format::general, 19, "1.000020027160644531"},
    {0x1.00015p0f, chars_format::general, 20, "1.0000200271606445312"}, // round to even
    {0x1.00015p0f, chars_format::general, 21, "1.00002002716064453125"}, // exact

    // Trim trailing zeros, scientific notation.
    {0x1.5cf752p-20f, chars_format::general, 1, "1e-06"}, // scientific notation trims decimal point
    {0x1.5cf752p-20f, chars_format::general, 2, "1.3e-06"},
    {0x1.5cf752p-20f, chars_format::general, 3, "1.3e-06"}, // general trims trailing zeros
    {0x1.5cf752p-20f, chars_format::general, 4, "1.3e-06"},
    {0x1.5cf752p-20f, chars_format::general, 5, "1.3e-06"},
    {0x1.5cf752p-20f, chars_format::general, 6, "1.3e-06"},
    {0x1.5cf752p-20f, chars_format::general, 7, "1.3e-06"},
    {0x1.5cf752p-20f, chars_format::general, 8, "1.3e-06"},
    {0x1.5cf752p-20f, chars_format::general, 9, "1.30000001e-06"},
    {0x1.5cf752p-20f, chars_format::general, 10, "1.300000008e-06"},
    {0x1.5cf752p-20f, chars_format::general, 11, "1.3000000081e-06"},
    {0x1.5cf752p-20f, chars_format::general, 12, "1.30000000809e-06"},
    {0x1.5cf752p-20f, chars_format::general, 13, "1.300000008086e-06"},
    {0x1.5cf752p-20f, chars_format::general, 14, "1.3000000080865e-06"},
    {0x1.5cf752p-20f, chars_format::general, 15, "1.3000000080865e-06"},
    {0x1.5cf752p-20f, chars_format::general, 16, "1.300000008086499e-06"},
    {0x1.5cf752p-20f, chars_format::general, 17, "1.3000000080864993e-06"},
    {0x1.5cf752p-20f, chars_format::general, 18, "1.30000000808649929e-06"},
    {0x1.5cf752p-20f, chars_format::general, 19, "1.300000008086499292e-06"},
    {0x1.5cf752p-20f, chars_format::general, 20, "1.3000000080864992924e-06"},
    {0x1.5cf752p-20f, chars_format::general, 21, "1.3000000080864992924e-06"},
    {0x1.5cf752p-20f, chars_format::general, 22, "1.300000008086499292403e-06"},
    {0x1.5cf752p-20f, chars_format::general, 23, "1.3000000080864992924035e-06"},
    {0x1.5cf752p-20f, chars_format::general, 24, "1.30000000808649929240346e-06"},
    {0x1.5cf752p-20f, chars_format::general, 25, "1.30000000808649929240346e-06"},
    {0x1.5cf752p-20f, chars_format::general, 26, "1.3000000080864992924034595e-06"},
    {0x1.5cf752p-20f, chars_format::general, 27, "1.30000000808649929240345955e-06"},
    {0x1.5cf752p-20f, chars_format::general, 28, "1.300000008086499292403459549e-06"},
    {0x1.5cf752p-20f, chars_format::general, 29, "1.300000008086499292403459549e-06"},
    {0x1.5cf752p-20f, chars_format::general, 30, "1.30000000808649929240345954895e-06"},
    {0x1.5cf752p-20f, chars_format::general, 31, "1.30000000808649929240345954895e-06"},
    {0x1.5cf752p-20f, chars_format::general, 32, "1.3000000080864992924034595489502e-06"},
    {0x1.5cf752p-20f, chars_format::general, 33, "1.3000000080864992924034595489502e-06"},
    {0x1.5cf752p-20f, chars_format::general, 34, "1.300000008086499292403459548950195e-06"},
    {0x1.5cf752p-20f, chars_format::general, 35, "1.3000000080864992924034595489501953e-06"},
    {0x1.5cf752p-20f, chars_format::general, 36, "1.30000000808649929240345954895019531e-06"},
    {0x1.5cf752p-20f, chars_format::general, 37, "1.300000008086499292403459548950195312e-06"}, // round to even
    {0x1.5cf752p-20f, chars_format::general, 38, "1.3000000080864992924034595489501953125e-06"}, // exact

    // Trim trailing zeros and decimal point, scientific notation.
    {0x1.92a738p-19f, chars_format::general, 1, "3e-06"}, // scientific notation trims decimal point
    {0x1.92a738p-19f, chars_format::general, 2, "3e-06"}, // general trims decimal point and trailing zeros
    {0x1.92a738p-19f, chars_format::general, 3, "3e-06"},
    {0x1.92a738p-19f, chars_format::general, 4, "3e-06"},
    {0x1.92a738p-19f, chars_format::general, 5, "3e-06"},
    {0x1.92a738p-19f, chars_format::general, 6, "3e-06"},
    {0x1.92a738p-19f, chars_format::general, 7, "3e-06"},
    {0x1.92a738p-19f, chars_format::general, 8, "3.0000001e-06"},
    {0x1.92a738p-19f, chars_format::general, 9, "3.00000011e-06"},
    {0x1.92a738p-19f, chars_format::general, 10, "3.000000106e-06"},
    {0x1.92a738p-19f, chars_format::general, 11, "3.0000001061e-06"},
    {0x1.92a738p-19f, chars_format::general, 12, "3.00000010611e-06"},
    {0x1.92a738p-19f, chars_format::general, 13, "3.000000106113e-06"},
    {0x1.92a738p-19f, chars_format::general, 14, "3.0000001061126e-06"},
    {0x1.92a738p-19f, chars_format::general, 15, "3.00000010611257e-06"},
    {0x1.92a738p-19f, chars_format::general, 16, "3.000000106112566e-06"},
    {0x1.92a738p-19f, chars_format::general, 17, "3.0000001061125658e-06"},
    {0x1.92a738p-19f, chars_format::general, 18, "3.00000010611256585e-06"},
    {0x1.92a738p-19f, chars_format::general, 19, "3.000000106112565845e-06"},
    {0x1.92a738p-19f, chars_format::general, 20, "3.0000001061125658453e-06"},
    {0x1.92a738p-19f, chars_format::general, 21, "3.00000010611256584525e-06"},
    {0x1.92a738p-19f, chars_format::general, 22, "3.000000106112565845251e-06"},
    {0x1.92a738p-19f, chars_format::general, 23, "3.0000001061125658452511e-06"},
    {0x1.92a738p-19f, chars_format::general, 24, "3.00000010611256584525108e-06"},
    {0x1.92a738p-19f, chars_format::general, 25, "3.000000106112565845251083e-06"},
    {0x1.92a738p-19f, chars_format::general, 26, "3.0000001061125658452510834e-06"},
    {0x1.92a738p-19f, chars_format::general, 27, "3.00000010611256584525108337e-06"},
    {0x1.92a738p-19f, chars_format::general, 28, "3.000000106112565845251083374e-06"},
    {0x1.92a738p-19f, chars_format::general, 29, "3.000000106112565845251083374e-06"},
    {0x1.92a738p-19f, chars_format::general, 30, "3.00000010611256584525108337402e-06"},
    {0x1.92a738p-19f, chars_format::general, 31, "3.000000106112565845251083374023e-06"},
    {0x1.92a738p-19f, chars_format::general, 32, "3.0000001061125658452510833740234e-06"},
    {0x1.92a738p-19f, chars_format::general, 33, "3.00000010611256584525108337402344e-06"},
    {0x1.92a738p-19f, chars_format::general, 34, "3.000000106112565845251083374023438e-06"}, // round to even
    {0x1.92a738p-19f, chars_format::general, 35, "3.0000001061125658452510833740234375e-06"}, // exact

    // Test a large precision with fixed notation and scientific notation,
    // verifying that we remain within the bounds of any lookup tables.
    {0x1.ba9fbep+0f, chars_format::general, 5000, "1.72899997234344482421875"},
    {0x1.d01ffap-20f, chars_format::general, 5000, "1.7290000187131226994097232818603515625e-06"},

    // Test the transitions between fixed notation and scientific notation.
    {5555555.0f, chars_format::general, 1, "6e+06"},
    {555555.0f, chars_format::general, 1, "6e+05"},
    {55555.0f, chars_format::general, 1, "6e+04"},
    {5555.0f, chars_format::general, 1, "6e+03"},
    {555.0f, chars_format::general, 1, "6e+02"},
    {55.0f, chars_format::general, 1, "6e+01"}, // round to even
    {5.0f, chars_format::general, 1, "5"},
    {0x1p-3f, chars_format::general, 1, "0.1"}, // 0.125
    {0x1p-6f, chars_format::general, 1, "0.02"}, // 0.015625
    {0x1p-9f, chars_format::general, 1, "0.002"}, // 0.001953125
    {0x1p-13f, chars_format::general, 1, "0.0001"}, // 0.0001220703125
    {0x1p-16f, chars_format::general, 1, "2e-05"}, // 1.52587890625e-05
    {0x1p-19f, chars_format::general, 1, "2e-06"}, // 1.9073486328125e-06

    {5555555.0f, chars_format::general, 2, "5.6e+06"},
    {555555.0f, chars_format::general, 2, "5.6e+05"},
    {55555.0f, chars_format::general, 2, "5.6e+04"},
    {5555.0f, chars_format::general, 2, "5.6e+03"},
    {555.0f, chars_format::general, 2, "5.6e+02"}, // round to even
    {55.0f, chars_format::general, 2, "55"},
    {5.0f, chars_format::general, 2, "5"},
    {0x1p-3f, chars_format::general, 2, "0.12"}, // round to even
    {0x1p-6f, chars_format::general, 2, "0.016"},
    {0x1p-9f, chars_format::general, 2, "0.002"},
    {0x1p-13f, chars_format::general, 2, "0.00012"},
    {0x1p-16f, chars_format::general, 2, "1.5e-05"},
    {0x1p-19f, chars_format::general, 2, "1.9e-06"},

    {5555555.0f, chars_format::general, 3, "5.56e+06"},
    {555555.0f, chars_format::general, 3, "5.56e+05"},
    {55555.0f, chars_format::general, 3, "5.56e+04"},
    {5555.0f, chars_format::general, 3, "5.56e+03"}, // round to even
    {555.0f, chars_format::general, 3, "555"},
    {55.0f, chars_format::general, 3, "55"},
    {5.0f, chars_format::general, 3, "5"},
    {0x1p-3f, chars_format::general, 3, "0.125"},
    {0x1p-6f, chars_format::general, 3, "0.0156"},
    {0x1p-9f, chars_format::general, 3, "0.00195"},
    {0x1p-13f, chars_format::general, 3, "0.000122"},
    {0x1p-16f, chars_format::general, 3, "1.53e-05"},
    {0x1p-19f, chars_format::general, 3, "1.91e-06"},

    {5555555.0f, chars_format::general, 4, "5.556e+06"},
    {555555.0f, chars_format::general, 4, "5.556e+05"},
    {55555.0f, chars_format::general, 4, "5.556e+04"}, // round to even
    {5555.0f, chars_format::general, 4, "5555"},
    {555.0f, chars_format::general, 4, "555"},
    {55.0f, chars_format::general, 4, "55"},
    {5.0f, chars_format::general, 4, "5"},
    {0x1p-3f, chars_format::general, 4, "0.125"},
    {0x1p-6f, chars_format::general, 4, "0.01562"}, // round to even
    {0x1p-9f, chars_format::general, 4, "0.001953"},
    {0x1p-13f, chars_format::general, 4, "0.0001221"},
    {0x1p-16f, chars_format::general, 4, "1.526e-05"},
    {0x1p-19f, chars_format::general, 4, "1.907e-06"},

    {5555555.0f, chars_format::general, 5, "5.5556e+06"},
    {555555.0f, chars_format::general, 5, "5.5556e+05"}, // round to even
    {55555.0f, chars_format::general, 5, "55555"},
    {5555.0f, chars_format::general, 5, "5555"},
    {555.0f, chars_format::general, 5, "555"},
    {55.0f, chars_format::general, 5, "55"},
    {5.0f, chars_format::general, 5, "5"},
    {0x1p-3f, chars_format::general, 5, "0.125"},
    {0x1p-6f, chars_format::general, 5, "0.015625"},
    {0x1p-9f, chars_format::general, 5, "0.0019531"},
    {0x1p-13f, chars_format::general, 5, "0.00012207"},
    {0x1p-16f, chars_format::general, 5, "1.5259e-05"},
    {0x1p-19f, chars_format::general, 5, "1.9073e-06"},

    // Tricky corner cases.
    // In these scenarios, rounding can adjust the scientific exponent X,
    // which affects the transition between fixed notation and scientific notation.
    {999.999f, chars_format::general, 1, "1e+03"}, // "%.0e" is "1e+03"; X == 3
    {999.999f, chars_format::general, 2, "1e+03"}, // "%.1e" is "1.0e+03"; X == 3
    {999.999f, chars_format::general, 3, "1e+03"}, // "%.2e" is "1.00e+03"; X == 3
    {999.999f, chars_format::general, 4, "1000"}, // "%.3e" is "1.000e+03"; X == 3
    {999.999f, chars_format::general, 5, "1000"}, // "%.4e" is "1.0000e+03"; X == 3
    {999.999f, chars_format::general, 6, "999.999"}, // "%.5e" is "9.99999e+02"; X == 2

    {999.99f, chars_format::general, 1, "1e+03"},
    {999.99f, chars_format::general, 2, "1e+03"},
    {999.99f, chars_format::general, 3, "1e+03"},
    {999.99f, chars_format::general, 4, "1000"},
    {999.99f, chars_format::general, 5, "999.99"},
    {999.99f, chars_format::general, 6, "999.99"},

    // C11's Standardese is slightly vague about how to perform the trial formatting in scientific notation,
    // but the intention is to use precision P - 1, which is what's used when scientific notation is actually chosen.
    // This example verifies this behavior. Here, P == 3 performs trial formatting with "%.2e", triggering rounding.
    // That increases X to 3, forcing scientific notation to be chosen.
    // If P == 3 performed trial formatting with "%.3e", rounding wouldn't happen,
    // X would be 2, and fixed notation would be chosen.
    {999.9f, chars_format::general, 1, "1e+03"}, // "%.0e" is "1e+03"; X == 3
    {999.9f, chars_format::general, 2, "1e+03"}, // "%.1e" is "1.0e+03"; X == 3
    {999.9f, chars_format::general, 3, "1e+03"}, // "%.2e" is "1.00e+03"; X == 3; SPECIAL CORNER CASE
    {999.9f, chars_format::general, 4, "999.9"}, // "%.3e" is "9.999e+02"; X == 2
    {999.9f, chars_format::general, 5, "999.9"}, // "%.4e" is "9.9990e+02"; X == 2
    {999.9f, chars_format::general, 6, "999.9"}, // "%.5e" is "9.99900e+02"; X == 2

    {999.0f, chars_format::general, 1, "1e+03"},
    {999.0f, chars_format::general, 2, "1e+03"},
    {999.0f, chars_format::general, 3, "999"},
    {999.0f, chars_format::general, 4, "999"},
    {999.0f, chars_format::general, 5, "999"},
    {999.0f, chars_format::general, 6, "999"},

    {99.9999f, chars_format::general, 1, "1e+02"},
    {99.9999f, chars_format::general, 2, "1e+02"},
    {99.9999f, chars_format::general, 3, "100"},
    {99.9999f, chars_format::general, 4, "100"},
    {99.9999f, chars_format::general, 5, "100"},
    {99.9999f, chars_format::general, 6, "99.9999"},

    {99.999f, chars_format::general, 1, "1e+02"},
    {99.999f, chars_format::general, 2, "1e+02"},
    {99.999f, chars_format::general, 3, "100"},
    {99.999f, chars_format::general, 4, "100"},
    {99.999f, chars_format::general, 5, "99.999"},
    {99.999f, chars_format::general, 6, "99.999"},

    {99.99f, chars_format::general, 1, "1e+02"},
    {99.99f, chars_format::general, 2, "1e+02"},
    {99.99f, chars_format::general, 3, "100"},
    {99.99f, chars_format::general, 4, "99.99"},
    {99.99f, chars_format::general, 5, "99.99"},
    {99.99f, chars_format::general, 6, "99.99"},

    {99.9f, chars_format::general, 1, "1e+02"},
    {99.9f, chars_format::general, 2, "1e+02"},
    {99.9f, chars_format::general, 3, "99.9"},
    {99.9f, chars_format::general, 4, "99.9"},
    {99.9f, chars_format::general, 5, "99.9"},
    {99.9f, chars_format::general, 6, "99.9"},

    {99.0f, chars_format::general, 1, "1e+02"},
    {99.0f, chars_format::general, 2, "99"},
    {99.0f, chars_format::general, 3, "99"},
    {99.0f, chars_format::general, 4, "99"},
    {99.0f, chars_format::general, 5, "99"},
    {99.0f, chars_format::general, 6, "99"},

    {9.99999f, chars_format::general, 1, "1e+01"},
    {9.99999f, chars_format::general, 2, "10"},
    {9.99999f, chars_format::general, 3, "10"},
    {9.99999f, chars_format::general, 4, "10"},
    {9.99999f, chars_format::general, 5, "10"},
    {9.99999f, chars_format::general, 6, "9.99999"},

    {9.9999f, chars_format::general, 1, "1e+01"},
    {9.9999f, chars_format::general, 2, "10"},
    {9.9999f, chars_format::general, 3, "10"},
    {9.9999f, chars_format::general, 4, "10"},
    {9.9999f, chars_format::general, 5, "9.9999"},
    {9.9999f, chars_format::general, 6, "9.9999"},

    {9.999f, chars_format::general, 1, "1e+01"},
    {9.999f, chars_format::general, 2, "10"},
    {9.999f, chars_format::general, 3, "10"},
    {9.999f, chars_format::general, 4, "9.999"},
    {9.999f, chars_format::general, 5, "9.999"},
    {9.999f, chars_format::general, 6, "9.999"},

    {9.99f, chars_format::general, 1, "1e+01"},
    {9.99f, chars_format::general, 2, "10"},
    {9.99f, chars_format::general, 3, "9.99"},
    {9.99f, chars_format::general, 4, "9.99"},
    {9.99f, chars_format::general, 5, "9.99"},
    {9.99f, chars_format::general, 6, "9.99"},

    {9.9f, chars_format::general, 1, "1e+01"},
    {9.9f, chars_format::general, 2, "9.9"},
    {9.9f, chars_format::general, 3, "9.9"},
    {9.9f, chars_format::general, 4, "9.9"},
    {9.9f, chars_format::general, 5, "9.9"},
    {9.9f, chars_format::general, 6, "9.9"},

    {9.0f, chars_format::general, 1, "9"},
    {9.0f, chars_format::general, 2, "9"},
    {9.0f, chars_format::general, 3, "9"},
    {9.0f, chars_format::general, 4, "9"},
    {9.0f, chars_format::general, 5, "9"},
    {9.0f, chars_format::general, 6, "9"},

    {0.999999f, chars_format::general, 1, "1"},
    {0.999999f, chars_format::general, 2, "1"},
    {0.999999f, chars_format::general, 3, "1"},
    {0.999999f, chars_format::general, 4, "1"},
    {0.999999f, chars_format::general, 5, "1"},
    {0.999999f, chars_format::general, 6, "0.999999"},

    {0.99999f, chars_format::general, 1, "1"},
    {0.99999f, chars_format::general, 2, "1"},
    {0.99999f, chars_format::general, 3, "1"},
    {0.99999f, chars_format::general, 4, "1"},
    {0.99999f, chars_format::general, 5, "0.99999"},
    {0.99999f, chars_format::general, 6, "0.99999"},

    {0.9999f, chars_format::general, 1, "1"},
    {0.9999f, chars_format::general, 2, "1"},
    {0.9999f, chars_format::general, 3, "1"},
    {0.9999f, chars_format::general, 4, "0.9999"},
    {0.9999f, chars_format::general, 5, "0.9999"},
    {0.9999f, chars_format::general, 6, "0.9999"},

    {0.999f, chars_format::general, 1, "1"},
    {0.999f, chars_format::general, 2, "1"},
    {0.999f, chars_format::general, 3, "0.999"},
    {0.999f, chars_format::general, 4, "0.999"},
    {0.999f, chars_format::general, 5, "0.999"},
    {0.999f, chars_format::general, 6, "0.999"},

    {0.99f, chars_format::general, 1, "1"},
    {0.99f, chars_format::general, 2, "0.99"},
    {0.99f, chars_format::general, 3, "0.99"},
    {0.99f, chars_format::general, 4, "0.99"},
    {0.99f, chars_format::general, 5, "0.99"},
    {0.99f, chars_format::general, 6, "0.99"},

    {0.9f, chars_format::general, 1, "0.9"},
    {0.9f, chars_format::general, 2, "0.9"},
    {0.9f, chars_format::general, 3, "0.9"},
    {0.9f, chars_format::general, 4, "0.9"},
    {0.9f, chars_format::general, 5, "0.9"},
    {0.9f, chars_format::general, 6, "0.9"},

    {0.0999999f, chars_format::general, 1, "0.1"},
    {0.0999999f, chars_format::general, 2, "0.1"},
    {0.0999999f, chars_format::general, 3, "0.1"},
    {0.0999999f, chars_format::general, 4, "0.1"},
    {0.0999999f, chars_format::general, 5, "0.1"},
    {0.0999999f, chars_format::general, 6, "0.0999999"},

    {0.099999f, chars_format::general, 1, "0.1"},
    {0.099999f, chars_format::general, 2, "0.1"},
    {0.099999f, chars_format::general, 3, "0.1"},
    {0.099999f, chars_format::general, 4, "0.1"},
    {0.099999f, chars_format::general, 5, "0.099999"},
    {0.099999f, chars_format::general, 6, "0.099999"},

    {0.09999f, chars_format::general, 1, "0.1"},
    {0.09999f, chars_format::general, 2, "0.1"},
    {0.09999f, chars_format::general, 3, "0.1"},
    {0.09999f, chars_format::general, 4, "0.09999"},
    {0.09999f, chars_format::general, 5, "0.09999"},
    {0.09999f, chars_format::general, 6, "0.09999"},

    {0.0999f, chars_format::general, 1, "0.1"},
    {0.0999f, chars_format::general, 2, "0.1"},
    {0.0999f, chars_format::general, 3, "0.0999"},
    {0.0999f, chars_format::general, 4, "0.0999"},
    {0.0999f, chars_format::general, 5, "0.0999"},
    {0.0999f, chars_format::general, 6, "0.0999"},

    {0.099f, chars_format::general, 1, "0.1"},
    {0.099f, chars_format::general, 2, "0.099"},
    {0.099f, chars_format::general, 3, "0.099"},
    {0.099f, chars_format::general, 4, "0.099"},
    {0.099f, chars_format::general, 5, "0.099"},
    {0.099f, chars_format::general, 6, "0.099"},

    {0.09f, chars_format::general, 1, "0.09"},
    {0.09f, chars_format::general, 2, "0.09"},
    {0.09f, chars_format::general, 3, "0.09"},
    {0.09f, chars_format::general, 4, "0.09"},
    {0.09f, chars_format::general, 5, "0.09"},
    {0.09f, chars_format::general, 6, "0.09"},

    {0.00999999f, chars_format::general, 1, "0.01"},
    {0.00999999f, chars_format::general, 2, "0.01"},
    {0.00999999f, chars_format::general, 3, "0.01"},
    {0.00999999f, chars_format::general, 4, "0.01"},
    {0.00999999f, chars_format::general, 5, "0.01"},
    {0.00999999f, chars_format::general, 6, "0.00999999"},

    {0.0099999f, chars_format::general, 1, "0.01"},
    {0.0099999f, chars_format::general, 2, "0.01"},
    {0.0099999f, chars_format::general, 3, "0.01"},
    {0.0099999f, chars_format::general, 4, "0.01"},
    {0.0099999f, chars_format::general, 5, "0.0099999"},
    {0.0099999f, chars_format::general, 6, "0.0099999"},

    {0.009999f, chars_format::general, 1, "0.01"},
    {0.009999f, chars_format::general, 2, "0.01"},
    {0.009999f, chars_format::general, 3, "0.01"},
    {0.009999f, chars_format::general, 4, "0.009999"},
    {0.009999f, chars_format::general, 5, "0.009999"},
    {0.009999f, chars_format::general, 6, "0.009999"},

    {0.00999f, chars_format::general, 1, "0.01"},
    {0.00999f, chars_format::general, 2, "0.01"},
    {0.00999f, chars_format::general, 3, "0.00999"},
    {0.00999f, chars_format::general, 4, "0.00999"},
    {0.00999f, chars_format::general, 5, "0.00999"},
    {0.00999f, chars_format::general, 6, "0.00999"},

    {0.0099f, chars_format::general, 1, "0.01"},
    {0.0099f, chars_format::general, 2, "0.0099"},
    {0.0099f, chars_format::general, 3, "0.0099"},
    {0.0099f, chars_format::general, 4, "0.0099"},
    {0.0099f, chars_format::general, 5, "0.0099"},
    {0.0099f, chars_format::general, 6, "0.0099"},

    {0.009f, chars_format::general, 1, "0.009"},
    {0.009f, chars_format::general, 2, "0.009"},
    {0.009f, chars_format::general, 3, "0.009"},
    {0.009f, chars_format::general, 4, "0.009"},
    {0.009f, chars_format::general, 5, "0.009"},
    {0.009f, chars_format::general, 6, "0.009"},

    {0.000999999f, chars_format::general, 1, "0.001"},
    {0.000999999f, chars_format::general, 2, "0.001"},
    {0.000999999f, chars_format::general, 3, "0.001"},
    {0.000999999f, chars_format::general, 4, "0.001"},
    {0.000999999f, chars_format::general, 5, "0.001"},
    {0.000999999f, chars_format::general, 6, "0.000999999"},

    {0.00099999f, chars_format::general, 1, "0.001"},
    {0.00099999f, chars_format::general, 2, "0.001"},
    {0.00099999f, chars_format::general, 3, "0.001"},
    {0.00099999f, chars_format::general, 4, "0.001"},
    {0.00099999f, chars_format::general, 5, "0.00099999"},
    {0.00099999f, chars_format::general, 6, "0.00099999"},

    {0.0009999f, chars_format::general, 1, "0.001"},
    {0.0009999f, chars_format::general, 2, "0.001"},
    {0.0009999f, chars_format::general, 3, "0.001"},
    {0.0009999f, chars_format::general, 4, "0.0009999"},
    {0.0009999f, chars_format::general, 5, "0.0009999"},
    {0.0009999f, chars_format::general, 6, "0.0009999"},

    {0.000999f, chars_format::general, 1, "0.001"},
    {0.000999f, chars_format::general, 2, "0.001"},
    {0.000999f, chars_format::general, 3, "0.000999"},
    {0.000999f, chars_format::general, 4, "0.000999"},
    {0.000999f, chars_format::general, 5, "0.000999"},
    {0.000999f, chars_format::general, 6, "0.000999"},

    {0.00099f, chars_format::general, 1, "0.001"},
    {0.00099f, chars_format::general, 2, "0.00099"},
    {0.00099f, chars_format::general, 3, "0.00099"},
    {0.00099f, chars_format::general, 4, "0.00099"},
    {0.00099f, chars_format::general, 5, "0.00099"},
    {0.00099f, chars_format::general, 6, "0.00099"},

    {0.0009f, chars_format::general, 1, "0.0009"},
    {0.0009f, chars_format::general, 2, "0.0009"},
    {0.0009f, chars_format::general, 3, "0.0009"},
    {0.0009f, chars_format::general, 4, "0.0009"},
    {0.0009f, chars_format::general, 5, "0.0009"},
    {0.0009f, chars_format::general, 6, "0.0009"},

    // Having a scientific exponent X == -5 triggers scientific notation.
    // If rounding adjusts this to X == -4, then fixed notation will be selected.
    {0.0000999999f, chars_format::general, 1, "0.0001"},
    {0.0000999999f, chars_format::general, 2, "0.0001"},
    {0.0000999999f, chars_format::general, 3, "0.0001"},
    {0.0000999999f, chars_format::general, 4, "0.0001"},
    {0.0000999999f, chars_format::general, 5, "0.0001"},
    {0.0000999999f, chars_format::general, 6, "9.99999e-05"},

    {0.000099999f, chars_format::general, 1, "0.0001"},
    {0.000099999f, chars_format::general, 2, "0.0001"},
    {0.000099999f, chars_format::general, 3, "0.0001"},
    {0.000099999f, chars_format::general, 4, "0.0001"},
    {0.000099999f, chars_format::general, 5, "9.9999e-05"},
    {0.000099999f, chars_format::general, 6, "9.9999e-05"},

    {0.00009999f, chars_format::general, 1, "0.0001"},
    {0.00009999f, chars_format::general, 2, "0.0001"},
    {0.00009999f, chars_format::general, 3, "0.0001"},
    {0.00009999f, chars_format::general, 4, "9.999e-05"},
    {0.00009999f, chars_format::general, 5, "9.999e-05"},
    {0.00009999f, chars_format::general, 6, "9.999e-05"},

    {0.0000999f, chars_format::general, 1, "0.0001"},
    {0.0000999f, chars_format::general, 2, "0.0001"},
    {0.0000999f, chars_format::general, 3, "9.99e-05"},
    {0.0000999f, chars_format::general, 4, "9.99e-05"},
    {0.0000999f, chars_format::general, 5, "9.99e-05"},
    {0.0000999f, chars_format::general, 6, "9.99e-05"},

    {0.000099f, chars_format::general, 1, "0.0001"},
    {0.000099f, chars_format::general, 2, "9.9e-05"},
    {0.000099f, chars_format::general, 3, "9.9e-05"},
    {0.000099f, chars_format::general, 4, "9.9e-05"},
    {0.000099f, chars_format::general, 5, "9.9e-05"},
    {0.000099f, chars_format::general, 6, "9.9e-05"},

    {0.00009f, chars_format::general, 1, "9e-05"},
    {0.00009f, chars_format::general, 2, "9e-05"},
    {0.00009f, chars_format::general, 3, "9e-05"},
    {0.00009f, chars_format::general, 4, "9e-05"},
    {0.00009f, chars_format::general, 5, "9e-05"},
    {0.00009f, chars_format::general, 6, "9e-05"},

    // Rounding test cases without exponent-adjusting behavior.
    {2999.999f, chars_format::general, 1, "3e+03"},
    {2999.999f, chars_format::general, 2, "3e+03"},
    {2999.999f, chars_format::general, 3, "3e+03"},
    {2999.999f, chars_format::general, 4, "3000"},
    {2999.999f, chars_format::general, 5, "3000"},
    {2999.999f, chars_format::general, 6, "3000"},

    {2999.99f, chars_format::general, 1, "3e+03"},
    {2999.99f, chars_format::general, 2, "3e+03"},
    {2999.99f, chars_format::general, 3, "3e+03"},
    {2999.99f, chars_format::general, 4, "3000"},
    {2999.99f, chars_format::general, 5, "3000"},
    {2999.99f, chars_format::general, 6, "2999.99"},

    {2999.9f, chars_format::general, 1, "3e+03"},
    {2999.9f, chars_format::general, 2, "3e+03"},
    {2999.9f, chars_format::general, 3, "3e+03"},
    {2999.9f, chars_format::general, 4, "3000"},
    {2999.9f, chars_format::general, 5, "2999.9"},
    {2999.9f, chars_format::general, 6, "2999.9"},

    {2999.0f, chars_format::general, 1, "3e+03"},
    {2999.0f, chars_format::general, 2, "3e+03"},
    {2999.0f, chars_format::general, 3, "3e+03"},
    {2999.0f, chars_format::general, 4, "2999"},
    {2999.0f, chars_format::general, 5, "2999"},
    {2999.0f, chars_format::general, 6, "2999"},

    {299.999f, chars_format::general, 1, "3e+02"},
    {299.999f, chars_format::general, 2, "3e+02"},
    {299.999f, chars_format::general, 3, "300"},
    {299.999f, chars_format::general, 4, "300"},
    {299.999f, chars_format::general, 5, "300"},
    {299.999f, chars_format::general, 6, "299.999"},

    {299.99f, chars_format::general, 1, "3e+02"},
    {299.99f, chars_format::general, 2, "3e+02"},
    {299.99f, chars_format::general, 3, "300"},
    {299.99f, chars_format::general, 4, "300"},
    {299.99f, chars_format::general, 5, "299.99"},
    {299.99f, chars_format::general, 6, "299.99"},

    {299.9f, chars_format::general, 1, "3e+02"},
    {299.9f, chars_format::general, 2, "3e+02"},
    {299.9f, chars_format::general, 3, "300"},
    {299.9f, chars_format::general, 4, "299.9"},
    {299.9f, chars_format::general, 5, "299.9"},
    {299.9f, chars_format::general, 6, "299.9"},

    {299.0f, chars_format::general, 1, "3e+02"},
    {299.0f, chars_format::general, 2, "3e+02"},
    {299.0f, chars_format::general, 3, "299"},
    {299.0f, chars_format::general, 4, "299"},
    {299.0f, chars_format::general, 5, "299"},
    {299.0f, chars_format::general, 6, "299"},

    {29.999f, chars_format::general, 1, "3e+01"},
    {29.999f, chars_format::general, 2, "30"},
    {29.999f, chars_format::general, 3, "30"},
    {29.999f, chars_format::general, 4, "30"},
    {29.999f, chars_format::general, 5, "29.999"},
    {29.999f, chars_format::general, 6, "29.999"},

    {29.99f, chars_format::general, 1, "3e+01"},
    {29.99f, chars_format::general, 2, "30"},
    {29.99f, chars_format::general, 3, "30"},
    {29.99f, chars_format::general, 4, "29.99"},
    {29.99f, chars_format::general, 5, "29.99"},
    {29.99f, chars_format::general, 6, "29.99"},

    {29.9f, chars_format::general, 1, "3e+01"},
    {29.9f, chars_format::general, 2, "30"},
    {29.9f, chars_format::general, 3, "29.9"},
    {29.9f, chars_format::general, 4, "29.9"},
    {29.9f, chars_format::general, 5, "29.9"},
    {29.9f, chars_format::general, 6, "29.9"},

    {29.0f, chars_format::general, 1, "3e+01"},
    {29.0f, chars_format::general, 2, "29"},
    {29.0f, chars_format::general, 3, "29"},
    {29.0f, chars_format::general, 4, "29"},
    {29.0f, chars_format::general, 5, "29"},
    {29.0f, chars_format::general, 6, "29"},

    {2.999f, chars_format::general, 1, "3"},
    {2.999f, chars_format::general, 2, "3"},
    {2.999f, chars_format::general, 3, "3"},
    {2.999f, chars_format::general, 4, "2.999"},
    {2.999f, chars_format::general, 5, "2.999"},
    {2.999f, chars_format::general, 6, "2.999"},

    {2.99f, chars_format::general, 1, "3"},
    {2.99f, chars_format::general, 2, "3"},
    {2.99f, chars_format::general, 3, "2.99"},
    {2.99f, chars_format::general, 4, "2.99"},
    {2.99f, chars_format::general, 5, "2.99"},
    {2.99f, chars_format::general, 6, "2.99"},

    {2.9f, chars_format::general, 1, "3"},
    {2.9f, chars_format::general, 2, "2.9"},
    {2.9f, chars_format::general, 3, "2.9"},
    {2.9f, chars_format::general, 4, "2.9"},
    {2.9f, chars_format::general, 5, "2.9"},
    {2.9f, chars_format::general, 6, "2.9"},

    {2.0f, chars_format::general, 1, "2"},
    {2.0f, chars_format::general, 2, "2"},
    {2.0f, chars_format::general, 3, "2"},
    {2.0f, chars_format::general, 4, "2"},
    {2.0f, chars_format::general, 5, "2"},
    {2.0f, chars_format::general, 6, "2"},

    {0.2999f, chars_format::general, 1, "0.3"},
    {0.2999f, chars_format::general, 2, "0.3"},
    {0.2999f, chars_format::general, 3, "0.3"},
    {0.2999f, chars_format::general, 4, "0.2999"},
    {0.2999f, chars_format::general, 5, "0.2999"},
    {0.2999f, chars_format::general, 6, "0.2999"},

    {0.299f, chars_format::general, 1, "0.3"},
    {0.299f, chars_format::general, 2, "0.3"},
    {0.299f, chars_format::general, 3, "0.299"},
    {0.299f, chars_format::general, 4, "0.299"},
    {0.299f, chars_format::general, 5, "0.299"},
    {0.299f, chars_format::general, 6, "0.299"},

    {0.29f, chars_format::general, 1, "0.3"},
    {0.29f, chars_format::general, 2, "0.29"},
    {0.29f, chars_format::general, 3, "0.29"},
    {0.29f, chars_format::general, 4, "0.29"},
    {0.29f, chars_format::general, 5, "0.29"},
    {0.29f, chars_format::general, 6, "0.29"},

    {0.2f, chars_format::general, 1, "0.2"},
    {0.2f, chars_format::general, 2, "0.2"},
    {0.2f, chars_format::general, 3, "0.2"},
    {0.2f, chars_format::general, 4, "0.2"},
    {0.2f, chars_format::general, 5, "0.2"},
    {0.2f, chars_format::general, 6, "0.2"},

    {0.02999f, chars_format::general, 1, "0.03"},
    {0.02999f, chars_format::general, 2, "0.03"},
    {0.02999f, chars_format::general, 3, "0.03"},
    {0.02999f, chars_format::general, 4, "0.02999"},
    {0.02999f, chars_format::general, 5, "0.02999"},
    {0.02999f, chars_format::general, 6, "0.02999"},

    {0.0299f, chars_format::general, 1, "0.03"},
    {0.0299f, chars_format::general, 2, "0.03"},
    {0.0299f, chars_format::general, 3, "0.0299"},
    {0.0299f, chars_format::general, 4, "0.0299"},
    {0.0299f, chars_format::general, 5, "0.0299"},
    {0.0299f, chars_format::general, 6, "0.0299"},

    {0.029f, chars_format::general, 1, "0.03"},
    {0.029f, chars_format::general, 2, "0.029"},
    {0.029f, chars_format::general, 3, "0.029"},
    {0.029f, chars_format::general, 4, "0.029"},
    {0.029f, chars_format::general, 5, "0.029"},
    {0.029f, chars_format::general, 6, "0.029"},

    {0.02f, chars_format::general, 1, "0.02"},
    {0.02f, chars_format::general, 2, "0.02"},
    {0.02f, chars_format::general, 3, "0.02"},
    {0.02f, chars_format::general, 4, "0.02"},
    {0.02f, chars_format::general, 5, "0.02"},
    {0.02f, chars_format::general, 6, "0.02"},

    {0.002999f, chars_format::general, 1, "0.003"},
    {0.002999f, chars_format::general, 2, "0.003"},
    {0.002999f, chars_format::general, 3, "0.003"},
    {0.002999f, chars_format::general, 4, "0.002999"},
    {0.002999f, chars_format::general, 5, "0.002999"},
    {0.002999f, chars_format::general, 6, "0.002999"},

    {0.00299f, chars_format::general, 1, "0.003"},
    {0.00299f, chars_format::general, 2, "0.003"},
    {0.00299f, chars_format::general, 3, "0.00299"},
    {0.00299f, chars_format::general, 4, "0.00299"},
    {0.00299f, chars_format::general, 5, "0.00299"},
    {0.00299f, chars_format::general, 6, "0.00299"},

    {0.0029f, chars_format::general, 1, "0.003"},
    {0.0029f, chars_format::general, 2, "0.0029"},
    {0.0029f, chars_format::general, 3, "0.0029"},
    {0.0029f, chars_format::general, 4, "0.0029"},
    {0.0029f, chars_format::general, 5, "0.0029"},
    {0.0029f, chars_format::general, 6, "0.0029"},

    {0.002f, chars_format::general, 1, "0.002"},
    {0.002f, chars_format::general, 2, "0.002"},
    {0.002f, chars_format::general, 3, "0.002"},
    {0.002f, chars_format::general, 4, "0.002"},
    {0.002f, chars_format::general, 5, "0.002"},
    {0.002f, chars_format::general, 6, "0.002"},

    {0.0002999f, chars_format::general, 1, "0.0003"},
    {0.0002999f, chars_format::general, 2, "0.0003"},
    {0.0002999f, chars_format::general, 3, "0.0003"},
    {0.0002999f, chars_format::general, 4, "0.0002999"},
    {0.0002999f, chars_format::general, 5, "0.0002999"},
    {0.0002999f, chars_format::general, 6, "0.0002999"},

    {0.000299f, chars_format::general, 1, "0.0003"},
    {0.000299f, chars_format::general, 2, "0.0003"},
    {0.000299f, chars_format::general, 3, "0.000299"},
    {0.000299f, chars_format::general, 4, "0.000299"},
    {0.000299f, chars_format::general, 5, "0.000299"},
    {0.000299f, chars_format::general, 6, "0.000299"},

    {0.00029f, chars_format::general, 1, "0.0003"},
    {0.00029f, chars_format::general, 2, "0.00029"},
    {0.00029f, chars_format::general, 3, "0.00029"},
    {0.00029f, chars_format::general, 4, "0.00029"},
    {0.00029f, chars_format::general, 5, "0.00029"},
    {0.00029f, chars_format::general, 6, "0.00029"},

    {0.0002f, chars_format::general, 1, "0.0002"},
    {0.0002f, chars_format::general, 2, "0.0002"},
    {0.0002f, chars_format::general, 3, "0.0002"},
    {0.0002f, chars_format::general, 4, "0.0002"},
    {0.0002f, chars_format::general, 5, "0.0002"},
    {0.0002f, chars_format::general, 6, "0.0002"},

    {0.00002999f, chars_format::general, 1, "3e-05"},
    {0.00002999f, chars_format::general, 2, "3e-05"},
    {0.00002999f, chars_format::general, 3, "3e-05"},
    {0.00002999f, chars_format::general, 4, "2.999e-05"},
    {0.00002999f, chars_format::general, 5, "2.999e-05"},
    {0.00002999f, chars_format::general, 6, "2.999e-05"},

    {0.0000299f, chars_format::general, 1, "3e-05"},
    {0.0000299f, chars_format::general, 2, "3e-05"},
    {0.0000299f, chars_format::general, 3, "2.99e-05"},
    {0.0000299f, chars_format::general, 4, "2.99e-05"},
    {0.0000299f, chars_format::general, 5, "2.99e-05"},
    {0.0000299f, chars_format::general, 6, "2.99e-05"},

    {0.000029f, chars_format::general, 1, "3e-05"},
    {0.000029f, chars_format::general, 2, "2.9e-05"},
    {0.000029f, chars_format::general, 3, "2.9e-05"},
    {0.000029f, chars_format::general, 4, "2.9e-05"},
    {0.000029f, chars_format::general, 5, "2.9e-05"},
    {0.000029f, chars_format::general, 6, "2.9e-05"},

    {0.00002f, chars_format::general, 1, "2e-05"},
    {0.00002f, chars_format::general, 2, "2e-05"},
    {0.00002f, chars_format::general, 3, "2e-05"},
    {0.00002f, chars_format::general, 4, "2e-05"},
    {0.00002f, chars_format::general, 5, "2e-05"},
    {0.00002f, chars_format::general, 6, "2e-05"},

    // Test the transitions between values of the scientific exponent X.
    // For brevity, we avoid testing all possible combinations of P and X. Instead, we test:
    // * All values of P where some X can be affected by rounding. (For float, this is [1, 7].)
    // * P == 25, which is arbitrary.
    // * P == numeric_limits::max_exponent10 + 1. This selects fixed notation for numeric_limits::max(),
    //   so it's the largest interesting value of P.
    // * Finally, we test the transitions around X == P - 1, ensuring that we can recognize every value of X.
    {0x1.8e7578p-14f, chars_format::general, 1, "9e-05"},
    {0x1.8e757ap-14f, chars_format::general, 1, "0.0001"},
    {0x1.f212d6p-11f, chars_format::general, 1, "0.0009"},
    {0x1.f212d8p-11f, chars_format::general, 1, "0.001"},
    {0x1.374bc6p-7f, chars_format::general, 1, "0.009"},
    {0x1.374bc8p-7f, chars_format::general, 1, "0.01"},
    {0x1.851eb8p-4f, chars_format::general, 1, "0.09"},
    {0x1.851ebap-4f, chars_format::general, 1, "0.1"},
    {0x1.e66666p-1f, chars_format::general, 1, "0.9"},
    {0x1.e66668p-1f, chars_format::general, 1, "1"},
    {0x1.2ffffep+3f, chars_format::general, 1, "9"},
    {0x1.300000p+3f, chars_format::general, 1, "1e+01"},
    {0x1.a1554ep-14f, chars_format::general, 2, "9.9e-05"},
    {0x1.a15550p-14f, chars_format::general, 2, "0.0001"},
    {0x1.04d550p-10f, chars_format::general, 2, "0.00099"},
    {0x1.04d552p-10f, chars_format::general, 2, "0.001"},
    {0x1.460aa6p-7f, chars_format::general, 2, "0.0099"},
    {0x1.460aa8p-7f, chars_format::general, 2, "0.01"},
    {0x1.978d4ep-4f, chars_format::general, 2, "0.099"},
    {0x1.978d50p-4f, chars_format::general, 2, "0.1"},
    {0x1.fd70a2p-1f, chars_format::general, 2, "0.99"},
    {0x1.fd70a4p-1f, chars_format::general, 2, "1"},
    {0x1.3e6666p+3f, chars_format::general, 2, "9.9"},
    {0x1.3e6668p+3f, chars_format::general, 2, "10"},
    {0x1.8dfffep+6f, chars_format::general, 2, "99"},
    {0x1.8e0000p+6f, chars_format::general, 2, "1e+02"},
    {0x1.a3387ep-14f, chars_format::general, 3, "9.99e-05"},
    {0x1.a33880p-14f, chars_format::general, 3, "0.0001"},
    {0x1.06034ep-10f, chars_format::general, 3, "0.000999"},
    {0x1.060350p-10f, chars_format::general, 3, "0.001"},
    {0x1.478422p-7f, chars_format::general, 3, "0.00999"},
    {0x1.478424p-7f, chars_format::general, 3, "0.01"},
    {0x1.99652ap-4f, chars_format::general, 3, "0.0999"},
    {0x1.99652cp-4f, chars_format::general, 3, "0.1"},
    {0x1.ffbe76p-1f, chars_format::general, 3, "0.999"},
    {0x1.ffbe78p-1f, chars_format::general, 3, "1"},
    {0x1.3fd70ap+3f, chars_format::general, 3, "9.99"},
    {0x1.3fd70cp+3f, chars_format::general, 3, "10"},
    {0x1.8fccccp+6f, chars_format::general, 3, "99.9"},
    {0x1.8fcccep+6f, chars_format::general, 3, "100"},
    {0x1.f3bffep+9f, chars_format::general, 3, "999"},
    {0x1.f3c000p+9f, chars_format::general, 3, "1e+03"},
    {0x1.a368d0p-14f, chars_format::general, 4, "9.999e-05"},
    {0x1.a368d2p-14f, chars_format::general, 4, "0.0001"},
    {0x1.062182p-10f, chars_format::general, 4, "0.0009999"},
    {0x1.062184p-10f, chars_format::general, 4, "0.001"},
    {0x1.47a9e2p-7f, chars_format::general, 4, "0.009999"},
    {0x1.47a9e4p-7f, chars_format::general, 4, "0.01"},
    {0x1.99945ap-4f, chars_format::general, 4, "0.09999"},
    {0x1.99945cp-4f, chars_format::general, 4, "0.1"},
    {0x1.fff972p-1f, chars_format::general, 4, "0.9999"},
    {0x1.fff974p-1f, chars_format::general, 4, "1"},
    {0x1.3ffbe6p+3f, chars_format::general, 4, "9.999"},
    {0x1.3ffbe8p+3f, chars_format::general, 4, "10"},
    {0x1.8ffae0p+6f, chars_format::general, 4, "99.99"},
    {0x1.8ffae2p+6f, chars_format::general, 4, "100"},
    {0x1.f3f998p+9f, chars_format::general, 4, "999.9"},
    {0x1.f3f99ap+9f, chars_format::general, 4, "1000"},
    {0x1.387bfep+13f, chars_format::general, 4, "9999"},
    {0x1.387c00p+13f, chars_format::general, 4, "1e+04"},
    {0x1.a36da4p-14f, chars_format::general, 5, "9.9999e-05"},
    {0x1.a36da6p-14f, chars_format::general, 5, "0.0001"},
    {0x1.062486p-10f, chars_format::general, 5, "0.00099999"},
    {0x1.062488p-10f, chars_format::general, 5, "0.001"},
    {0x1.47ada8p-7f, chars_format::general, 5, "0.0099999"},
    {0x1.47adaap-7f, chars_format::general, 5, "0.01"},
    {0x1.999912p-4f, chars_format::general, 5, "0.099999"},
    {0x1.999914p-4f, chars_format::general, 5, "0.1"},
    {0x1.ffff58p-1f, chars_format::general, 5, "0.99999"},
    {0x1.ffff5ap-1f, chars_format::general, 5, "1"},
    {0x1.3fff96p+3f, chars_format::general, 5, "9.9999"},
    {0x1.3fff98p+3f, chars_format::general, 5, "10"},
    {0x1.8fff7cp+6f, chars_format::general, 5, "99.999"},
    {0x1.8fff7ep+6f, chars_format::general, 5, "100"},
    {0x1.f3ff5cp+9f, chars_format::general, 5, "999.99"},
    {0x1.f3ff5ep+9f, chars_format::general, 5, "1000"},
    {0x1.387f98p+13f, chars_format::general, 5, "9999.9"},
    {0x1.387f9ap+13f, chars_format::general, 5, "10000"},
    {0x1.869f7ep+16f, chars_format::general, 5, "99999"},
    {0x1.869f80p+16f, chars_format::general, 5, "1e+05"},
    {0x1.a36e20p-14f, chars_format::general, 6, "9.99999e-05"},
    {0x1.a36e22p-14f, chars_format::general, 6, "0.0001"},
    {0x1.0624d4p-10f, chars_format::general, 6, "0.000999999"},
    {0x1.0624d6p-10f, chars_format::general, 6, "0.001"},
    {0x1.47ae08p-7f, chars_format::general, 6, "0.00999999"},
    {0x1.47ae0ap-7f, chars_format::general, 6, "0.01"},
    {0x1.99998cp-4f, chars_format::general, 6, "0.0999999"},
    {0x1.99998ep-4f, chars_format::general, 6, "0.1"},
    {0x1.ffffeep-1f, chars_format::general, 6, "0.999999"},
    {0x1.fffff0p-1f, chars_format::general, 6, "1"},
    {0x1.3ffff4p+3f, chars_format::general, 6, "9.99999"},
    {0x1.3ffff6p+3f, chars_format::general, 6, "10"},
    {0x1.8ffff2p+6f, chars_format::general, 6, "99.9999"},
    {0x1.8ffff4p+6f, chars_format::general, 6, "100"},
    {0x1.f3ffeep+9f, chars_format::general, 6, "999.999"},
    {0x1.f3fff0p+9f, chars_format::general, 6, "1000"},
    {0x1.387ff4p+13f, chars_format::general, 6, "9999.99"},
    {0x1.387ff6p+13f, chars_format::general, 6, "10000"},
    {0x1.869ff2p+16f, chars_format::general, 6, "99999.9"},
    {0x1.869ff4p+16f, chars_format::general, 6, "100000"},
    {0x1.e847eep+19f, chars_format::general, 6, "999999"},
    {0x1.e847f0p+19f, chars_format::general, 6, "1e+06"},
    {0x1.a36e2cp-14f, chars_format::general, 7, "9.999999e-05"},
    {0x1.a36e2ep-14f, chars_format::general, 7, "0.0001"},
    {0x1.0624dcp-10f, chars_format::general, 7, "0.0009999999"},
    {0x1.0624dep-10f, chars_format::general, 7, "0.001"},
    {0x1.47ae12p-7f, chars_format::general, 7, "0.009999999"},
    {0x1.47ae14p-7f, chars_format::general, 7, "0.01"},
    {0x1.999998p-4f, chars_format::general, 7, "0.09999999"},
    {0x1.99999ap-4f, chars_format::general, 7, "0.1"},
    {0x1.fffffep-1f, chars_format::general, 7, "0.9999999"},
    {0x1.000000p+0f, chars_format::general, 7, "1"},
    {0x1.3ffffep+3f, chars_format::general, 7, "9.999999"},
    {0x1.400000p+3f, chars_format::general, 7, "10"},
    {0x1.8ffffep+6f, chars_format::general, 7, "99.99999"},
    {0x1.900000p+6f, chars_format::general, 7, "100"},
    {0x1.f3fffep+9f, chars_format::general, 7, "999.9999"},
    {0x1.f40000p+9f, chars_format::general, 7, "1000"},
    {0x1.387ffep+13f, chars_format::general, 7, "9999.999"},
    {0x1.388000p+13f, chars_format::general, 7, "10000"},
    {0x1.869ffep+16f, chars_format::general, 7, "99999.99"},
    {0x1.86a000p+16f, chars_format::general, 7, "100000"},
    {0x1.e847fep+19f, chars_format::general, 7, "999999.9"},
    {0x1.e84800p+19f, chars_format::general, 7, "1000000"},
    {0x1.312cfep+23f, chars_format::general, 7, "9999999"},
    {0x1.312d00p+23f, chars_format::general, 7, "1e+07"},
    {0x1.7d783ep+26f, chars_format::general, 8, "99999992"},
    {0x1.7d7840p+26f, chars_format::general, 8, "1e+08"},
    {0x1.dcd64ep+29f, chars_format::general, 9, "999999936"},
    {0x1.dcd650p+29f, chars_format::general, 9, "1e+09"},
    {0x1.2a05f0p+33f, chars_format::general, 10, "9999998976"},
    {0x1.2a05f2p+33f, chars_format::general, 10, "1e+10"},
    {0x1.74876ep+36f, chars_format::general, 11, "99999997952"},
    {0x1.748770p+36f, chars_format::general, 11, "1.0000000614e+11"},
    {0x1.d1a94ap+39f, chars_format::general, 12, "999999995904"},
    {0x1.d1a94cp+39f, chars_format::general, 12, "1.00000006144e+12"},
    {0x1.2309cep+43f, chars_format::general, 13, "9999999827968"},
    {0x1.2309d0p+43f, chars_format::general, 13, "1.000000087654e+13"},
    {0x1.6bcc40p+46f, chars_format::general, 14, "99999991988224"},
    {0x1.6bcc42p+46f, chars_format::general, 14, "1.0000000037683e+14"},
    {0x1.c6bf52p+49f, chars_format::general, 15, "999999986991104"},
    {0x1.c6bf54p+49f, chars_format::general, 15, "1.00000005409997e+15"},
    {0x1.1c3792p+53f, chars_format::general, 16, "9999999198822400"},
    {0x1.1c3794p+53f, chars_format::general, 16, "1.000000027256422e+16"},
    {0x1.634578p+56f, chars_format::general, 17, "99999998430674944"},
    {0x1.63457ap+56f, chars_format::general, 17, "1.0000000702060954e+17"},
    {0x1.bc16d6p+59f, chars_format::general, 18, "999999984306749440"},
    {0x1.bc16d8p+59f, chars_format::general, 18, "1.00000005302622618e+18"},
    {0x1.158e46p+63f, chars_format::general, 19, "9999999980506447872"},
    {0x1.158e48p+63f, chars_format::general, 19, "1.000000108001807565e+19"},
    {0x1.5af1d6p+66f, chars_format::general, 20, "99999993207994712064"},
    {0x1.5af1d8p+66f, chars_format::general, 20, "1.0000000200408773427e+20"},
    {0x1.b1ae4cp+69f, chars_format::general, 21, "999999949672133165056"},
    {0x1.b1ae4ep+69f, chars_format::general, 21, "1.00000002004087734272e+21"},
    {0x1.0f0cf0p+73f, chars_format::general, 22, "9999999778196308361216"},
    {0x1.0f0cf2p+73f, chars_format::general, 22, "1.000000090409621520384e+22"},
    {0x1.52d02cp+76f, chars_format::general, 23, "99999997781963083612160"},
    {0x1.52d02ep+76f, chars_format::general, 23, "1.0000000678916233835315e+23"},
    {0x1.a78436p+79f, chars_format::general, 24, "999999941790833817157632"},
    {0x1.a78438p+79f, chars_format::general, 24, "1.00000001384842785508557e+24"},
    {0x1.a36e2ep-14f, chars_format::general, 25, "9.999999747378751635551453e-05"},
    {0x1.a36e30p-14f, chars_format::general, 25, "0.0001000000047497451305389404"},
    {0x1.0624dcp-10f, chars_format::general, 25, "0.0009999999310821294784545898"},
    {0x1.0624dep-10f, chars_format::general, 25, "0.001000000047497451305389404"},
    {0x1.47ae14p-7f, chars_format::general, 25, "0.009999999776482582092285156"},
    {0x1.47ae16p-7f, chars_format::general, 25, "0.01000000070780515670776367"},
    {0x1.999998p-4f, chars_format::general, 25, "0.0999999940395355224609375"},
    {0x1.99999ap-4f, chars_format::general, 25, "0.1000000014901161193847656"},
    {0x1.fffffep-1f, chars_format::general, 25, "0.999999940395355224609375"},
    {0x1.000000p+0f, chars_format::general, 25, "1"},
    {0x1.3ffffep+3f, chars_format::general, 25, "9.99999904632568359375"},
    {0x1.400000p+3f, chars_format::general, 25, "10"},
    {0x1.8ffffep+6f, chars_format::general, 25, "99.99999237060546875"},
    {0x1.900000p+6f, chars_format::general, 25, "100"},
    {0x1.f3fffep+9f, chars_format::general, 25, "999.99993896484375"},
    {0x1.f40000p+9f, chars_format::general, 25, "1000"},
    {0x1.387ffep+13f, chars_format::general, 25, "9999.9990234375"},
    {0x1.388000p+13f, chars_format::general, 25, "10000"},
    {0x1.869ffep+16f, chars_format::general, 25, "99999.9921875"},
    {0x1.86a000p+16f, chars_format::general, 25, "100000"},
    {0x1.e847fep+19f, chars_format::general, 25, "999999.9375"},
    {0x1.e84800p+19f, chars_format::general, 25, "1000000"},
    {0x1.312cfep+23f, chars_format::general, 25, "9999999"},
    {0x1.312d00p+23f, chars_format::general, 25, "10000000"},
    {0x1.7d783ep+26f, chars_format::general, 25, "99999992"},
    {0x1.7d7840p+26f, chars_format::general, 25, "100000000"},
    {0x1.dcd64ep+29f, chars_format::general, 25, "999999936"},
    {0x1.dcd650p+29f, chars_format::general, 25, "1000000000"},
    {0x1.2a05f0p+33f, chars_format::general, 25, "9999998976"},
    {0x1.2a05f2p+33f, chars_format::general, 25, "10000000000"},
    {0x1.74876ep+36f, chars_format::general, 25, "99999997952"},
    {0x1.748770p+36f, chars_format::general, 25, "100000006144"},
    {0x1.d1a94ap+39f, chars_format::general, 25, "999999995904"},
    {0x1.d1a94cp+39f, chars_format::general, 25, "1000000061440"},
    {0x1.2309cep+43f, chars_format::general, 25, "9999999827968"},
    {0x1.2309d0p+43f, chars_format::general, 25, "10000000876544"},
    {0x1.6bcc40p+46f, chars_format::general, 25, "99999991988224"},
    {0x1.6bcc42p+46f, chars_format::general, 25, "100000000376832"},
    {0x1.c6bf52p+49f, chars_format::general, 25, "999999986991104"},
    {0x1.c6bf54p+49f, chars_format::general, 25, "1000000054099968"},
    {0x1.1c3792p+53f, chars_format::general, 25, "9999999198822400"},
    {0x1.1c3794p+53f, chars_format::general, 25, "10000000272564224"},
    {0x1.634578p+56f, chars_format::general, 25, "99999998430674944"},
    {0x1.63457ap+56f, chars_format::general, 25, "100000007020609536"},
    {0x1.bc16d6p+59f, chars_format::general, 25, "999999984306749440"},
    {0x1.bc16d8p+59f, chars_format::general, 25, "1000000053026226176"},
    {0x1.158e46p+63f, chars_format::general, 25, "9999999980506447872"},
    {0x1.158e48p+63f, chars_format::general, 25, "10000001080018075648"},
    {0x1.5af1d6p+66f, chars_format::general, 25, "99999993207994712064"},
    {0x1.5af1d8p+66f, chars_format::general, 25, "100000002004087734272"},
    {0x1.b1ae4cp+69f, chars_format::general, 25, "999999949672133165056"},
    {0x1.b1ae4ep+69f, chars_format::general, 25, "1000000020040877342720"},
    {0x1.0f0cf0p+73f, chars_format::general, 25, "9999999778196308361216"},
    {0x1.0f0cf2p+73f, chars_format::general, 25, "10000000904096215203840"},
    {0x1.52d02cp+76f, chars_format::general, 25, "99999997781963083612160"},
    {0x1.52d02ep+76f, chars_format::general, 25, "100000006789162338353152"},
    {0x1.a78436p+79f, chars_format::general, 25, "999999941790833817157632"},
    {0x1.a78438p+79f, chars_format::general, 25, "1000000013848427855085568"},
    {0x1.08b2a2p+83f, chars_format::general, 25, "9999999562023526247432192"},
    {0x1.08b2a4p+83f, chars_format::general, 25, "1.000000071494503085427917e+25"},
    {0x1.4adf4ap+86f, chars_format::general, 26, "99999993314392253260627968"},
    {0x1.4adf4cp+86f, chars_format::general, 26, "1.0000000253776429011540378e+26"},
    {0x1.9d971ep+89f, chars_format::general, 27, "999999988484154753734934528"},
    {0x1.9d9720p+89f, chars_format::general, 27, "1.00000006227113104857314099e+27"},
    {0x1.027e72p+93f, chars_format::general, 28, "9999999442119689768320106496"},
    {0x1.027e74p+93f, chars_format::general, 28, "1.000000062271131048573140992e+28"},
    {0x1.431e0ep+96f, chars_format::general, 29, "99999992060013656248378458112"},
    {0x1.431e10p+96f, chars_format::general, 29, "1.000000015047466219876688855e+29"},
    {0x1.93e592p+99f, chars_format::general, 30, "999999939489602493962365435904"},
    {0x1.93e594p+99f, chars_format::general, 30, "1.00000001504746621987668885504e+30"},
    {0x1.f8def8p+102f, chars_format::general, 31, "9999999848243207295109594873856"},
    {0x1.f8defap+102f, chars_format::general, 31, "1.000000045270611710242418222694e+31"},
    {0x1.3b8b5ap+106f, chars_format::general, 32, "99999993646728794492579249913856"},
    {0x1.3b8b5cp+106f, chars_format::general, 32, "1.0000000331813535140961264756326e+32"},
    {0x1.8a6e32p+109f, chars_format::general, 33, "999999994495727286427992885035008"},
    {0x1.8a6e34p+109f, chars_format::general, 33, "1.00000007186697974176426006623027e+33"},
    {0x1.ed09bep+112f, chars_format::general, 34, "9999999790214767953607394487959552"},
    {0x1.ed09c0p+112f, chars_format::general, 34, "1.000000040918478759629753193752166e+34"},
    {0x1.342616p+116f, chars_format::general, 35, "99999994188327561679933120182222848"},
    {0x1.342618p+116f, chars_format::general, 35, "1.0000000409184787596297531937521664e+35"},
    {0x1.812f9cp+119f, chars_format::general, 36, "999999961690316245365415600208216064"},
    {0x1.812f9ep+119f, chars_format::general, 36, "1.0000000409184787596297531937521664e+36"},
    {0x1.e17b84p+122f, chars_format::general, 37, "9999999933815812510711506376257961984"},
    {0x1.e17b86p+122f, chars_format::general, 37, "1.000000056764111262482620712460956467e+37"},
    {0x1.2ced32p+126f, chars_format::general, 38, "99999996802856924650656260769173209088"},
    {0x1.2ced34p+126f, chars_format::general, 38, "1.000000069440617264764914727427988521e+38"},
    {0x1.a36e2ep-14f, chars_format::general, 39, "9.99999974737875163555145263671875e-05"},
    {0x1.a36e30p-14f, chars_format::general, 39, "0.0001000000047497451305389404296875"},
    {0x1.0624dcp-10f, chars_format::general, 39, "0.00099999993108212947845458984375"},
    {0x1.0624dep-10f, chars_format::general, 39, "0.001000000047497451305389404296875"},
    {0x1.47ae14p-7f, chars_format::general, 39, "0.00999999977648258209228515625"},
    {0x1.47ae16p-7f, chars_format::general, 39, "0.010000000707805156707763671875"},
    {0x1.999998p-4f, chars_format::general, 39, "0.0999999940395355224609375"},
    {0x1.99999ap-4f, chars_format::general, 39, "0.100000001490116119384765625"},
    {0x1.fffffep-1f, chars_format::general, 39, "0.999999940395355224609375"},
    {0x1.000000p+0f, chars_format::general, 39, "1"},
    {0x1.3ffffep+3f, chars_format::general, 39, "9.99999904632568359375"},
    {0x1.400000p+3f, chars_format::general, 39, "10"},
    {0x1.8ffffep+6f, chars_format::general, 39, "99.99999237060546875"},
    {0x1.900000p+6f, chars_format::general, 39, "100"},
    {0x1.f3fffep+9f, chars_format::general, 39, "999.99993896484375"},
    {0x1.f40000p+9f, chars_format::general, 39, "1000"},
    {0x1.387ffep+13f, chars_format::general, 39, "9999.9990234375"},
    {0x1.388000p+13f, chars_format::general, 39, "10000"},
    {0x1.869ffep+16f, chars_format::general, 39, "99999.9921875"},
    {0x1.86a000p+16f, chars_format::general, 39, "100000"},
    {0x1.e847fep+19f, chars_format::general, 39, "999999.9375"},
    {0x1.e84800p+19f, chars_format::general, 39, "1000000"},
    {0x1.312cfep+23f, chars_format::general, 39, "9999999"},
    {0x1.312d00p+23f, chars_format::general, 39, "10000000"},
    {0x1.7d783ep+26f, chars_format::general, 39, "99999992"},
    {0x1.7d7840p+26f, chars_format::general, 39, "100000000"},
    {0x1.dcd64ep+29f, chars_format::general, 39, "999999936"},
    {0x1.dcd650p+29f, chars_format::general, 39, "1000000000"},
    {0x1.2a05f0p+33f, chars_format::general, 39, "9999998976"},
    {0x1.2a05f2p+33f, chars_format::general, 39, "10000000000"},
    {0x1.74876ep+36f, chars_format::general, 39, "99999997952"},
    {0x1.748770p+36f, chars_format::general, 39, "100000006144"},
    {0x1.d1a94ap+39f, chars_format::general, 39, "999999995904"},
    {0x1.d1a94cp+39f, chars_format::general, 39, "1000000061440"},
    {0x1.2309cep+43f, chars_format::general, 39, "9999999827968"},
    {0x1.2309d0p+43f, chars_format::general, 39, "10000000876544"},
    {0x1.6bcc40p+46f, chars_format::general, 39, "99999991988224"},
    {0x1.6bcc42p+46f, chars_format::general, 39, "100000000376832"},
    {0x1.c6bf52p+49f, chars_format::general, 39, "999999986991104"},
    {0x1.c6bf54p+49f, chars_format::general, 39, "1000000054099968"},
    {0x1.1c3792p+53f, chars_format::general, 39, "9999999198822400"},
    {0x1.1c3794p+53f, chars_format::general, 39, "10000000272564224"},
    {0x1.634578p+56f, chars_format::general, 39, "99999998430674944"},
    {0x1.63457ap+56f, chars_format::general, 39, "100000007020609536"},
    {0x1.bc16d6p+59f, chars_format::general, 39, "999999984306749440"},
    {0x1.bc16d8p+59f, chars_format::general, 39, "1000000053026226176"},
    {0x1.158e46p+63f, chars_format::general, 39, "9999999980506447872"},
    {0x1.158e48p+63f, chars_format::general, 39, "10000001080018075648"},
    {0x1.5af1d6p+66f, chars_format::general, 39, "99999993207994712064"},
    {0x1.5af1d8p+66f, chars_format::general, 39, "100000002004087734272"},
    {0x1.b1ae4cp+69f, chars_format::general, 39, "999999949672133165056"},
    {0x1.b1ae4ep+69f, chars_format::general, 39, "1000000020040877342720"},
    {0x1.0f0cf0p+73f, chars_format::general, 39, "9999999778196308361216"},
    {0x1.0f0cf2p+73f, chars_format::general, 39, "10000000904096215203840"},
    {0x1.52d02cp+76f, chars_format::general, 39, "99999997781963083612160"},
    {0x1.52d02ep+76f, chars_format::general, 39, "100000006789162338353152"},
    {0x1.a78436p+79f, chars_format::general, 39, "999999941790833817157632"},
    {0x1.a78438p+79f, chars_format::general, 39, "1000000013848427855085568"},
    {0x1.08b2a2p+83f, chars_format::general, 39, "9999999562023526247432192"},
    {0x1.08b2a4p+83f, chars_format::general, 39, "10000000714945030854279168"},
    {0x1.4adf4ap+86f, chars_format::general, 39, "99999993314392253260627968"},
    {0x1.4adf4cp+86f, chars_format::general, 39, "100000002537764290115403776"},
    {0x1.9d971ep+89f, chars_format::general, 39, "999999988484154753734934528"},
    {0x1.9d9720p+89f, chars_format::general, 39, "1000000062271131048573140992"},
    {0x1.027e72p+93f, chars_format::general, 39, "9999999442119689768320106496"},
    {0x1.027e74p+93f, chars_format::general, 39, "10000000622711310485731409920"},
    {0x1.431e0ep+96f, chars_format::general, 39, "99999992060013656248378458112"},
    {0x1.431e10p+96f, chars_format::general, 39, "100000001504746621987668885504"},
    {0x1.93e592p+99f, chars_format::general, 39, "999999939489602493962365435904"},
    {0x1.93e594p+99f, chars_format::general, 39, "1000000015047466219876688855040"},
    {0x1.f8def8p+102f, chars_format::general, 39, "9999999848243207295109594873856"},
    {0x1.f8defap+102f, chars_format::general, 39, "10000000452706117102424182226944"},
    {0x1.3b8b5ap+106f, chars_format::general, 39, "99999993646728794492579249913856"},
    {0x1.3b8b5cp+106f, chars_format::general, 39, "100000003318135351409612647563264"},
    {0x1.8a6e32p+109f, chars_format::general, 39, "999999994495727286427992885035008"},
    {0x1.8a6e34p+109f, chars_format::general, 39, "1000000071866979741764260066230272"},
    {0x1.ed09bep+112f, chars_format::general, 39, "9999999790214767953607394487959552"},
    {0x1.ed09c0p+112f, chars_format::general, 39, "10000000409184787596297531937521664"},
    {0x1.342616p+116f, chars_format::general, 39, "99999994188327561679933120182222848"},
    {0x1.342618p+116f, chars_format::general, 39, "100000004091847875962975319375216640"},
    {0x1.812f9cp+119f, chars_format::general, 39, "999999961690316245365415600208216064"},
    {0x1.812f9ep+119f, chars_format::general, 39, "1000000040918478759629753193752166400"},
    {0x1.e17b84p+122f, chars_format::general, 39, "9999999933815812510711506376257961984"},
    {0x1.e17b86p+122f, chars_format::general, 39, "10000000567641112624826207124609564672"},
    {0x1.2ced32p+126f, chars_format::general, 39, "99999996802856924650656260769173209088"},
    {0x1.2ced34p+126f, chars_format::general, 39, "100000006944061726476491472742798852096"},
    {0x1.fffffep+127f, chars_format::general, 39, "340282346638528859811704183484516925440"},
};

#endif // FLOAT_GENERAL_PRECISION_TO_CHARS_TEST_CASES_HPP
