/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var floor = require( '@stdlib/math/base/special/floor' );
var randu = require( '@stdlib/random/base/randu' );
var Float64Array = require( '@stdlib/array/float64' );
var dasum = require( './../lib/dasum.js' );


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.strictEqual( typeof dasum, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function has an arity of 3', function test( t ) {
	t.strictEqual( dasum.length, 3, 'arity of 3' );
	t.end();
});

tape( 'the function computes the sum of absolute values', function test( t ) {
	var x;
	var y;

	x = new Float64Array( [ 1.0, -2.0, 3.0, -4.0, 5.0 ] );
	y = dasum( x.length, x, 1 );

	t.strictEqual( y, 15.0, 'returns 15' );
	t.end();
});

tape( 'the function supports an `x` stride', function test( t ) {
	var x;
	var y;
	var N;

	x = new Float64Array([
		1.0,  // 1
		-2.0,
		3.0,  // 2
		-4.0,
		5.0   // 3
	]);
	N = 3;

	y = dasum( N, x, 2 );

	t.strictEqual( y, 9.0, 'returns expected value' );
	t.end();
});

tape( 'if provided an `N` parameter less than or equal to `0`, the function returns `0`', function test( t ) {
	var x;
	var y;

	x = new Float64Array( [ 1.0, -2.0, 3.0, -4.0, 5.0 ] );

	y = dasum( 0, x, 1 );

	t.strictEqual( y, 0.0, 'returns expected value' );
	t.end();
});

tape( 'if provided a `stride` parameter less than or equal to `0`, the function returns `0`', function test( t ) {
	var x;
	var y;

	x = new Float64Array( [ 1.0, -2.0, 3.0, -4.0, 5.0 ] );

	y = dasum( x.length, x, -1 );

	t.strictEqual( y, 0.0, 'returns expected value' );
	t.end();
});

tape( 'the function supports view offsets', function test( t ) {
	var x0;
	var x1;
	var y;
	var N;

	// Initial array...
	x0 = new Float64Array([
		1.0,
		-2.0, // 1
		3.0,
		-4.0, // 2
		5.0,
		-6.0  // 3
	]);

	// Create an offset view...
	x1 = new Float64Array( x0.buffer, x0.BYTES_PER_ELEMENT*1 ); // begin at 2nd element

	N = floor( x0.length / 2 );

	y = dasum( N, x1, 2 );

	t.strictEqual( y, 12.0, 'returns expected value' );
	t.end();
});

tape( 'if the stride equals `1`, the function efficiently sums the absolute values', function test( t ) {
	var sign;
	var x;
	var y;
	var i;

	x = new Float64Array( 100 );
	for ( i = 0; i < x.length; i++ ) {
		sign = randu();
		if ( sign < 0.5 ) {
			sign = -1.0;
		} else {
			sign = 1.0;
		}
		x[ i ] = sign * (i+1);
	}

	y = dasum( x.length, x, 1 );

	// Compare to closed-form formula:
	t.strictEqual( y, x.length*(x.length+1)/2, 'returns expected sum' );

	x = new Float64Array( 240 );
	for ( i = 0; i < x.length; i++ ) {
		sign = randu();
		if ( sign < 0.5 ) {
			sign = -1.0;
		} else {
			sign = 1.0;
		}
		x[ i ] = sign * (i+1);
	}

	y = dasum( x.length, x, 1 );

	// Compare to closed-form formula:
	t.strictEqual( y, x.length*(x.length+1)/2, 'returns expected sum' );

	t.end();
});
