/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/erlang/cdf' );
import Erlang = require( './../../../../../base/dists/erlang/ctor' );
import entropy = require( './../../../../../base/dists/erlang/entropy' );
import kurtosis = require( './../../../../../base/dists/erlang/kurtosis' );
import logpdf = require( './../../../../../base/dists/erlang/logpdf' );
import mean = require( './../../../../../base/dists/erlang/mean' );
import mgf = require( './../../../../../base/dists/erlang/mgf' );
import mode = require( './../../../../../base/dists/erlang/mode' );
import pdf = require( './../../../../../base/dists/erlang/pdf' );
import quantile = require( './../../../../../base/dists/erlang/quantile' );
import skewness = require( './../../../../../base/dists/erlang/skewness' );
import stdev = require( './../../../../../base/dists/erlang/stdev' );
import variance = require( './../../../../../base/dists/erlang/variance' );

/**
* Interface describing the `erlang` namespace.
*/
interface Namespace {
	/**
	* Erlang distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param k - shape parameter
	* @param lambda - rate parameter
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 2.0, 8, 3.0 );
	* // returns ~0.256
	*
	* y = ns.cdf( 0.0, 1, 1.0 );
	* // returns 0.0
	*
	* var mycdf = ns.cdf.factory( 2, 0.5 );
	* y = mycdf( 6.0 );
	* // returns ~0.801
	*
	* y = mycdf( 2.0 );
	* // returns ~0.264
	*/
	cdf: typeof cdf;

	/**
	* Erlang distribution.
	*/
	Erlang: typeof Erlang;

	/**
	* Returns the differential entropy of an Erlang distribution.
	*
	* ## Notes
	*
	* -   If not provided a positive integer for `k`, the function returns `NaN`.
	* -   If provided a non-positive value for `lambda`, the function returns `NaN`.
	*
	* @param k - shape parameter
	* @param lambda - rate parameter
	* @returns differential entropy
	*
	* @example
	* var v = ns.entropy( 1, 1.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.entropy( 4, 12.0 );
	* // returns ~-0.462
	*
	* @example
	* var v = ns.entropy( 8, 2.0 );
	* // returns ~1.723
	*
	* @example
	* var v = ns.entropy( 1.5, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( 1, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( 2, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( NaN, 2.0 );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis of an Erlang distribution.
	*
	* ## Notes
	*
	* -   If not provided a positive integer for `k`, the function returns `NaN`.
	* -   If provided a non-positive value for `lambda`, the function returns `NaN`.
	*
	* @param k - shape parameter
	* @param lambda - rate parameter
	* @returns excess kurtosis
	*
	* @example
	* var v = ns.kurtosis( 1, 1.0 );
	* // returns 6.0
	*
	* @example
	* var v = ns.kurtosis( 4, 12.0 );
	* // returns 1.5
	*
	* @example
	* var v = ns.kurtosis( 8, 2.0 );
	* // returns 0.75
	*
	* @example
	* var v = ns.kurtosis( 1.5, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 1, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 2, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN, 2.0 );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Erlang distribution natural logarithm of the probability density function (logPDF).
	*
	* @param x - input value
	* @param k - shape parameter
	* @param lambda - rate parameter
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 0.5, 2, 2.5 );
	* // returns ~-0.111
	*
	* var myLogPDF = ns.logpdf.factory( 6, 7.0 );
	* y = myLogPDF( 7.0 );
	* // returns ~-1.864
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value of an Erlang distribution.
	*
	* ## Notes
	*
	* -   If not provided a positive integer for `k`, the function returns `NaN`.
	* -   If provided a non-positive value for `lambda`, the function returns `NaN`.
	*
	* @param k - shape parameter
	* @param lambda - rate parameter
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 1, 1.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.mean( 4, 12.0 );
	* // returns ~0.333
	*
	* @example
	* var v = ns.mean( 8, 2.0 );
	* // returns 4.0
	*
	* @example
	* var v = ns.mean( 1.5, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 1, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 2, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN, 2.0 );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Erlang distribution moment-generating function (MGF).
	*
	* @param t - input value
	* @param k - shape parameter
	* @param lambda - rate parameter
	* @returns evaluated MGF
	*
	* @example
	* var y = ns.mgf( 0.3, 1, 1.0 );
	* // returns ~1.429
	*
	* var myMGF = ns.mgf.factory( 2, 0.5 );
	*
	* y = myMGF( 0.2 );
	* // returns ~2.778
	*
	* y = myMGF( -0.5 );
	* // returns 0.25
	*/
	mgf: typeof mgf;

	/**
	* Returns the mode of an Erlang distribution.
	*
	* ## Notes
	*
	* -   If not provided a positive integer for `k`, the function returns `NaN`.
	* -   If provided a non-positive value for `lambda`, the function returns `NaN`.
	*
	* @param k - shape parameter
	* @param lambda - rate parameter
	* @returns mode
	*
	* @example
	* var v = ns.mode( 1, 1.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.mode( 4, 12.0 );
	* // returns 0.25
	*
	* @example
	* var v = ns.mode( 8, 2.0 );
	* // returns 3.5
	*
	* @example
	* var v = ns.mode( 1.5, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 1, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 2, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN, 2.0 );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Erlang distribution probability density function (PDF).
	*
	* @param x - input value
	* @param k - shape parameter
	* @param lambda - rate parameter
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 0.5, 2, 2.5 );
	* // returns ~0.895
	*
	* var myPDF = ns.pdf.factory( 6, 7.0 );
	* y = myPDF( 7.0 );
	* // returns ~0.155
	*/
	pdf: typeof pdf;

	/**
	* Erlang distribution quantile function.
	*
	* @param p - input value
	* @param k - shape parameter
	* @param lambda - rate parameter
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 1, 1.0 );
	* // returns ~1.609
	*
	* var myQuantile = ns.quantile.factory( 10, 2.0 );
	* y = myQuantile( 0.4 );
	* // returns ~4.452
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of an Erlang distribution.
	*
	* ## Notes
	*
	* -   If not provided a positive integer for `k`, the function returns `NaN`.
	* -   If provided a non-positive value for `lambda`, the function returns `NaN`.
	*
	* @param k - shape parameter
	* @param lambda - rate parameter
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 1, 1.0 );
	* // returns 2.0
	*
	* @example
	* var v = ns.skewness( 4, 12.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.skewness( 8, 2.0 );
	* // returns ~0.707
	*
	* @example
	* var v = ns.skewness( 1.5, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 1, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 2, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN, 2.0 );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of an Erlang distribution.
	*
	* ## Notes
	*
	* -   If not provided a positive integer for `k`, the function returns `NaN`.
	* -   If provided a non-positive value for `lambda`, the function returns `NaN`.
	*
	* @param k - shape parameter
	* @param lambda - rate parameter
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 1, 1.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.stdev( 4, 12.0 );
	* // returns ~0.167
	*
	* @example
	* var v = ns.stdev( 8, 2.0 );
	* // returns ~1.414
	*
	* @example
	* var v = ns.stdev( 1.5, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 1, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 2, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN, 2.0 );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of an Erlang distribution.
	*
	* ## Notes
	*
	* -   If not provided a positive integer for `k`, the function returns `NaN`.
	* -   If provided a non-positive value for `lambda`, the function returns `NaN`.
	*
	* @param k - shape parameter
	* @param lambda - rate parameter
	* @returns variance
	*
	* @example
	* var v = ns.variance( 1, 1.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.variance( 4, 12.0 );
	* // returns ~0.028
	*
	* @example
	* var v = ns.variance( 8, 2.0 );
	* // returns 2.0
	*
	* @example
	* var v = ns.variance( 1.5, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 1, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 2, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN, 2.0 );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Erlang distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
