///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file ClusterAtomsModifier.h
 * \brief Contains the definition of the AtomViz::ClusterAtomsModifier class.
 */

#ifndef __CLUSTER_ATOMS_MODIFIER_H
#define __CLUSTER_ATOMS_MODIFIER_H

#include <core/Core.h>
#include <core/gui/properties/PropertiesEditor.h>

#include <atomviz/AtomViz.h>
#include "../../AtomsObjectAnalyzerBase.h"

namespace AtomViz {

/******************************************************************************
* This modifier tries to find clusters of atoms.
* A user-defined cutoff radius is criteria which atoms belong to the same
* cluster.
******************************************************************************/
class ATOMVIZ_DLLEXPORT ClusterAtomsModifier : public AtomsObjectAnalyzerBase
{
public:

	/// Default constructor.
	ClusterAtomsModifier(bool isLoading = false);

protected:

	// RefTarget virtual functions:
	virtual void saveToStream(ObjectSaveStream& stream);
	virtual void loadFromStream(ObjectLoadStream& stream);
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);

	/// This is the actual analysis method.
	/// It is responsible for storing the analysis results in the results container object that
	/// will be stored along with the modifier application object.
	virtual EvaluationStatus doAnalysis(TimeTicks time, bool suppressDialogs);

	/// Applies the previously calculated analysis results to the atoms object.
	virtual EvaluationStatus applyResult(TimeTicks time, TimeInterval& validityInterval);

	/// Performs the analysis.
	/// Throws an exception on error.
	/// Returns false when the operation has been canceled by the user.
	bool calculate(AtomsObject* atomsObject, bool suppressDialogs = false);

	/// Returns a data channel with the cluster number for each of the atoms.
	DataChannel* atomClusters() const { return _atomClustersChannel; }

	/// Returns whether the algorithm operates only on selected atoms.
	bool onlySelectedAtoms() const { return _onlySelectedAtoms; }
	/// Sets whether the algorithm operates only on selected atoms.
	void setOnlySelectedAtoms(bool flag) { _onlySelectedAtoms = flag; }

public:

	Q_PROPERTY(bool onlySelectedAtoms READ onlySelectedAtoms WRITE setOnlySelectedAtoms)

private:

	/// This is the number of cluster that have been found.
	int numClusters;

	/// This stores the computed cluster numbers.
	ReferenceField<DataChannel> _atomClustersChannel;

	/// Controls whether the algorithm operates only on selected atoms.
	PropertyField<bool> _onlySelectedAtoms;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(ClusterAtomsModifier)
	DECLARE_REFERENCE_FIELD(_atomClustersChannel)
	DECLARE_PROPERTY_FIELD(_onlySelectedAtoms)
};

/******************************************************************************
* A properties editor for the ClusterAtomsModifier class.
******************************************************************************/
class ATOMVIZ_DLLEXPORT ClusterAtomsModifierEditor : public AtomsObjectModifierEditorBase
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

protected Q_SLOTS:

	/// Is called when the user presses the Recalculate button.
	void onRecalculate();

private:

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(ClusterAtomsModifierEditor)
};

};	// End of namespace AtomViz

#endif // __CLUSTER_ATOMS_MODIFIER_H
