///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __ATOM_PICKER_H
#define __ATOM_PICKER_H

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/scene/animation/AnimManager.h>

#include <atomviz/AtomViz.h>
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

/******************************************************************************
* Provides support for picking of atoms.
******************************************************************************/
class ATOMVIZ_DLLEXPORT AtomPicker
{
public:

	struct PickAtomResult {

		/// Constructor.
		PickAtomResult() : hitDistance(FLOATTYPE_MAX), index(-1), atomsObject(NULL) {}

		/// The position of the picked atom in local coordinates.
		Point3 localPos;

		/// The position of the picked atom in world coordinates.
		Point3 worldPos;

		/// The radius of the picked atom.
		FloatType radius;

		/// The distance of the picked atom from the camera.
		FloatType hitDistance;

		/// The index of the picked atom in the AtomsObject.
		int index;

		/// The AtomsObject to which the picked atom belongs.
		AtomsObject::SmartPtr atomsObject;
	};

public:

	/// \brief Constructor.
	AtomPicker() {}

protected:

	/// \brief Picks the atom under the mouse cursor from a certain AtomsObject.
	/// \param vp The viewport to perform hit testing in.
	/// \param clickPoint The position of the mouse cursor in the viewport.
	/// \param atoms The atoms object.
	/// \param time The animation at which hit testing is performed.
	/// \param tm The local to world transformation matrix for the atoms object.
	/// \param result The method stores information about the picked atom in this output parameter.
	/// \return \c true if an atoms has been hit; \c false otherwise.
	bool pickAtom(Viewport& vp, const QPoint& clickPoint, AtomsObject* atoms, TimeTicks time, const AffineTransformation& tm, PickAtomResult& result);

	/// \brief Picks an atom under the mouse cursor from any AtomsObject in the scene.
	/// \param vp The viewport to perform hit testing in.
	/// \param clickPoint The position of the mouse cursor in the viewport.
	/// \param time The animation at which hit testing is performed.
	/// \param result The method stores information about the picked atom in this output parameter.
	/// \return \c true if an atoms has been hit; \c false otherwise.
	bool pickAtom(Viewport& vp, const QPoint& clickPoint, TimeTicks time, PickAtomResult& result);

	/// \brief Renders the atom selection overlay in a viewport.
	/// \param vp The viewport into which the selection marker should be rendered.
	/// \param pickRecord Specifies the atom for which the selection marker should be rendered.
	void renderSelectionMarker(Viewport* vp, const PickAtomResult& pickRecord);

private:

	/// Cached shape of the atom selection marker.
	static QVector<Point3> atomMarker;
};

};	// End of namespace AtomViz

#endif // __ATOM_PICKER_H
