///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/utilities/ProgressIndicator.h>
#include "LAMMPSDataWriter.h"
#include <atomviz/atoms/AtomsObject.h>
#include <atomviz/atoms/datachannels/AtomTypeDataChannel.h>

#include <boost/iostreams/device/file.hpp>
#include <boost/iostreams/stream.hpp>
#include <boost/iostreams/filter/gzip.hpp>
#include <boost/iostreams/filtering_stream.hpp>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(LAMMPSDataWriter, MultiFileWriter)

/******************************************************************************
* Writes the output file.
******************************************************************************/
bool LAMMPSDataWriter::writeAtomsFile(const QString& filepath, DataSet* dataset, const QVector<TimeTicks>& exportFrames, bool suppressDialogs)
{
	using namespace boost::iostreams;

	if(exportFrames.size() != 1)
		MsgLogger() << "Warning: The LAMMPS data writer can only write single frames." << endl;

	TimeTicks time = exportFrames[0];
	int frame = time / dataset->animationSettings()->ticksPerFrame();

	// Extract the atoms to be exported from the scene.
	PipelineFlowState flowState = retrieveAtoms(dataset, time);
	AtomsObject* atoms = dynamic_object_cast<AtomsObject>(flowState.result());
	if(atoms == NULL)
		throw Exception(tr("The scene does not contain any atoms that could be exported (at animation frame %1).").arg(frame));

	DataChannel* posChannel = atoms->getStandardDataChannel(DataChannel::PositionChannel);
	AtomTypeDataChannel* atomTypeChannel = static_object_cast<AtomTypeDataChannel>(atoms->getStandardDataChannel(DataChannel::AtomTypeChannel));
	DataChannel* atomIndexChannel = atoms->getStandardDataChannel(DataChannel::AtomIndexChannel);
	DataChannel* velocityChannel = atoms->getStandardDataChannel(DataChannel::VelocityChannel);
	if(atomTypeChannel && atomTypeChannel->atomTypes().empty()) atomTypeChannel = NULL;

	if(posChannel == NULL)
		throw Exception(tr("Cannot export atoms to LAMMPS data file because there is no position data channel present in the source data."));

	MsgLogger() << "Opening LAMMPS data file" << filepath << "for writing." << endl;

	bool isGZipped = filepath.endsWith(QLatin1String(".gz"));
	std::ios_base::openmode openmode = std::ios_base::out;
	if(isGZipped) openmode |= std::ios_base::binary;
	stream<file_sink> file_stream(filepath.toUtf8().constData(), openmode);
	if(!file_stream->is_open())
		throw Exception(tr("Failed to open LAMMPS data file %1 for writing.").arg(filepath));

	// Automatically compress gzipped dump files.
	filtering_stream<output> output_stream;
	if(isGZipped) output_stream.push(gzip_compressor());
	output_stream.push(file_stream);

	int numAtoms = atoms->atomsCount();
	ProgressIndicator progress(tr("Writing LAMMPS data file (Frame %1)").arg(frame), numAtoms, suppressDialogs);

	TimeInterval interval;
	AffineTransformation simCell = atoms->simulationCell()->cellMatrix();

	FloatType xlo = simCell.getTranslation().X;
	FloatType ylo = simCell.getTranslation().Y;
	FloatType zlo = simCell.getTranslation().Z;
	FloatType xhi = simCell.column(0).X + xlo;
	FloatType yhi = simCell.column(1).Y + ylo;
	FloatType zhi = simCell.column(2).Z + zlo;
	FloatType xy = simCell.column(1).X;
	FloatType xz = simCell.column(2).X;
	FloatType yz = simCell.column(2).Y;

	output_stream << "# LAMMPS data file written by " << QCoreApplication::applicationName().toStdString() << endl;
	output_stream << numAtoms << " atoms" << endl;
	output_stream.precision(12);

	QVector<string> atomTypeStrings;
	if(atomTypeChannel) {
		// We have to obey that some of the atom types might be NULL.
		int numAtomTypes = 0;
		for(int i=0; i<atomTypeChannel->atomTypes().size(); i++) {
			QByteArray s = QByteArray::number(numAtomTypes+1);
			atomTypeStrings.push_back(s.constData());
			if(atomTypeChannel->atomTypes()[i] != NULL)
				numAtomTypes++;
		}
		output_stream << numAtomTypes << " atom types" << endl;
	}
	else output_stream << "1 atom types" << endl;

	output_stream << xlo << " " << xhi << " xlo xhi" << endl;
	output_stream << ylo << " " << yhi << " ylo yhi" << endl;
	output_stream << zlo << " " << zhi << " zlo zhi" << endl;
	if(xy != 0 || xz != 0 || yz != 0) {
		output_stream << xy << " " << xz << " " << yz << " xy xz yz" << endl;
	}
	output_stream << endl;

	// Write atomic positions.
	output_stream << "Atoms" << endl << endl;
	for(int i=0; i<numAtoms; i++) {
		// Update progress indicator.
		if((i % 1000) == 0) {
			progress.setValue(i);
			progress.isCanceled();
		}

		if(atomIndexChannel)
			output_stream << atomIndexChannel->getInt(i) << " ";
		else
			output_stream << (i+1) << " ";

		if(atomTypeChannel)
			output_stream << atomTypeStrings[atomTypeChannel->getInt(i) % atomTypeStrings.size()] << " ";
		else
			output_stream << "1 ";

		const Point3& v = posChannel->getPoint3(i);
		output_stream << v.X << " " << v.Y << " " << v.Z << endl;
	}

	// Write atomic velocities
	if(velocityChannel) {
		output_stream << endl << "Velocities" << endl << endl;
		for(int i=0; i<numAtoms; i++) {
			// Update progress indicator.
			if((i % 1000) == 0) {
				progress.setValue(i);
				progress.isCanceled();
			}

			if(atomIndexChannel)
				output_stream << atomIndexChannel->getInt(i) << " ";
			else
				output_stream << (i+1) << " ";

			const Vector3& v = velocityChannel->getVector3(i);
			output_stream << v.X << " " << v.Y << " " << v.Z << endl;
		}
	}
	return true;
}

};	// End of namespace AtomViz
