///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <base/Base.h>
#include <base/linalg/AffineTransformation.h>

namespace Base {

/******************************************************************************
* Generates a rotation matrix around the given axis.
******************************************************************************/
AffineTransformation AffineTransformation::rotation(const Rotation& rot)
{
	FloatType c = cos(rot.angle);
	FloatType s = sin(rot.angle);
	FloatType t = 1.0 - c;
    const Vector3& a = rot.axis;
	OVITO_ASSERT_MSG(abs(LengthSquared(a) - 1.0) <= FLOATTYPE_EPSILON, "AffineTransformation::rotation", "Rotation axis vector must be normalized.");

	// Make sure the result is a pure rotation matrix.
#ifdef _DEBUG
	AffineTransformation tm(	t * a.X * a.X + c,       t * a.X * a.Y - s * a.Z, t * a.X * a.Z + s * a.Y, 0.0,
						t * a.X * a.Y + s * a.Z, t * a.Y * a.Y + c,       t * a.Y * a.Z - s * a.X, 0.0,
						t * a.X * a.Z - s * a.Y, t * a.Y * a.Z + s * a.X, t * a.Z * a.Z + c      , 0.0);
    OVITO_ASSERT_MSG(tm.isRotationMatrix(), "AffineTransformation::rotation(const Rotation&)" , "Result is not a pure rotation matrix.");
#endif

	return AffineTransformation(	t * a.X * a.X + c,       t * a.X * a.Y - s * a.Z, t * a.X * a.Z + s * a.Y, 0.0,
					t * a.X * a.Y + s * a.Z, t * a.Y * a.Y + c,       t * a.Y * a.Z - s * a.X, 0.0,
					t * a.X * a.Z - s * a.Y, t * a.Y * a.Z + s * a.X, t * a.Z * a.Z + c      , 0.0);
}

/******************************************************************************
* Generates a rotation matrix from a quaternion.
******************************************************************************/
AffineTransformation AffineTransformation::rotation(const Quaternion& q)
{
#ifdef _DEBUG
	if(abs(DotProduct(q,q) - 1.0) > FLOATTYPE_EPSILON) {
		MsgLogger() << "Invalid quaternion is: " << q;
		OVITO_ASSERT_MSG(false, "AffineTransformation::rotation(const Quaternion&)", "Quaternion must be normalized.");
	}

	// Make sure the result is a pure rotation matrix.
	AffineTransformation tm(1.0 - 2.0*(q.Y*q.Y + q.Z*q.Z),       2.0*(q.X*q.Y - q.W*q.Z),       2.0*(q.X*q.Z + q.W*q.Y), 0.0,
	        2.0*(q.X*q.Y + q.W*q.Z), 1.0 - 2.0*(q.X*q.X + q.Z*q.Z),       2.0*(q.Y*q.Z - q.W*q.X), 0.0,
            2.0*(q.X*q.Z - q.W*q.Y),       2.0*(q.Y*q.Z + q.W*q.X), 1.0 - 2.0*(q.X*q.X + q.Y*q.Y), 0.0);
    OVITO_ASSERT_MSG(tm.isRotationMatrix(), "AffineTransformation::rotation(const Quaternion&)" , "Result is not a pure rotation matrix.");
#endif
	return AffineTransformation(1.0 - 2.0*(q.Y*q.Y + q.Z*q.Z),       2.0*(q.X*q.Y - q.W*q.Z),       2.0*(q.X*q.Z + q.W*q.Y), 0.0,
				        2.0*(q.X*q.Y + q.W*q.Z), 1.0 - 2.0*(q.X*q.X + q.Z*q.Z),       2.0*(q.Y*q.Z - q.W*q.X), 0.0,
			            2.0*(q.X*q.Z - q.W*q.Y),       2.0*(q.Y*q.Z + q.W*q.X), 1.0 - 2.0*(q.X*q.X + q.Y*q.Y), 0.0);
}

/******************************************************************************
* Generates a scaling matrix.
******************************************************************************/
AffineTransformation AffineTransformation::scaling(const Scaling& scaling)
{
	Matrix3 U = Matrix3::rotation(scaling.Q);
	Matrix3 K = Matrix3(scaling.S.X, 0.0, 0.0,
						0.0, scaling.S.Y, 0.0,
						0.0, 0.0, scaling.S.Z);
	return (U * K * U.transposed());
}

/******************************************************************************
* Generates a rotation matrix around the X axis.
******************************************************************************/
AffineTransformation AffineTransformation::rotationX(FloatType angle)
{
	FloatType c = cos(angle);
	FloatType s = sin(angle);
	return AffineTransformation(1.0, 0.0, 0.0, 0.0,
				  0.0,   c,  -s, 0.0,
				  0.0,   s,   c, 0.0);
}

/******************************************************************************
* Generates a rotation matrix around the Y axis.
******************************************************************************/
AffineTransformation AffineTransformation::rotationY(FloatType angle)
{
	FloatType c = cos(angle);
	FloatType s = sin(angle);
	return AffineTransformation(  c, 0.0,   s, 0.0,
				  0.0, 1.0, 0.0, 0.0,
				   -s, 0.0,   c, 0.0);
}

/******************************************************************************
* Generates a rotation matrix around the Z axis.
******************************************************************************/
AffineTransformation AffineTransformation::rotationZ(FloatType angle)
{
	FloatType c = cos(angle);
	FloatType s = sin(angle);
	return AffineTransformation(  c,  -s, 0.0, 0.0,
				    s,   c, 0.0, 0.0,
				  0.0, 0.0, 1.0, 0.0);
}

/******************************************************************************
* Generates a translation matrix.
******************************************************************************/
AffineTransformation AffineTransformation::translation(const Vector3& t)
{
	return AffineTransformation(1.0, 0.0, 0.0, t.X,
				  0.0, 1.0, 0.0, t.Y,
				  0.0, 0.0, 1.0, t.Z);
}

/******************************************************************************
* Generates a diagonal scaling matrix.
******************************************************************************/
AffineTransformation AffineTransformation::scaling(FloatType s)
{
	return AffineTransformation(  s, 0.0, 0.0, 0.0,
				  0.0,   s, 0.0, 0.0,
				  0.0, 0.0,   s, 0.0);
}

/******************************************************************************
* Generates a look at matrix.
******************************************************************************/
AffineTransformation AffineTransformation::lookAt(const Point3& camera, const Point3& target, const Vector3& upVector)
{
	Vector3 zaxis = Normalize(camera - target);
	Vector3 xaxis = CrossProduct(upVector, zaxis);
	if(xaxis == NULL_VECTOR) {
		xaxis = CrossProduct(Vector3(0,1,0), zaxis);
		if(xaxis == NULL_VECTOR) {
			xaxis = CrossProduct(Vector3(0,0,1), zaxis);
			OVITO_ASSERT(xaxis != NULL_VECTOR);
		}
	}
	xaxis = Normalize(xaxis);
	Vector3 yaxis = CrossProduct(zaxis, xaxis);

	return AffineTransformation(	xaxis.X, xaxis.Y, xaxis.Z, -DotProduct(xaxis, camera - ORIGIN),
					yaxis.X, yaxis.Y, yaxis.Z, -DotProduct(yaxis, camera - ORIGIN),
					zaxis.X, zaxis.Y, zaxis.Z, -DotProduct(zaxis, camera - ORIGIN));
}


/******************************************************************************
* Generates a matrix with shearing normal to the z-axis.
******************************************************************************/
AffineTransformation AffineTransformation::shear(FloatType gammaX, FloatType gammaY)
{
	return AffineTransformation(1.0, 0.0, gammaX, 0.0,
				  0.0, 1.0, gammaY, 0.0,
				  0.0, 0.0, 1.0, 0.0);
}


};	// End of namespace Base
