///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file DefaultSceneRenderer.h
 * \brief Contains the definition of the Core::DefaultSceneRenderer class.
 */

#include <core/Core.h>
#include <core/viewport/SceneRenderer.h>
#include <base/utilities/Color.h>

namespace Core {

class SceneNode;		// defined in SceneNode.h
class ObjectNode;		// defined in ObjectNode.h
class ModifiedObject;	// defined in ModifiedObject.h

/**
 * \brief Provides a default implementation for the SceneRenderer interface.
 *
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT DefaultSceneRenderer : public SceneRenderer
{
	Q_OBJECT

public:

	/// The scaling factor that is applied to the bounding box of a selected
	/// object to build the selection brackets shown in the viewports.
	static const FloatType SELECTION_MARKER_SCALING;

	/// \brief Computes the bounding box of the scene.
	/// \param settings The settings of the viewport which is used as context for the bounding box computation.
	/// \param time The animation at which to compute the scene bounding box.
	/// \param mode Specifies what should be included in the bounding box.
	/// \return The calculated bounding box.
	virtual Box3 sceneExtents(ViewportRecord* settings, TimeTicks time, SceneExtentsMode mode);

protected:

	/////////////////////////////////// from SceneRenderer /////////////////////////////

	/// \brief Is called by the Viewport class to let this scene renderer do its job.
	virtual void renderScene();

	/// \brief Performs hit testing on all visible nodes in the scene and returns a list of nodes that
	///        are within the given pick region.
	///
	/// \param pickRegion The pick region to use for hit testing.
	/// \return The list of picked nodes.
	virtual QVector<SceneNode*> hitTestSceneNodes(const PickRegion& pickRegion);

	////////////////////////////////// Internal implementation //////////////////////////

	/// \brief Renders a single node.
	virtual void renderNode(SceneNode* node);

	/// \brief Renders the selection marker around a selected scene node.
	virtual void renderSelectionMarker(SceneNode* node, const ColorA& color);

	/// \brief Takes the node's material and creates a viewport material from it.
	virtual void setupNodeMaterial(ObjectNode* objNode);

	/// \brief Render the selected modifiers.
	void renderModifiedObject(ModifiedObject* modObj, ObjectNode* objNode);

	/// \brief Renders the motion trajectory of a node.
	void renderTrajectory(SceneNode* node);

protected:

	/// If true than no rendering occurs.
	bool computeBBox;
};

};
