/**
//@HEADER
// ************************************************************************
//
//                   Trios: Trilinos I/O Support
//                 Copyright 2011 Sandia Corporation
//
// Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
// the U.S. Government retains certain rights in this software.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//Questions? Contact Ron A. Oldfield (raoldfi@sandia.gov)
//
// *************************************************************************
//@HEADER
 */
/*
 * This is sample code generated by rpcgen.
 * These are only templates and you can use them
 * as a guideline for developing your own functions.
 */

#include <inttypes.h>

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <pnetcdf.h>
#include <map>
#include <string>
#include <vector>
using namespace std;

#include <Trios_nssi_client.h>

#include "netcdf_args.h"
#include "netcdf_debug.h"
#include "netcdf_client_private.h"

#include "NcGroupInfo.h"
#include "NcVarInfo.h"

#undef USE_NC_TYPE
#define USE_NC_TYPE 1

#include "Trios_timer.h"

/** Association between ncid and MPI communicators */
map<int, MPI_Comm> comm_map;


/* ************************ Private functions ******************* */



/**
 * Initialize the netcdf client interface.
 *
 * The init function relies on environment variables to know where to
 * find the netcdf service.
 */
extern "C"
int ncmpi_client_init(void)
{
    int rc = NC_NOERR;
    extern int netcdf_client_init();

    rc = netcdf_client_init();

    return rc;
}

extern "C"
int ncmpi_fini(void)
{
    int rc = NC_NOERR;
    extern int netcdf_client_fini(void);

    rc=netcdf_client_fini();

    return rc;
}



/* ************************ NETCDF LIBRARY STUBS ******************* */



/*
 *
 */
extern "C"
const char *ncmpi_strerror(int ncerr)
{
    extern const char *nc_strerror(int);
    return nc_strerror(ncerr);
}


/**
 * Collective call to create a netcdf dataset.
 */
extern "C"
int ncmpi_create(
        MPI_Comm comm,
                const char *path,
                int cmode,
                MPI_Info info,
                int *ncidp)
{
    int rc = NC_NOERR;
    log_level debug_level = netcdf_debug_level;
    int rank, np;
    extern int netcdf_client_init(void);
    extern int nc_create(const char *, int, int *);

    trios_declare_timer(CreateTime);
    trios_start_timer(CreateTime);

    MPI_Comm_rank(comm, &rank);
    MPI_Comm_size(comm, &np);

    trios_declare_timer(InitTime);
    trios_start_timer(InitTime);
    rc = ncmpi_client_init();
    trios_stop_timer("init", InitTime);

    trios_declare_timer(BcastTime);
    trios_start_timer(BcastTime);
#if defined(TRIOS_USE_TIMERS)
    MPI_Barrier(comm);
#endif
    trios_stop_timer("after init barrier", BcastTime);

    log_debug(debug_level, "calling nc_create(%s, %d)", path, cmode);

    log_debug(netcdf_debug_level, "Calling nc_create");
    rc =  nc_create(path, cmode, ncidp);

    trios_start_timer(BcastTime);
#if defined(TRIOS_USE_TIMERS)
    MPI_Barrier(comm);
#endif
    trios_stop_timer("after create barrier", BcastTime);

    if (rc != NC_NOERR) {
        if (rank == 0) log_error(netcdf_debug_level, "%s", ncmpi_strerror(rc));
        return rc;
    }

    trios_start_timer(BcastTime);
    nc_set_file_state(*ncidp);
    trios_stop_timer("set file state", BcastTime);

    /* duplicate the client-side metadata for non-root processes */
    trios_start_timer(BcastTime);
    if (rank != 0) {
        if (group_map.find(*ncidp) == group_map.end()) {
            NcGroupInfo *group = new NcGroupInfo(*ncidp,
                    NcFileInfo(path, cmode, 0, 0));
            group_map[*ncidp] = group;
        }
        else {
            log_error(netcdf_debug_level, "dataset already exists for ncid=%d", *ncidp);
            rc = NC_EEXIST;
        }
    }
    trios_stop_timer("dup client-side metadata", BcastTime);

    trios_start_timer(BcastTime);
    MPI_Bcast(&rc, 1, MPI_INT, 1%np, comm);
    trios_stop_timer("group replicate rc Bcast", BcastTime);

    if (rc == NC_NOERR) {
        comm_map[*ncidp] = comm;
    }

    /* sync all the processors */
    trios_start_timer(BcastTime);
    MPI_Barrier(comm);
    trios_stop_timer("end barrier", BcastTime);

    trios_stop_timer("Create", CreateTime);

    return rc;
}



extern "C"
int ncmpi_open(
        MPI_Comm comm,
        const char *path,
        int mode,
        MPI_Info info,
        int *ncidp)
{
    int rc = NSSI_OK;
    log_level debug_level = netcdf_debug_level;
    char *encoded_group = NULL;
    struct nc_group group;
    int rank, np;
    int root = 0;
    XDR group_xdrs;
    size_t group_size;

    /* prototypes */
    extern int nc_open(const char *, int, int *);

    trios_declare_timer(OpenTime);
    trios_start_timer(OpenTime);

    MPI_Comm_rank(comm, &rank);
    MPI_Comm_size(comm, &np);

    trios_declare_timer(InitTime);
    trios_start_timer(InitTime);
    rc = ncmpi_client_init();
    trios_stop_timer("init", InitTime);

    memset(&group, 0, sizeof(struct nc_group));

    log_debug(debug_level, "calling nc_open(%s, %d)", path, mode);

    log_debug(netcdf_debug_level, "Calling nc_open");
    rc = nc_open(path, mode, ncidp);

    if (rc != NC_NOERR) {
        log_error(debug_level, "Could not open dataset");
        goto cleanup;
    }

    nc_set_file_state(*ncidp);

    if (rank == root) log_debug(debug_level, "Dataset open on root");

    /* At this point, the root has an open netcdf dataset.   The next
     * step is to use XDR to serialize the nc_group, send it to all
     * the client processes so they all have a copy of the dataset
     * metadata.
     */

    /* extract the nc_group and calculate its size */
    if (rank == root) {
        /* First, we need to convert the NcGroupInfo to struct nc_group */
        if (group_map.find(*ncidp) == group_map.end()) {
            log_error(debug_level, "could not find group for dataset");
            group_size = 0;
        }
        else {
            group_map[*ncidp]->copyTo(group);
            group_size = xdr_sizeof((xdrproc_t)&xdr_nc_group, &group);
        }
    }

    MPI_Bcast(&group_size, sizeof(size_t), MPI_BYTE, root, comm);

    if (rank == root) log_debug(debug_level, "group_size = %d", int(group_size));

    if (group_size <= 0) {
        rc = NC_EINVAL;
        if (rank == root) log_error(debug_level, "expected non-zero size for group, got %d",
                (int)group_size);
        goto cleanup;
    }

    /* allocate space for the serialized group */
    encoded_group = (char *)calloc(group_size, sizeof(char));

    /* serialize the nc_group */
    if (rank == root) {

        /* create a memory stream for the result */
        xdrmem_create(&group_xdrs, encoded_group, group_size, XDR_ENCODE);

        /* encode the result */
        if (! xdr_nc_group(&group_xdrs, &group)) {
            rc = NSSI_EENCODE;
        }
    }

    MPI_Bcast(&rc, 1, MPI_INT, root, comm);

    if (rc != NSSI_OK) {
        if (rank == root)
            log_error(debug_level, "unable to encode nc_group");
        goto cleanup;
    }

    /* broadcast encoded result */
    MPI_Bcast(encoded_group, group_size, MPI_BYTE, root, comm);

    /* decode the result on non-root nodes */
    if (rank != root) {

        /* create a memory stream for decoding */
        xdrmem_create(&group_xdrs, encoded_group, group_size, XDR_DECODE);

        /* decode the result */
        if (! xdr_nc_group(&group_xdrs, &group)) {
            rc = NSSI_EDECODE;
        }
    }

    MPI_Bcast(&rc, 1, MPI_INT, ((root+1)%np), comm);
    if (rc != NSSI_OK) {
        if (rank == root)
            log_error(debug_level, "unable to decode result");
        goto cleanup;
    }

    /* At this point, every process should have a copy of the nc_group.
     * We need to create identical dataset entries on non-root nodes.
     */

    if (rank != root) {

        /* create an entry in list of open datasets */
        if (group_map.find(*ncidp) == group_map.end()) {
            NcGroupInfo *groupInfo = new NcGroupInfo(group, NcFileInfo(path, mode, 0, 0));
            group_map[*ncidp] = groupInfo;
        }
        else {
            log_error(debug_level, "dataset already exists for ncid=%d", *ncidp);
            rc = NC_EEXIST;
            goto cleanup;
        }
    }

    /* Associate this communicator with the dataset */
    comm_map[*ncidp] = comm;


cleanup:

    /* cleanup data structure allocated by XDR encoding */
    xdr_free((xdrproc_t)xdr_nc_group, (char *)&group);

    if (encoded_group) free(encoded_group);

    /* sync all the processors */
    MPI_Barrier(comm);

    trios_stop_timer("Open", OpenTime);

    if (rank == root) log_debug(debug_level, "Finished opening dataset ncid=%d, rc=%d", *ncidp, rc);
    return rc;
}


/**
 * Delete a dataset.
 *
 * This operation isn't mentioned in the netcdf-api document, but it is in the
 * include file.  It is just a wrapper around the MPI_File_delete function.
 */
extern "C"
int ncmpi_delete(char *filename, MPI_Info info)
{
    return MPI_File_delete(filename, info);
}

/* **************** DEFINE MODE FUNCTIONS **************** */

/**
 * Collective call to add a new dimension to an open netCDF dataset.
 * All processes in the communicator must call them with the same values.
 * If values do not match, an error is returned by ncmpi_enddef.
 *
 * The correct way to implement this is to construct a log of operations,
 * then execute the sequence of operations when the enddef method is called.
 */
extern "C"
int ncmpi_def_dim(
                int ncid,
                const char *name,
                MPI_Offset len,
                int *dimidp)
{
    int rc = NC_NOERR;
    int rank;
    MPI_Comm comm = comm_map[ncid];
    extern int nc_def_dim(int, const char *, size_t, int *);

    rc = MPI_Comm_rank(comm, &rank);

    trios_declare_timer(DefDimTime);
    trios_start_timer(DefDimTime);
    rc = nc_def_dim(ncid, name, (size_t) len, dimidp);
    trios_stop_timer("DefDim", DefDimTime);

    if (rank != 0) {
        /* add the dimension to the open dataset */
        if (group_map.find(ncid) != group_map.end()) {
            group_map[ncid]->def_dim(*dimidp, name, len);
        }
        else {
            log_error(netcdf_debug_level, "dataset for ncid=%d not found", ncid);
            rc = NC_EBADID;
        }
    }

    return rc;
}


/**
 * Define a variable for an open netCDF dataset.
 */
extern "C"
int ncmpi_def_var(
                int ncid,
                const char *name,
                nc_type xtype,
                int ndims,
                const int dimids[],
                int *varidp)
{
    int rc = NC_NOERR;
    int rank;
    MPI_Comm comm = comm_map[ncid];
#if USE_NC_TYPE
    extern int nc_def_var(int, const char *, nc_type, int, const int *, int *);
#else
    extern int nc_def_var(int, const char *, int, int, const int *, int *);
#endif


    rc = MPI_Comm_rank(comm, &rank);

    trios_declare_timer(DefVarTime);
    trios_start_timer(DefVarTime);
    rc = nc_def_var(ncid, name, xtype, ndims, dimids, varidp);
    trios_stop_timer("DefVar", DefVarTime);

    if (rank != 0) {
        /* add the variable to the open dataset */
        if (group_map.find(ncid) != group_map.end()) {
            group_map[ncid]->def_var(*varidp, name, xtype, ndims, dimids);
        }
        else {
            log_error(netcdf_debug_level, "dataset for ncid=%d not found", ncid);
            rc = NC_EBADID;
        }
    }

    return rc;
}


/* **************** ATTRIBUTE FUNCTIONS **************** */



/**
 * Collective function for put_att.
 */
extern "C"
int _ncmpi_put_att_type(
        int ncid,
        int varid,
        const char *name,
        nc_type xtype,
        arg_type atype,
        size_t len,
        const void *obj_p)
{
    int rc = NC_NOERR;
    int rank;
    MPI_Comm comm = comm_map[ncid];

    /* prototpye for nc_put_att_type */
#if USE_NC_TYPE
    extern int _nc_put_att_type(int, int, const char *, nc_type, arg_type, size_t, const void *);
#else
    extern int _nc_put_att_type(int, int, const char *, int, arg_type, size_t, const void *);
#endif

    rc = MPI_Comm_rank(comm, &rank);

    rc = _nc_put_att_type(ncid, varid, name, xtype, atype, len, obj_p);

    MPI_Bcast(&rc, 1, MPI_INT, 0, comm);

    /* Everyone else needs to store the data in the metadata cache */
    if ((rc == NC_NOERR) && (rank != 0)) {
        /* add the attribute to the local metadata */
        if (group_map.find(ncid) != group_map.end()) {
            NcGroupInfo *info = group_map[ncid];

            if (varid == NC_GLOBAL) {
#if USE_NC_TYPE
                info->def_att(name, xtype, len);
#else
                info->def_att(name, (int)xtype, len);
#endif
            }
            else {
                if (info->_vars.find(varid) != info->_vars.end()) {
                    info->_vars[varid]->def_att(name, (nc_type)xtype, len);
                }
                else {
                    log_error(netcdf_debug_level, "unable to find varid=%d", varid);
                    rc = NC_ENOTVAR;
                    goto cleanup;
                }
            }
        }
        else {
            log_error(netcdf_debug_level, "unable to find ncid=%d", ncid);
            rc = NC_ENOTNC;
            goto cleanup;
        }
    }

cleanup:
    return rc;
}

extern "C"
int ncmpi_get_att(
        int ncid,
        int varid,
        const char *name,
        void *data)
{
    extern int nc_get_att(int, int, const char *, void *);

    int rc, rank;
    MPI_Comm comm = comm_map[ncid];
    nc_type xtype;
    MPI_Offset len;
    size_t nbytes;
    log_level debug_level = netcdf_debug_level;

    MPI_Comm_rank(comm, &rank);

    rc = ncmpi_inq_att(ncid, varid, name, &xtype, &len);
    if (rc != NC_NOERR) {
        log_error(debug_level, "%s", ncmpi_strerror(rc));
        goto cleanup;
    }

    switch (xtype) {

    case NC_BYTE:
        nbytes = len;
        break;

    case NC_CHAR:
        nbytes = len * sizeof(char);
        break;

    case NC_SHORT:
        nbytes = len * sizeof(short);
        break;

    case NC_INT:
        nbytes = len * sizeof(int);
        break;

    case NC_FLOAT:
        nbytes = len * sizeof(float);
        break;

    case NC_DOUBLE:
        nbytes = len * sizeof(double);
        break;

    default:
        rc = NC_EBADTYPE;
        goto cleanup;
    }

    rc = nc_get_att(ncid, varid, name, data);

    MPI_Bcast(&rc, 1, MPI_INT, 0, comm);
    if (rc != NC_NOERR) {
        if (rank == 0) log_error(debug_level, "%s", ncmpi_strerror(rc));
        goto cleanup;
    }

    MPI_Bcast(data, nbytes, MPI_BYTE, 0, comm);

cleanup:
    return rc;
}

extern "C"
int ncmpi_get_att_text(
        int ncid,
        int varid,
        const char *name,
        char *data)
{
    int rc;
    nc_type xtype;
    log_level debug_level = netcdf_debug_level;

    rc = ncmpi_inq_atttype(ncid, varid, name, &xtype);
    if (rc != NC_NOERR) {
        log_error(debug_level, "%s", ncmpi_strerror(rc));
        goto cleanup;
    }

    if (xtype != NC_CHAR) {
        log_error(debug_level, "Type mismatch");
        rc = NC_EIOMISMATCH;
        goto cleanup;
    }

    rc = ncmpi_get_att(ncid, varid, name, data);

cleanup:
    return rc;
}

extern "C"
int ncmpi_get_att_short(
        int ncid,
        int varid,
        const char *name,
        short *data)
{
    int rc;
    nc_type xtype;
    log_level debug_level = netcdf_debug_level;

    rc = ncmpi_inq_atttype(ncid, varid, name, &xtype);
    if (rc != NC_NOERR) {
        log_error(debug_level, "%s", ncmpi_strerror(rc));
        goto cleanup;
    }

    if (xtype != NC_SHORT) {
        log_error(debug_level, "Type mismatch");
        rc = NC_EIOMISMATCH;
        goto cleanup;
    }

    rc = ncmpi_get_att(ncid, varid, name, data);

cleanup:
    return rc;
}

extern "C"
int ncmpi_get_att_int(
        int ncid,
        int varid,
        const char *name,
        int *data)
{
    int rc;
    nc_type xtype;
    log_level debug_level = netcdf_debug_level;

    rc = ncmpi_inq_atttype(ncid, varid, name, &xtype);
    if (rc != NC_NOERR) {
        log_error(debug_level, "%s", ncmpi_strerror(rc));
        goto cleanup;
    }

    if (xtype != NC_INT) {
        log_error(debug_level, "Type mismatch");
        rc = NC_EIOMISMATCH;
        goto cleanup;
    }

    rc = ncmpi_get_att(ncid, varid, name, data);

cleanup:
    return rc;
}

extern "C"
int ncmpi_get_att_float(
        int ncid,
        int varid,
        const char *name,
        float*data)
{
    int rc;
    nc_type xtype;
    log_level debug_level = netcdf_debug_level;

    rc = ncmpi_inq_atttype(ncid, varid, name, &xtype);
    if (rc != NC_NOERR) {
        log_error(debug_level, "%s", ncmpi_strerror(rc));
        goto cleanup;
    }

    if (xtype != NC_FLOAT) {
        log_error(debug_level, "Type mismatch");
        rc = NC_EIOMISMATCH;
        goto cleanup;
    }

    rc = ncmpi_get_att(ncid, varid, name, data);

cleanup:
    return rc;
}

extern "C"
int ncmpi_get_att_double(
        int ncid,
        int varid,
        const char *name,
        double *data)
{
    int rc;
    nc_type xtype;
    log_level debug_level = netcdf_debug_level;

    rc = ncmpi_inq_atttype(ncid, varid, name, &xtype);
    if (rc != NC_NOERR) {
        log_error(debug_level, "%s", ncmpi_strerror(rc));
        goto cleanup;
    }

    if (xtype != NC_DOUBLE) {
        log_error(debug_level, "Type mismatch");
        rc = NC_EIOMISMATCH;
        goto cleanup;
    }

    rc = ncmpi_get_att(ncid, varid, name, data);

cleanup:
    return rc;
}

/**
 * Add or change a variable attribute of an open netCDF dataset.
 */
extern "C"
int ncmpi_put_att (
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                size_t len,
                const void *obj_p)
{
    return _ncmpi_put_att_type(ncid, varid, name, xtype, NC_ARG_VOID, len, obj_p);
}

extern "C"
int ncmpi_put_att_text(
                int ncid,
                int varid,
                const char *name,
                MPI_Offset len,
                const char *tp)
{
    return _ncmpi_put_att_type(ncid, varid, name, NC_CHAR, NC_ARG_TEXT, len*sizeof(char), tp);
}

extern "C"
int ncmpi_put_att_uchar(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                MPI_Offset len,
                const unsigned char *up)
{
    return _ncmpi_put_att_type(ncid, varid, name, xtype, NC_ARG_UCHAR, len*sizeof(unsigned char), up);
}

extern "C"
int ncmpi_put_att_schar(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                MPI_Offset len,
                const signed char *cp)
{
    return _ncmpi_put_att_type(ncid, varid, name, xtype, NC_ARG_SCHAR, len*sizeof(signed char), cp);
}

extern "C"
int ncmpi_put_att_short (
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                MPI_Offset len,
                const short *sp)
{
    return _ncmpi_put_att_type(ncid, varid, name, xtype, NC_ARG_SHORT, len*sizeof(short), sp);
}

extern "C"
int ncmpi_put_att_int(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                MPI_Offset len,
                const int *ip)
{
    return _ncmpi_put_att_type(ncid, varid, name, xtype, NC_ARG_INT, len*sizeof(int), ip);
}

extern "C"
int ncmpi_put_att_long(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                MPI_Offset len,
                const long *lp)
{
    return _ncmpi_put_att_type(ncid, varid, name, xtype, NC_ARG_LONG, len*sizeof(long), lp);
}

extern "C"
int ncmpi_put_att_float(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                MPI_Offset len,
                const float *fp)
{
    return _ncmpi_put_att_type(ncid, varid, name, xtype, NC_ARG_FLOAT, len*sizeof(float), fp);
}

extern "C"
int ncmpi_put_att_double(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                MPI_Offset len,
                const double *dp)
{
    return _ncmpi_put_att_type(ncid, varid, name, xtype, NC_ARG_DOUBLE, len*sizeof(double), dp);
}

extern "C"
int ncmpi_put_att_ubyte(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                MPI_Offset len,
                const unsigned char *op)
{
    return _ncmpi_put_att_type(ncid, varid, name, xtype, NC_ARG_UBYTE, len*sizeof(unsigned char), op);
}

extern "C"
int ncmpi_put_att_ushort(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                MPI_Offset len,
                const unsigned short *op)
{
    return _ncmpi_put_att_type(ncid, varid, name, xtype, NC_ARG_USHORT, len*sizeof(unsigned short), op);
}

extern "C"
int ncmpi_put_att_uint(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                MPI_Offset len,
                const unsigned int *op)
{
    return _ncmpi_put_att_type(ncid, varid, name, xtype, NC_ARG_UINT, len*sizeof(unsigned int), op);
}

extern "C"
int ncmpi_put_att_longlong(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                MPI_Offset len,
                const long long *op)
{
    return _ncmpi_put_att_type(ncid, varid, name, xtype, NC_ARG_LONGLONG, len*sizeof(long long), op);
}

extern "C"
int ncmpi_put_att_ulonglong(
                int ncid,
                int varid,
                const char *name,
                nc_type xtype,
                MPI_Offset len,
                const unsigned long long *op)
{
    return _ncmpi_put_att_type(ncid, varid, name, xtype, NC_ARG_ULONGLONG, len*sizeof(unsigned long long), op);
}


/* ******************* INQUIRY FUNCTIONS ********************** */


extern "C"
int ncmpi_inq(
        int ncid,
        int *ndimsp,
        int *nvarsp,
        int *nattsp,
        int *unlimdimidp)
{
    extern int nc_inq(int ncid, int *ndimsp, int *nvarsp, int *nattsp, int *unlimdimidp);

    return nc_inq(ncid, ndimsp, nvarsp, nattsp, unlimdimidp);
}

extern "C"
int ncmpi_inq_att(
        int ncid,
        int varid,
        const char *name,
        nc_type *xtypep,
        MPI_Offset *lenp)
{
    int rc = NC_NOERR;

    rc = ncmpi_inq_attlen(ncid, varid, name, lenp);
    if (rc != NC_NOERR) {
        log_error(netcdf_debug_level, "%s", ncmpi_strerror(rc));
        return rc;
    }

    rc = ncmpi_inq_atttype(ncid, varid, name, xtypep);
    if (rc != NC_NOERR) {
        log_error(netcdf_debug_level, "%s", ncmpi_strerror(rc));
        return rc;
    }

    return rc;
}

extern "C"
int ncmpi_inq_attlen(
        int ncid,
        int varid,
        const char *name,
        MPI_Offset *lenp)
{
    int rc = NC_NOERR;
    size_t len;

    if (group_map.find(ncid) == group_map.end()) {
        return NC_EBADID;
    }

    /* global attribute */
    rc = group_map[ncid]->inq_attlen(varid, name, &len);
    *lenp = len;

    return rc;
}

extern "C"
int ncmpi_inq_atttype(
        int ncid,
        int varid,
        const char *name,
        nc_type *xtypep)
{
    int rc = NC_NOERR;

    if (group_map.find(ncid) == group_map.end()) {
        return NC_EBADID;
    }

    /* global attribute */
#if USE_NC_TYPE
    rc = group_map[ncid]->inq_atttype(varid, name, xtypep);
#else
    rc = group_map[ncid]->inq_atttype(varid, name, (int *)xtypep);
#endif

    return rc;
}

extern "C"
int ncmpi_inq_attname(
        int ncid,
        int varid,
        int attnum,
        char *name)
{
    extern int nc_inq_attname(int, int, int, char *);

    return nc_inq_attname(ncid, varid, attnum, name);
}


extern "C"
int ncmpi_inq_dim(
        int ncid,
        int dimid,
        char *name,
        MPI_Offset *lenp)
{
    extern int nc_inq_dim(int ncid, int dimid, char *name, size_t *lenp);
    int rc=0;
    size_t nc_len=0;

    rc = nc_inq_dim(ncid, dimid, name, &nc_len);
    *lenp = nc_len;
    return rc;
}

extern "C"
int ncmpi_inq_dimlen(
        int ncid,
        int dimid,
        MPI_Offset *lenp)
{
    extern int nc_inq_dimlen(int ncid, int dimid, size_t *lenp);
    int rc=0;
    size_t nc_len=0;

    rc = nc_inq_dimlen(ncid, dimid, &nc_len);
    *lenp = nc_len;
    return rc;
}

/* The metadata should be stored on every client, so the inquiry functions should
 * not require any network ops.
 */
extern "C"
int ncmpi_inq_var(
                int ncid,
                int varid,
                char *name,
                nc_type *xtypep,
                int *ndimsp,
                int *dimids,
                int *nattsp)
{
    extern int nc_inq_var(int, int, char *, nc_type *, int *, int *, int *);

    return nc_inq_var(ncid, varid, name, xtypep, ndimsp, dimids, nattsp);
}

extern "C"
int ncmpi_inq_varid(
        int ncid,
        const char *name,
        int *varidp)
{
    extern int nc_inq_varid(int, const char *, int *);

    return nc_inq_varid(ncid, name, varidp);
}

extern "C"
int ncmpi_inq_varndims(
        int ncid,
        int varid,
        int *ndimsp)
{
    extern int nc_inq_varndims(int, int, int*);

    return nc_inq_varndims(ncid, varid, ndimsp);
}

extern "C"
int ncmpi_inq_vartype(
                int ncid,
                int varid,
                nc_type *xtypep)
{
    extern int nc_inq_vartype(int, int, nc_type *);

    return nc_inq_vartype(ncid, varid, xtypep);
}

extern "C"
int ncmpi_inq_type(
                int ncid,
                nc_type xtype,
                char *name,
                size_t *sizep)
{
    extern int nc_inq_type(int, nc_type, char *, size_t *);

    return nc_inq_type(ncid, xtype, name, sizep);
}



/* ********************* FLEXIBLE DATA MODE FUNCTIONS **********************/

/**
 * Non-collective function to store an array of variables.
 *
 * This implementation adjusts the count array to include the size of the
 * MPI datatype, then it calls the nc_put_vars function directly.
 *
 * The semantics of this function are slightly different than the real
 * ncmpi_put_vars.  This function sets default values for startp and countp
 * if they are NULL.  The ncmpi spec only allows the stridep array to be null.
 *
 */
extern "C"
int ncmpi_put_vars(
        int ncid,
        int varid,
        const MPI_Offset startp[],
        const MPI_Offset countp[],
        const MPI_Offset stridep[],
        const void *buf,
        MPI_Offset bufcount,
        MPI_Datatype datatype)
{
    int rc = NC_NOERR;
    int mpi_datatype_size;
    size_t varsize;
    nc_type vartype;
    int ndims;
    log_level debug_level = netcdf_debug_level;

    /* prototype for nc_put_vars */
    extern int nc_put_vars(int, int, const size_t *, const size_t *, const ptrdiff_t *, const void *);

    size_t *start_copy = NULL;
    size_t *count_copy = NULL;
    ptrdiff_t *stride_copy = NULL;
    int *dimids = NULL;

    log_debug(debug_level, "Starting ncmpi_put_vars");

    rc = ncmpi_inq_varndims(ncid, varid, &ndims);
    if (rc != NC_NOERR) {
        log_error(debug_level, "Error getting number of dimensions");
        goto cleanup;
    }

    if (ndims) {
        /* Check to see of the MPI datatype size is the same as the nc_type (no conversion yet) */
        rc = MPI_Type_size(datatype, &mpi_datatype_size);
        if (rc != MPI_SUCCESS) {
            log_error(debug_level, "Error getting size of MPI_Datatype");
            goto cleanup;
        }

        /* We need to get the datatype from the metadata so we know how big the buffer should be */
        rc = ncmpi_inq_vartype(ncid, varid, &vartype);
        if (rc != NC_NOERR) {
            log_error(netcdf_debug_level, "Unable to get variable type");
            goto cleanup;
        }

        rc = ncmpi_inq_type(ncid, vartype, NULL, &varsize);
        if (rc != NC_NOERR) {
            log_error(netcdf_debug_level, "Unable to get variable type");
            goto cleanup;
        }

        if (varsize != (size_t)mpi_datatype_size) {
            log_error(netcdf_debug_level, "MPI_Datatype does not match nc_type (no conversion support)");
            rc = NC_EIOMISMATCH;
            goto  cleanup;
        }


        if (startp) {
            start_copy = (size_t *) calloc(ndims, sizeof(size_t));
            copy(startp, startp+ndims, start_copy);
        }

        if (countp) {
            count_copy = (size_t *) calloc(ndims, sizeof(size_t));
            copy(countp, countp+ndims, count_copy);
        }

        if (stridep) {
            stride_copy = (ptrdiff_t *) calloc(ndims, sizeof(ptrdiff_t));
            copy(stridep, stridep + ndims, stride_copy);
        }

        /* call the netcdf version of this function */
        rc = nc_put_vars(ncid, varid, start_copy, count_copy, stride_copy, buf);
    }


cleanup:
        if (start_copy) free(start_copy);
        if (count_copy) free(count_copy);
        if (stride_copy) free(stride_copy);
        if (dimids) free(dimids);
    return rc;
}


/**
 * Write the entire contents of a text variable.
 */
extern "C"
int ncmpi_put_var_text(int ncid, int varid, const char *tp)
{
    log_debug(netcdf_debug_level, "Calling nc_put_var_text");
    return ncmpi_put_vars(ncid, varid, NULL, NULL, NULL, tp, -1, MPI_CHAR);
}

/**
 * Write the entire contents of a text variable.
 */
extern "C"
int ncmpi_put_var_int(int ncid, int varid, const int *ip)
{
    log_debug(netcdf_debug_level, "Calling nc_put_var_int");
    return ncmpi_put_vars(ncid, varid, NULL, NULL, NULL, ip, -1, MPI_INT);
}

extern "C"
int ncmpi_put_var_float(int ncid, int varid, const float *ip)
{
    log_debug(netcdf_debug_level, "Calling nc_put_var_float");
    return ncmpi_put_vars(ncid, varid, NULL, NULL, NULL, ip, -1, MPI_FLOAT);
}

extern "C"
int ncmpi_put_vara(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        const void *data,
        MPI_Offset bufcount,
        MPI_Datatype datatype)
{
    log_debug(netcdf_debug_level, "Calling nc_get_vara_float");
    return ncmpi_put_vars(ncid, varid, startp, countp,
            NULL, data, bufcount, datatype);
}

extern "C"
int ncmpi_put_vara_short(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        const short *data)
{
    log_debug(netcdf_debug_level, "Calling nc_put_vara_short");
    return ncmpi_put_vars(ncid, varid, startp, countp, NULL, data, -1, MPI_SHORT);
}

extern "C"
int ncmpi_put_vara_int(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        const int *data)
{
    log_debug(netcdf_debug_level, "Calling nc_put_vara_int");
    return ncmpi_put_vars(ncid, varid, startp, countp, NULL, data, -1, MPI_INT);
}

extern "C"
int ncmpi_put_vara_float(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        const float *data)
{
    log_debug(netcdf_debug_level, "Calling nc_put_vara_float");
    return ncmpi_put_vars(ncid, varid, startp, countp, NULL, data, -1, MPI_FLOAT);
}

extern "C"
int ncmpi_put_vara_double(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        const double*data)
{
    log_debug(netcdf_debug_level, "Calling nc_put_vara_double");
    return ncmpi_put_vars(ncid, varid, startp, countp, NULL, data, -1, MPI_DOUBLE);
}

/* ********************* INPUT FUNCTIONS **********************/

/**
 * Non-collective function to get an array of variables.
 *
 * This implementation adjusts the count array to include the size of the
 * MPI datatype, then it calls the nc_put_vars function directly.
 *
 * The semantics of this function are slightly different than the real
 * ncmpi_put_vars.  This function sets default values for startp and countp
 * if they are NULL.  The ncmpi spec only allows the stridep array to be null.
 *
 */
extern "C"
int ncmpi_get_vars(
        int ncid,
        int varid,
        const MPI_Offset startp[],
        const MPI_Offset countp[],
        const MPI_Offset stridep[],
        void *buf,
        MPI_Offset bufcount,
        MPI_Datatype datatype)
{
    int rc = NC_NOERR;
    int mpi_datatype_size;
    size_t varsize;
    nc_type vartype;
    int ndims;
    log_level debug_level = netcdf_debug_level;

    /* prototype for nc_put_vars */
    extern int nc_get_vars(int, int, const size_t *, const size_t *, const ptrdiff_t *,  void *);

    size_t *start_copy = NULL;
    size_t *count_copy = NULL;
    ptrdiff_t *stride_copy = NULL;

    rc = ncmpi_inq_varndims(ncid, varid, &ndims);
    if (rc != NC_NOERR) {
        log_error(debug_level, "Error getting number of dimensions");
        goto cleanup;
    }

    if (ndims) {
        /* Check to see of the MPI datatype size is the same as the nc_type (no conversion yet) */
        rc = MPI_Type_size(datatype, &mpi_datatype_size);
        if (rc != MPI_SUCCESS) {
            log_error(debug_level, "Error getting size of MPI_Datatype");
            goto cleanup;
        }

        /* We need to get the datatype from the metadata so we know how big the buffer should be */
        rc = ncmpi_inq_vartype(ncid, varid, &vartype);
        if (rc != NC_NOERR) {
            log_error(netcdf_debug_level, "Unable to get variable type");
            goto cleanup;
        }

        rc = ncmpi_inq_type(ncid, vartype, NULL, &varsize);
        if (rc != NC_NOERR) {
            log_error(netcdf_debug_level, "Unable to get variable type");
            goto cleanup;
        }

        if (varsize != (size_t)mpi_datatype_size) {
            log_error(netcdf_debug_level, "MPI_Datatype does not match nc_type (no conversion support)");
            rc = NC_EIOMISMATCH;
            goto  cleanup;
        }


        if (startp) {
            start_copy = (size_t *) calloc(ndims, sizeof(size_t));
            copy(startp, startp+ndims, start_copy);
        }

        if (countp) {
            count_copy = (size_t *) calloc(ndims, sizeof(size_t));
            copy(countp, countp+ndims, count_copy);
        }

        if (stridep) {
            stride_copy = (ptrdiff_t *) calloc(ndims, sizeof(ptrdiff_t));
            copy(stridep, stridep + ndims, stride_copy);
        }

        /* call the netcdf version of this function */
        rc = nc_get_vars(ncid, varid, start_copy, count_copy, stride_copy, buf);
    }


cleanup:
        if (start_copy) free(start_copy);
        if (count_copy) free(count_copy);
        if (stride_copy) free(stride_copy);
    return rc;
}



/**
 * Get the entire contents of a text variable.
 */
extern "C"
int ncmpi_get_var_text(int ncid, int varid, char *tp)
{
    log_debug(netcdf_debug_level, "Calling nc_put_var_text");
    return ncmpi_get_vars(ncid, varid, NULL, NULL, NULL, tp, -1, MPI_CHAR);
}

/**
 * Get the entire contents of a text variable.
 */
extern "C"
int ncmpi_get_var_int(int ncid, int varid, int *ip)
{
    log_debug(netcdf_debug_level, "Calling nc_get_var_int");
    return ncmpi_get_vars(ncid, varid, NULL, NULL, NULL, ip, -1, MPI_INT);
}

/**
 * Get the entire contents of a text variable.
 */
extern "C"
int ncmpi_get_var_float(int ncid, int varid, float *ip)
{
    log_debug(netcdf_debug_level, "Calling nc_get_var_float");
    return ncmpi_get_vars(ncid, varid, NULL, NULL, NULL, ip, -1, MPI_FLOAT);
}

extern "C"
int ncmpi_get_vara(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        void *data,
        MPI_Offset bufcount,
        MPI_Datatype datatype)
{
    log_debug(netcdf_debug_level, "Calling nc_get_vara_float");
    return ncmpi_get_vars(ncid, varid, startp, countp,
            NULL, data, bufcount, datatype);
}

extern "C"
int ncmpi_get_vara_short(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        short*data)
{
    log_debug(netcdf_debug_level, "Calling nc_get_vara_short");
    return ncmpi_get_vars(ncid, varid, startp, countp, NULL, data, -1, MPI_SHORT);
}

extern "C"
int ncmpi_get_vara_int(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        int*data)
{
    log_debug(netcdf_debug_level, "Calling nc_get_vara_int");
    return ncmpi_get_vars(ncid, varid, startp, countp, NULL, data, -1, MPI_INT);
}

extern "C"
int ncmpi_get_vara_float(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        float *data)
{
    log_debug(netcdf_debug_level, "Calling nc_get_vara_float");
    return ncmpi_get_vars(ncid, varid, startp, countp, NULL, data, -1, MPI_FLOAT);
}

extern "C"
int ncmpi_get_vara_double(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        double*data)
{
    log_debug(netcdf_debug_level, "Calling nc_get_vara_double");
    return ncmpi_get_vars(ncid, varid, startp, countp, NULL, data, -1, MPI_DOUBLE);
}


/* ********************* FLEXIBLE DATA MODE FUNCTIONS **********************/

/**
 * Collective function to store an array of variables.
 *
 * This implementation adjusts the count array to include the size of the
 * MPI datatype, then it calls the nc_put_vars function directly.
 *
 * The semantics of this function are slightly different than the real
 * ncmpi_put_vars.  This function sets default values for startp and countp
 * if they are NULL.  The ncmpi spec only allows the stridep array to be null.
 *
 */
extern "C"
int ncmpi_put_vars_all(
        int ncid,
        int varid,
        const MPI_Offset startp[],
        const MPI_Offset countp[],
        const MPI_Offset stridep[],
        const void *buf,
        MPI_Offset bufcount,
        MPI_Datatype datatype)
{
    int rc = NC_NOERR;
    int mpi_datatype_size;
    size_t varsize;
    nc_type vartype;
    int ndims;
    log_level debug_level = netcdf_debug_level;

    /* prototype for nc_put_vars */
    extern int nc_put_vars(int, int, const size_t *, const size_t *, const ptrdiff_t *, const void *);

    size_t *start_copy = NULL;
    size_t *count_copy = NULL;
    ptrdiff_t *stride_copy = NULL;
    int *dimids = NULL;

    log_debug(debug_level, "Starting ncmpi_put_vars_all");

    rc = ncmpi_inq_varndims(ncid, varid, &ndims);
    if (rc != NC_NOERR) {
        log_error(debug_level, "Error getting number of dimensions");
        goto cleanup;
    }

    if (ndims) {
        /* Check to see of the MPI datatype size is the same as the nc_type (no conversion yet) */
        rc = MPI_Type_size(datatype, &mpi_datatype_size);
        if (rc != MPI_SUCCESS) {
            log_error(debug_level, "Error getting size of MPI_Datatype");
            goto cleanup;
        }

        /* We need to get the datatype from the metadata so we know how big the buffer should be */
        rc = ncmpi_inq_vartype(ncid, varid, &vartype);
        if (rc != NC_NOERR) {
            log_error(netcdf_debug_level, "Unable to get variable type");
            goto cleanup;
        }

        rc = ncmpi_inq_type(ncid, vartype, NULL, &varsize);
        if (rc != NC_NOERR) {
            log_error(netcdf_debug_level, "Unable to get variable type");
            goto cleanup;
        }

        if (varsize != (size_t)mpi_datatype_size) {
            log_error(netcdf_debug_level, "MPI_Datatype does not match nc_type (no conversion support)");
            rc = NC_EIOMISMATCH;
            goto  cleanup;
        }


        if (startp) {
            start_copy = (size_t *) calloc(ndims, sizeof(size_t));
            copy(startp, startp+ndims, start_copy);
        }

        if (countp) {
            count_copy = (size_t *) calloc(ndims, sizeof(size_t));
            copy(countp, countp+ndims, count_copy);
        }

        if (stridep) {
            stride_copy = (ptrdiff_t *) calloc(ndims, sizeof(ptrdiff_t));
            copy(stridep, stridep + ndims, stride_copy);
        }

        /* call the netcdf version of this function */
        rc = nc_put_vars(ncid, varid, start_copy, count_copy, stride_copy, buf);
    }

    MPI_Barrier(comm_map[ncid]);


cleanup:
        if (start_copy) free(start_copy);
        if (count_copy) free(count_copy);
        if (stride_copy) free(stride_copy);
        if (dimids) free(dimids);
    return rc;
}


/**
 * Write the entire contents of a text variable.
 */
extern "C"
int ncmpi_put_var_text_all(int ncid, int varid, const char *tp)
{
    log_debug(netcdf_debug_level, "Calling ncmpi_put_vars_all");
    return ncmpi_put_vars_all(ncid, varid, NULL, NULL, NULL, tp, -1, MPI_CHAR);
}

/**
 * Write the entire contents of a text variable.
 */
extern "C"
int ncmpi_put_var_int_all(int ncid, int varid, const int *ip)
{
    log_debug(netcdf_debug_level, "Calling ncmpi_put_vars_all");
    return ncmpi_put_vars_all(ncid, varid, NULL, NULL, NULL, ip, -1, MPI_INT);
}

extern "C"
int ncmpi_put_var_float_all(int ncid, int varid, const float *ip)
{
    log_debug(netcdf_debug_level, "Calling ncmpi_put_vars_all");
    return ncmpi_put_vars_all(ncid, varid, NULL, NULL, NULL, ip, -1, MPI_FLOAT);
}

extern "C"
int ncmpi_put_vara_all(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        const void *data,
        MPI_Offset bufcount,
        MPI_Datatype datatype)
{
    log_debug(netcdf_debug_level, "Calling ncmpi_get_vars_all");
    return ncmpi_put_vars_all(ncid, varid, startp, countp,
            NULL, data, bufcount, datatype);
}

extern "C"
int ncmpi_put_vara_short_all(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        const short *data)
{
    log_debug(netcdf_debug_level, "Calling ncmpi_put_vars_all");
    return ncmpi_put_vars_all(ncid, varid, startp, countp, NULL, data, -1, MPI_SHORT);
}

extern "C"
int ncmpi_put_vara_int_all(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        const int *data)
{
    log_debug(netcdf_debug_level, "Calling ncmpi_put_vars_all");
    return ncmpi_put_vars_all(ncid, varid, startp, countp, NULL, data, -1, MPI_INT);
}

extern "C"
int ncmpi_put_vara_float_all(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        const float *data)
{
    log_debug(netcdf_debug_level, "Calling ncmpi_put_vars_all");
    return ncmpi_put_vars_all(ncid, varid, startp, countp, NULL, data, -1, MPI_FLOAT);
}

extern "C"
int ncmpi_put_vara_double_all(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        const double*data)
{
    log_debug(netcdf_debug_level, "Calling ncmpi_put_vars_all");
    return ncmpi_put_vars_all(ncid, varid, startp, countp, NULL, data, -1, MPI_DOUBLE);
}

/* ********************* INPUT FUNCTIONS **********************/

/**
 * Collective function to get an array of variables.
 *
 * This implementation adjusts the count array to include the size of the
 * MPI datatype, then it calls the nc_put_vars function directly.
 *
 * The semantics of this function are slightly different than the real
 * ncmpi_put_vars.  This function sets default values for startp and countp
 * if they are NULL.  The ncmpi spec only allows the stridep array to be null.
 *
 */
extern "C"
int ncmpi_get_vars_all(
        int ncid,
        int varid,
        const MPI_Offset startp[],
        const MPI_Offset countp[],
        const MPI_Offset stridep[],
        void *buf,
        MPI_Offset bufcount,
        MPI_Datatype datatype)
{
    int rc = NC_NOERR;
    int mpi_datatype_size;
    size_t varsize;
    nc_type vartype;
    int ndims;
    log_level debug_level = netcdf_debug_level;

    /* prototype for nc_put_vars */
    extern int nc_get_vars(int, int, const size_t *, const size_t *, const ptrdiff_t *,  void *);

    size_t *start_copy = NULL;
    size_t *count_copy = NULL;
    ptrdiff_t *stride_copy = NULL;

    rc = ncmpi_inq_varndims(ncid, varid, &ndims);
    if (rc != NC_NOERR) {
        log_error(debug_level, "Error getting number of dimensions");
        goto cleanup;
    }

    if (ndims) {
        /* Check to see of the MPI datatype size is the same as the nc_type (no conversion yet) */
        rc = MPI_Type_size(datatype, &mpi_datatype_size);
        if (rc != MPI_SUCCESS) {
            log_error(debug_level, "Error getting size of MPI_Datatype");
            goto cleanup;
        }

        /* We need to get the datatype from the metadata so we know how big the buffer should be */
        rc = ncmpi_inq_vartype(ncid, varid, &vartype);
        if (rc != NC_NOERR) {
            log_error(netcdf_debug_level, "Unable to get variable type");
            goto cleanup;
        }

        rc = ncmpi_inq_type(ncid, vartype, NULL, &varsize);
        if (rc != NC_NOERR) {
            log_error(netcdf_debug_level, "Unable to get variable type");
            goto cleanup;
        }

        if (varsize != (size_t)mpi_datatype_size) {
            log_error(netcdf_debug_level, "MPI_Datatype does not match nc_type (no conversion support)");
            rc = NC_EIOMISMATCH;
            goto  cleanup;
        }


        if (startp) {
            start_copy = (size_t *) calloc(ndims, sizeof(size_t));
            copy(startp, startp+ndims, start_copy);
        }

        if (countp) {
            count_copy = (size_t *) calloc(ndims, sizeof(size_t));
            copy(countp, countp+ndims, count_copy);
        }

        if (stridep) {
            stride_copy = (ptrdiff_t *) calloc(ndims, sizeof(ptrdiff_t));
            copy(stridep, stridep + ndims, stride_copy);
        }

        /* call the netcdf version of this function */
        rc = nc_get_vars(ncid, varid, start_copy, count_copy, stride_copy, buf);
    }

    MPI_Barrier(comm_map[ncid]);


cleanup:
        if (start_copy) free(start_copy);
        if (count_copy) free(count_copy);
        if (stride_copy) free(stride_copy);
    return rc;
}



/**
 * Get the entire contents of a text variable.
 */
extern "C"
int ncmpi_get_var_text_all(int ncid, int varid, char *tp)
{
    log_debug(netcdf_debug_level, "Calling ncmpi_put_vars_all");
    return ncmpi_get_vars_all(ncid, varid, NULL, NULL, NULL, tp, -1, MPI_CHAR);
}

/**
 * Get the entire contents of a text variable.
 */
extern "C"
int ncmpi_get_var_int_all(int ncid, int varid, int *ip)
{
    log_debug(netcdf_debug_level, "Calling ncmpi_get_vars_all");
    return ncmpi_get_vars_all(ncid, varid, NULL, NULL, NULL, ip, -1, MPI_INT);
}

/**
 * Get the entire contents of a text variable.
 */
extern "C"
int ncmpi_get_var_float_all(int ncid, int varid, float *ip)
{
    log_debug(netcdf_debug_level, "Calling ncmpi_get_vars_all");
    return ncmpi_get_vars_all(ncid, varid, NULL, NULL, NULL, ip, -1, MPI_FLOAT);
}

extern "C"
int ncmpi_get_vara_all(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        void *data,
        MPI_Offset bufcount,
        MPI_Datatype datatype)
{
    log_debug(netcdf_debug_level, "Calling ncmpi_get_vars_all");
    return ncmpi_get_vars_all(ncid, varid, startp, countp,
            NULL, data, bufcount, datatype);
}

extern "C"
int ncmpi_get_vara_short_all(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        short*data)
{
    log_debug(netcdf_debug_level, "Calling ncmpi_get_vars_all");
    return ncmpi_get_vars_all(ncid, varid, startp, countp, NULL, data, -1, MPI_SHORT);
}

extern "C"
int ncmpi_get_vara_int_all(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        int*data)
{
    log_debug(netcdf_debug_level, "Calling ncmpi_get_vars_all");
    return ncmpi_get_vars_all(ncid, varid, startp, countp, NULL, data, -1, MPI_INT);
}

extern "C"
int ncmpi_get_vara_float_all(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        float *data)
{
    log_debug(netcdf_debug_level, "Calling ncmpi_get_vars_all");
    return ncmpi_get_vars_all(ncid, varid, startp, countp, NULL, data, -1, MPI_FLOAT);
}

extern "C"
int ncmpi_get_vara_double_all(
        int ncid,
        int varid,
        const MPI_Offset *startp,
        const MPI_Offset *countp,
        double*data)
{
    log_debug(netcdf_debug_level, "Calling ncmpi_get_vars_all");
    return ncmpi_get_vars_all(ncid, varid, startp, countp, NULL, data, -1, MPI_DOUBLE);
}


/* *************** DEFINE MODE FUNCTIONS **************** */

extern "C"
int ncmpi_redef(int ncid)
{
    int rc = NC_NOERR;
    int rank;
    MPI_Comm comm = comm_map[ncid];
    extern int nc_redef(int ncid);

    MPI_Comm_rank(comm, &rank);

    /* sync all the processors */
    /* ensure that everyone has completed their data mode ops before we switch to define mode */
    MPI_Barrier(comm);

    rc = nc_redef(ncid);

    MPI_Bcast(&rc, 1, MPI_INT, 0, comm);

    nc_set_file_state(ncid);

    return rc;
}

extern "C"
int ncmpi_enddef(int ncid)
{
    int rc = NC_NOERR;
    int rank;
    MPI_Comm comm = comm_map[ncid];
    trios_declare_timer(callTime);
    extern int nc_enddef(int ncid);

    rc = MPI_Comm_rank(comm, &rank);

    trios_start_timer(callTime);
    rc = nc_enddef(ncid);
    trios_stop_timer("EndDef", callTime);

    MPI_Bcast(&rc, 1, MPI_INT, 0, comm);

    nc_set_file_state(ncid);

    /* sync all the processors */
    /* ensure that we have switched to data mode before everyone starts doing data mode ops */
    /* TODO: is this necessary?  doesn't the MPI_Bcast ensure the no one will proceed until after rank0 calls nc_enddef? */
    trios_start_timer(callTime);
    MPI_Barrier(comm);
    trios_stop_timer("after enddef barrier", callTime);

    return rc;
}


/**
 * Sync an open netCDF dataset.
 */
extern "C"
int ncmpi_sync(int ncid)
{
    int rc = NC_NOERR;
    int rank;
    MPI_Comm comm = comm_map[ncid];
    extern int nc_sync(int ncid);
    log_level debug_level = netcdf_debug_level;

    log_debug(debug_level, "Calling nc_sync");

    MPI_Comm_rank(comm, &rank);

    /* sync all the processors */
    /* ensure that everyone has completed their outstanding ops before we sync */
    log_debug(netcdf_debug_level, "sync barrier");
    MPI_Barrier(comm);

    log_debug(netcdf_debug_level, "syncing");
    rc = nc_sync(ncid);

    log_debug(netcdf_debug_level, "sync bcast");
    MPI_Bcast(&rc, 1, MPI_INT, 0, comm);

    return rc;
}

extern "C"
int ncmpi_sync_wait(int ncid)
{
    int rc = NC_NOERR;
    int rank;
    MPI_Comm comm = comm_map[ncid];
    log_level debug_level = netcdf_debug_level;
    trios_declare_timer(SyncTime);

    log_debug(debug_level, "Calling nc_sync_wait");

    MPI_Comm_rank(comm, &rank);

    trios_start_timer(SyncTime);
    log_debug(netcdf_debug_level, "sync waiting");
    rc = nc_sync_wait(ncid);
    trios_stop_timer("SyncWait nc_sync_wait()", SyncTime);

    trios_start_timer(SyncTime);
    log_debug(netcdf_debug_level, "sync bcast");
    MPI_Bcast(&rc, 1, MPI_INT, 0, comm);
    trios_stop_timer("SyncWait bcast()", SyncTime);

    return rc;
}

/**
 * Close an open netCDF dataset.
 */
extern "C"
int ncmpi_close(int ncid)
{
    int rc = NC_NOERR;
    int rank;
    MPI_Comm comm = comm_map[ncid];
    extern int nc_close(int ncid);
    log_level debug_level = netcdf_debug_level;
    trios_declare_timer(CloseTime);

    log_debug(debug_level, "Calling nc_close");

    MPI_Comm_rank(comm, &rank);

    /* sync all the processors */
    /* ensure that everyone has completed their outstanding ops before we close */
    trios_start_timer(CloseTime);
    log_debug(netcdf_debug_level, "close barrier");
    MPI_Barrier(comm);
    trios_stop_timer("Close Barrier", CloseTime);

    trios_start_timer(CloseTime);
    log_debug(netcdf_debug_level, "closing");
    rc = nc_close(ncid);
    trios_stop_timer("Close nc_close()", CloseTime);

    trios_start_timer(CloseTime);
    log_debug(netcdf_debug_level, "close bcast");
    MPI_Bcast(&rc, 1, MPI_INT, 0, comm);
    trios_stop_timer("Close bcast()", CloseTime);

    nc_set_file_state(ncid);

    return rc;
}

extern "C"
int ncmpi_close_wait(int ncid)
{
    int rc = NC_NOERR;
    int rank;
    MPI_Comm comm = comm_map[ncid];
    log_level debug_level = netcdf_debug_level;
    trios_declare_timer(CloseTime);

    log_debug(debug_level, "Calling nc_close_wait");

    MPI_Comm_rank(comm, &rank);

    trios_start_timer(CloseTime);
    log_debug(netcdf_debug_level, "close waiting");
    rc = nc_close_wait(ncid);
    trios_stop_timer("CloseWait nc_close_wait()", CloseTime);

    trios_start_timer(CloseTime);
    log_debug(netcdf_debug_level, "close bcast");
    MPI_Bcast(&rc, 1, MPI_INT, 0, comm);
    trios_stop_timer("CloseWait bcast()", CloseTime);

    trios_start_timer(CloseTime);
    /* remove data structures associated with ncid */
    if (group_map.find(ncid) != group_map.end()) {
        delete group_map[ncid];
        group_map.erase(ncid);
    }
    trios_stop_timer("group_map.erase()", CloseTime);

    trios_start_timer(CloseTime);
    nc_set_file_state(ncid);
    trios_stop_timer("nc_set_file_state()", CloseTime);

    return rc;
}

extern "C"
int ncmpi_begin_indep_data(int ncid)
{
    int rc = NC_NOERR;

    rc = nc_begin_indep_data(ncid);

    nc_set_file_state(ncid);

    return rc;
}

extern "C"
int ncmpi_end_indep_data(int ncid)
{
    int rc = NC_NOERR;

    rc = nc_end_indep_data(ncid);

    nc_set_file_state(ncid);

    return rc;
}
