/*  display.c
 *
 *  Display functions for xdemorse application
 */

/*
 *  xdemorse: An application to decode Morse code signals to text
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#include "display.h"
#include "shared.h"

/*------------------------------------------------------------------------*/

/*  Display_Detector()
 *
 *  Updates the Detector scope display
 */

  void
Display_Detector( int plot )
{
  /* Points to plot */
  static GdkPoint *points = NULL;
  static int points_idx = 0;
  static cairo_t *cr = NULL;

  /* Initialize on first call */
  if( points == NULL )
  {
	if( !mem_alloc( (void *)&points,
		  (size_t)gbl_wfall_width * sizeof(GdkPoint)) )
	  return;
  }

  /* Initialize cairo */
  if( cr == NULL ) cr = gdk_cairo_create( gbl_scope->window );

  /* Save values to be plotted  */
  points[points_idx].y = gbl_scope_height - plot - 1;
  if( points[points_idx].y <= 0 )
	points[points_idx].y = 1;
  if( points[points_idx].y >= gbl_scope_height )
	points[points_idx].y = gbl_scope_height - 1;

  points[points_idx].x = points_idx;

  /* Recycle buffer idx when full and plot */
  points_idx++;
  if( points_idx == gbl_scope_width )
  {
	int idx;

	/* Draw scope background */
	cairo_set_source_rgb( cr, 0.0, .3, 0.0 );
	cairo_rectangle(
		cr, 0.0, 0.0,
		(double)gbl_scope_width,
		(double)gbl_scope_height );
	cairo_fill( cr );

	/* Plot signal graph */
	cairo_set_source_rgb( cr, 0.0, 1.0, 0.0 );
	cairo_move_to( cr, (double)points[0].x, (double)points[0].y );
	for( idx = 1; idx < points_idx; idx++ )
	  cairo_line_to( cr, (double)points[idx].x, (double)points[idx].y );

	/* Stroke paths */
	cairo_stroke( cr );

	/* Draw Mark/Space threshold references */
	cairo_set_source_rgb( cr, 1.0, 1.0, 1.0 );
	int temp = gbl_scope_height - rc_data.det_threshold;
	cairo_move_to( cr, 0.0, (double)temp );
	cairo_line_to( cr, (double)gbl_scope_width, (double)temp );

	/* Stroke paths */
	cairo_stroke( cr );

	points_idx = 0;
  } /* if( points_idx == gbl_scope_width ) */

} /* Display_Detector() */

/*------------------------------------------------------------------------*/

/*  Display_Signal()
 *
 *  Updates the Signal scope display
 */

  void
Display_Signal( int plot )
{
  /* Points to plot */
  static GdkPoint *points = NULL;
  static int points_idx = 0;
  static cairo_t *cr = NULL;

  /* Initialize on first call */
  if( points == NULL )
  {
	if( !mem_alloc( (void *)&points,
		  (size_t)gbl_wfall_width * sizeof(GdkPoint)) )
	  return;
  }

  /* Initialize cairo */
  if( cr == NULL )
	cr = gdk_cairo_create( gbl_scope->window );

  /* Save values to be plotted (scaled to fit display) */
  points[points_idx].y = gbl_scope_height - plot - 1;
  if( points[points_idx].y <= 0 )
	points[points_idx].y = 1;
  if( points[points_idx].y >= gbl_scope_height )
	points[points_idx].y = gbl_scope_height - 1;

  points[points_idx].x = points_idx;

  /* Recycle buffer idx when full and plot */
  points_idx++;
  if( points_idx == gbl_scope_width )
  {
	int idx;

	/* Draw scope background */
	cairo_set_source_rgb( cr, 0.0, .3, 0.0 );
	cairo_rectangle(
		cr, 0.0, 0.0,
		(double)gbl_scope_width,
		(double)gbl_scope_height );
	cairo_fill( cr );

	/* Plot signal graph */
	cairo_set_source_rgb( cr, 0.0, 1.0, 0.0 );
	cairo_move_to( cr, (double)points[0].x, (double)points[0].y );
	for( idx = 1; idx < points_idx; idx++ )
	  cairo_line_to( cr, (double)points[idx].x, (double)points[idx].y );

	/* Stroke paths */
	cairo_stroke( cr );

	points_idx = 0;
  } /* if( points_idx == DRAWABLE_LEN ) */

} /* Display_Signal */

/*------------------------------------------------------------------------*/

/* Display_Waterfall()
 *
 * Displays audio spectrum as "waterfall"
 */

  void
Display_Waterfall( void )
{
  int
	idh, idv,  /* Index to hor. and vert. position in warterfall */
	pixel_val, /* Greyscale value of pixel derived from fft o/p  */
	fft_idx;   /* Index to fft output array */

  /* Constant needed to draw center line in waterfall */
  int center_line = gbl_wfall_width / 2;

  int
	bin_val, /* Value of fft output "bin" */
	bin_max; /* Maximum value of fft bins */

  /* Sliding window average of bin max values */
  static int ave_max = 10000;

  /* Pointer to current pixel */
  static guchar *pix;

  /* Draw a vertical white line in waterfall at detector's freq. */
  if( isFlagSet(CENTERLINE) )
  {
	pix = gbl_wfall_pixels + gbl_wfall_rowstride + gbl_wfall_n_channels;
	pix += gbl_wfall_n_channels * center_line;
	pix[0] = pix[1] = pix[2] = 0xff;
  }

  /* Copy each line of waterfall to next one */
  for( idv = gbl_wfall_height-2; idv > 0; idv-- )
  {
	pix = gbl_wfall_pixels + gbl_wfall_rowstride * idv + gbl_wfall_n_channels;

	for( idh = 0; idh < gbl_wfall_width; idh++ )
	{
	  pix[0] = pix[ -gbl_wfall_rowstride];
	  pix[1] = pix[1-gbl_wfall_rowstride];
	  pix[2] = pix[2-gbl_wfall_rowstride];
	  pix += gbl_wfall_n_channels;
	}
  }

  /* Go to top left +1 hor. +1 vert. of pixbuf */
  pix = gbl_wfall_pixels + gbl_wfall_rowstride + gbl_wfall_n_channels;

  /* First column of display (DC) not used */
  gbl_bin_ave[0] = 0;

  /* Do the FFT on input array */
  Ifft( FFT_INPUT_SIZE, gbl_wfall_width );
  bin_max = 0;
  for( fft_idx = 0; fft_idx < gbl_wfall_width; fft_idx++ )
  {
	/* Calculate signal power at each frequency (bin) */
	fft_out_r[fft_idx] /= 1024;
	fft_out_i[fft_idx] /= 1024;
	bin_val = 
	  fft_out_r[fft_idx] * fft_out_r[fft_idx] +
	  fft_out_i[fft_idx] * fft_out_i[fft_idx];

	/* Calculate average bin values */
	gbl_bin_ave[fft_idx] =
	  ( (gbl_bin_ave[fft_idx] * (BINMAX_AVE_WIN-1) + bin_val) / BINMAX_AVE_WIN );

	/* Record max bin value */
	if( bin_max < bin_val )
	  bin_max = bin_val;

	/* Scale pixel values to 255 depending on ave max value */
	pixel_val = (255 * bin_val) / ave_max;
	if( pixel_val > 255 ) pixel_val = 255;

	/* Color code signal strength */
	int n;
	if( pixel_val < 85 )
	{
	  pix[0] = 0;
	  pix[1] = 0;
	  pix[2] = 3 + (guchar)pixel_val * 3;
	}
	else if( pixel_val < 170 )
	{
	  n = pixel_val - 85;
	  pix[0] = 0;
	  pix[1] = 3 + (guchar)n * 3;
	  pix[2] = 255 - (guchar)n * 3;
	}
	else
	{
	  n = pixel_val - 170;
	  pix[0] = (guchar)pixel_val;
	  pix[1] = 255 - (guchar)n * 3;
	  pix[2] = 0;
	}

	pix += gbl_wfall_n_channels;

  } /* for( fft_idx = 0; fft_idx < fft_end; fft_idx++ ) */

  /* Calculate sliding window average of max bin value */
  ave_max =
	( ave_max * (BINMAX_AVE_WIN - 1) + bin_max) / BINMAX_AVE_WIN;
  if( !ave_max ) ave_max = 1;

  /* At last draw waterfall */
  gtk_widget_queue_draw( gbl_waterfall );

} /* Display_Waterfall() */

/*------------------------------------------------------------------------*/

