#!/usr/bin/env python

import nsysstats

class VulkanGpuMarkerSummary(nsysstats.Report):

    usage = f"""{{SCRIPT}} -- Vulkan GPU Range Summary

    No arguments.

    Output: All time values default to nanoseconds
        Time(%) : Percentage of "Total Time"
        Total Time : The total time used by all instances of this range
        Instances: The number of instances of this range
        Average : The average execution time of this range
        Minimum : The smallest execution time of this range
        Maximum : The largest execution time of this range
        StdDev : The standard deviation of execution time of this range
        Range : The name of the range

    This report provides a summary of Vulkan GPU debug markers,
    and their execution times. Note that the "Time(%)" column
    is calculated using a summation of the "Total Time" column, and represents
    that range's percent of the execution time of the ranges listed, and not a
    percentage of the application wall or CPU execution time.
"""

    query = f"""
WITH
    cmdDebugUtilsLabelEXTRanges AS (
        SELECT
            *
        FROM
            VULKAN_WORKLOAD
        WHERE textId IS NOT NULL
    ),
    summary AS (
        SELECT
            cmdDebugUtilsLabelEXTRanges.textId AS textId,
            sum(cmdDebugUtilsLabelEXTRanges.end - cmdDebugUtilsLabelEXTRanges.start) AS total,
            count(*) AS num,
            avg(cmdDebugUtilsLabelEXTRanges.end - cmdDebugUtilsLabelEXTRanges.start) AS avg,
            min(cmdDebugUtilsLabelEXTRanges.end - cmdDebugUtilsLabelEXTRanges.start) AS min,
            max(cmdDebugUtilsLabelEXTRanges.end - cmdDebugUtilsLabelEXTRanges.start) AS max,
            stdev(cmdDebugUtilsLabelEXTRanges.end - cmdDebugUtilsLabelEXTRanges.start) AS stddev
        FROM
            cmdDebugUtilsLabelEXTRanges
        GROUP BY 1
    ),
    totals AS (
        SELECT sum(total) AS total
        FROM summary
    )

SELECT
    round(summary.total * 100.0 / (SELECT total FROM totals), 1) AS "Time(%)",
    summary.total AS "Total Time:dur_ns",
    summary.num AS "Instances",
    round(summary.avg, 1) AS "Average:dur_ns",
    summary.min AS "Minimum:dur_ns",
    summary.max AS "Maximum:dur_ns",
    round(summary.stddev, 1) AS "StdDev:dur_ns",
    ids.value AS "Range"
FROM
    summary
LEFT JOIN
    StringIds AS ids
    ON ids.id == summary.textId
WHERE summary.total > 0
ORDER BY 2 DESC
;
"""

    table_checks = {
        'VULKAN_WORKLOAD':
            "{DBFILE} does not contain GPU Vulkan Debug Extension (GPU Vulkan Debug markers) data"
    }

if __name__ == "__main__":
    VulkanGpuMarkerSummary.Main()
