/* $Id: kmo_std_star-test.c,v 1.28 2013-10-21 13:44:55 aagudo Exp $
 *
 * This file is part of the KMOS Library
 * Copyright (C) 2002-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: aagudo $
 * $Date: 2013-10-21 13:44:55 $
 * $Revision: 1.28 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <sys/stat.h>
#include <math.h>

#include <cpl.h>

#include "kmclipm_functions.h"

#include "kmo_dfs.h"

const char  *test_global_path_test_data;
float   test_global_seed_data, test_global_seed_noise;
int     test_global_size_x, test_global_size_y, test_global_size_z, test_global_nr_frames;
void kmo_get_pipe_command(char*, const char*, const char*, int);
int kmo_test_file_exists(const char*);
void kmo_test_verbose_off();
void kmo_test_verbose_on();
double kmo_test_esorex_data(const char*, int);
const char* kmo_test_cat_strings(char*, const char*, const char*);
int  kmo_test_create_RAW_data(int, int, const char*, char**, char**, int*, int, char**, char**, int*, int, float, float);
int  kmo_test_create_RAW_data_zero(int, int, const char*);
int  kmo_test_create_F2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2D_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data_noise(int, int, int, const char*, char *, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_infinite(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data2(int, int, int, int, int*, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_noise(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_B2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);

const char      *path_recipe       = "kmo_std_star/";

const char      *valid_files[]     = {"std_obj_123.fits",
                                      "std_sky_123.fits",
                                      "kmos_solar_k_1700.fits",
                                      "kmos_atmos_k.fits",
                                      "kmos_spec_type.fits",
                                      "kmos_wave_band.fits"};

/**
    @defgroup kmo_std_star_test  kmo_std_star_test unit tests

    @{
*/

/**
  @brief   test
 */
static void test_std_star(const char *star_path, const char *sky_path,
                          const char *solar_path, const char *atmos_path,
                          const char *spec_path, int ret_val)
{
    // create sof-file
    const char *sof_path = "test_std_star.sof";
    FILE *fh = fopen(sof_path, "w");
    char *my_path = cpl_sprintf("%s/ref_data", getenv("srcdir"));
    fprintf (fh,
             "%s                          STD\n"
             "%s                          STD\n"
             "xcal_KKK.fits               XCAL\n"
             "ycal_KKK.fits               YCAL\n"
             "lcal_KKK.fits               LCAL\n"
             "master_flat_KKK.fits        MASTER_FLAT\n"
             "illum_corr_KKK.fits         ILLUM_CORR\n"
             "%s/%s             WAVE_BAND\n",
             star_path, sky_path, my_path, valid_files[5]);
    if (solar_path != NULL) {
        fprintf(fh,
                "%s/%s              SOLAR_SPEC\n", my_path, solar_path);
    }
    if (atmos_path != NULL) {
        fprintf(fh,
                "%s/%s              ATMOS_MODEL\n", my_path, atmos_path);
    }
    if (spec_path != NULL) {
        fprintf(fh,
                "%s/%s              SPEC_TYPE_LOOKUP\n", my_path, spec_path);
    }
    fclose(fh);
    cpl_free(my_path);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_std_star  --startype=\"A0I\" "
            "--imethod=NN --magnitude=8 --b_start=1.948 --b_end=2.351 %s", sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_std_star.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

/**
  @brief    Test of kmo_std_star recipe.
  @param    argc   the number of parameters
  @param    argv   the parameter list

  Call @c kmo_std_star-test @c --generate when only the test data for this
  recipe should be generated.
  Call @c kmo_std_star-test @c --no-gen when only the tests should be
  executed and no data should be generated.

 */
int main(int argc, char *argv[])
{
    float   tol = 0.01;

    char    test_path[256],
            file_path[256],
            op_file1[256],
            op_file2[256],
            op_file3[256],
            op_file4[256],
            op_file5[256];

    cpl_propertylist *h = NULL;

    cpl_test_init("<kmos-spark@mpe.mpg.de>", CPL_MSG_WARNING);

    strcpy(test_path, test_global_path_test_data);
    strcat(test_path, path_recipe);

    if (kmo_test_file_exists(test_path) == FALSE) {
        mkdir(test_path, 0777);
    }

    // get rid of warning...
    if ((argc == 5) && (strcmp(argv[1], "xxx") != 0)) {
    }

    FILE *fd = fopen("log_kmo_std_star.txt", "w");
    fprintf(fd, "%s\n", " ");
    fclose(fd);

    //
    //  PIPELINE TEST (must be called after kmo_illumination-test, since the
    //  following recipes rely on this output)
    //
    cpl_msg_info(cpl_func, "Testing kmo_std_star with real pipeline data.");
    sprintf(test_path, "%s%s", test_global_path_test_data, "pipeline/");

    kmo_test_cat_strings(op_file1, test_path, valid_files[0]);
    kmo_test_cat_strings(op_file2, test_path, valid_files[1]);
    strcpy(op_file3, valid_files[2]);
    strcpy(op_file4, valid_files[3]);
    strcpy(op_file5, valid_files[4]);
    test_std_star(op_file1, op_file2, op_file3, op_file4, op_file5, 0);

    // check std_image_KKK.fits
    strcpy(file_path, "std_image_KKK");
    cpl_test_abs(kmo_test_esorex_data(file_path, 1), 10943.4, tol*10500);
    strcat(file_path, ".fits");

    h = kmclipm_propertylist_load(file_path, 0);
    cpl_test_eq(cpl_propertylist_get_int(h, QC_NR_STD_STARS), 6);
    cpl_test_eq(cpl_propertylist_has(h, QC_STD_TRACE), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_THROUGHPUT_SDV), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_THROUGHPUT_MEAN), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_THROUGHPUT), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_SPAT_RES), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_ZEROPOINT), 0);

    cpl_propertylist_delete(h); h = NULL;

    h = kmclipm_propertylist_load(file_path, 2);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_SPAT_RES), 0.544499, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_STD_TRACE), 0.906642, tol);
    cpl_test_eq(cpl_propertylist_has(h, QC_NR_STD_STARS), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_THROUGHPUT_SDV), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_THROUGHPUT_MEAN), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_THROUGHPUT), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_ZEROPOINT), 0);

    cpl_propertylist_delete(h); h = NULL;

    // check telluric_KKK.fits
    strcpy(file_path, "telluric_KKK");
    cpl_test_abs(kmo_test_esorex_data(file_path, 1), 5.61881, .3);
    strcat(file_path, ".fits");

    h = kmclipm_propertylist_load(file_path, 0);
    cpl_test_eq(cpl_propertylist_get_int(h, QC_NR_STD_STARS), 6);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_THROUGHPUT_MEAN), 0.0228217, tol*2);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_THROUGHPUT_SDV), 0.00958603, tol*1.5);
    cpl_test_eq(cpl_propertylist_has(h, QC_STD_TRACE), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_THROUGHPUT), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_SPAT_RES), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_ZEROPOINT), 0);
    cpl_propertylist_delete(h); h = NULL;

    h = kmclipm_propertylist_load(file_path, 3);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_THROUGHPUT), 0.0424846, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_ZEROPOINT), 15.657, tol*2);
    cpl_test_eq(cpl_propertylist_has(h, QC_NR_STD_STARS), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_STD_TRACE), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_THROUGHPUT_SDV), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_THROUGHPUT_MEAN), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_SPAT_RES), 0);
    cpl_propertylist_delete(h); h = NULL;

    // check star_spec_KKK.fits
    strcpy(file_path, "star_spec_KKK");
    cpl_test_abs(kmo_test_esorex_data(file_path, 1), 2.10111e6, tol*4e6);
    strcat(file_path, ".fits");

    h = kmclipm_propertylist_load(file_path, 0);
    cpl_test_eq(cpl_propertylist_get_int(h, QC_NR_STD_STARS), 6);
    cpl_test_eq(cpl_propertylist_has(h, QC_STD_TRACE), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_THROUGHPUT_SDV), 1);
    cpl_test_eq(cpl_propertylist_has(h, QC_THROUGHPUT_MEAN), 1);
    cpl_test_eq(cpl_propertylist_has(h, QC_THROUGHPUT), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_SPAT_RES), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_ZEROPOINT), 0);

    cpl_propertylist_delete(h); h = NULL;

    h = kmclipm_propertylist_load(file_path, 3);
    cpl_test_eq(cpl_propertylist_has(h, QC_NR_STD_STARS), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_STD_TRACE), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_THROUGHPUT_SDV), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_THROUGHPUT_MEAN), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_THROUGHPUT), 1);
    cpl_test_eq(cpl_propertylist_has(h, QC_SPAT_RES), 0);
    cpl_test_eq(cpl_propertylist_has(h, QC_ZEROPOINT), 1);
    cpl_propertylist_delete(h); h = NULL;

    return cpl_test_end(0);
}

/** @} */
