/*
 *  Copyright (c) 2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */
#ifndef _TEMPLATE_AST_H_
#define _TEMPLATE_AST_H_

#include <GTLCore/String.h>

namespace GTLCore {
  class Type;
}

namespace OpenCTL {
  class TemplateGenerationContext;
  namespace TemplateAST {
    /**
     * @internal
     * @ingroup OpenCTL
     * This class represents a node in the AST tree of CTL Templates.
     */
    class Node {
      public:
        virtual ~Node();
        virtual void generate(TemplateGenerationContext* _context) = 0;
      protected:
        /**
         * Utility function that returns the string corresponding to the type.
         */
        static GTLCore::String typeToString(const GTLCore::Type* _type);
        static GTLCore::String typeMax(const GTLCore::Type* _type);
        static GTLCore::String typeMin(const GTLCore::Type* _type);
        static GTLCore::String typeUnit(const GTLCore::Type* _type);
    };
    /**
     * @internal
     * @ingroup OpenCTL
     *
     * Represent a list of \p Node in the AST.
     */
    class NodesList : public Node {
      public:
        NodesList( const std::list<Node*>& _nodes);
        virtual ~NodesList();
        virtual void generate(TemplateGenerationContext* _context);
      private:
        std::list<Node* > m_nodes;
    };
    /**
     * @internal
     * @ingroup OpenCTL
     *
     * Represent a text entry in the AST.
     */
    class TextNode : public Node  {
      public:
        TextNode(const GTLCore::String& _text);
        virtual ~TextNode();
        virtual void generate(TemplateGenerationContext* _context);
      private:
        GTLCore::String m_text;
    };
    /**
     * @internal
     * @ingroup OpenCTL
     *
     * Represent an operation in the AST.
     */
    class OperationNode : public Node {
      public:
        OperationNode( const GTLCore::String& _name, unsigned int inputs, const GTLCore::String& _arg, const GTLCore::String& _optArg);
        virtual ~OperationNode();
        virtual void generate(TemplateGenerationContext* _context);
      private:
        GTLCore::String m_name;
        int m_inputs;
        GTLCore::String m_arg, m_optArg;
    };
    /**
     * @internal
     * @ingroup OpenCTL
     *
     * Represent a channel operand.
     */
    class AllChannelsNode : public Node {
      public:
        enum WhichChannel {
          ColorChannel = 1, ///< operand only on color channel
          AlphaChannel = 2, ///< operand only on alpha channel
          AllChannel = ColorChannel | AlphaChannel ///< operand on color and alpha channel
        };
      public:
        AllChannelsNode( NodesList* _nodesList, WhichChannel _whichChannel );
        virtual ~AllChannelsNode();
        virtual void generate(TemplateGenerationContext* _context);
      private:
        NodesList* m_nodesList;
        WhichChannel m_whichChannel;
    };
    /**
     * @internal
     * @ingroup OpenCTL
     *
     * Represent an alpha channel.
     */
    class AlphaNode : public Node {
      public:
        virtual void generate(TemplateGenerationContext* _context);
    };
    /**
     * @internal
     * @ingroup OpenCTL
     *
     * Maximum value of the alpha channel.
     */
    class AlphaMaxNode : public Node {
      public:
        virtual void generate(TemplateGenerationContext* _context);
    };
    /**
     * @internal
     * @ingroup OpenCTL
     *
     * Minimum value of the alpha channel.
     */
    class AlphaMinNode : public Node {
      public:
        virtual void generate(TemplateGenerationContext* _context);
    };
    /**
     * @internal
     * @ingroup OpenCTL
     *
     * Unit of alpha channel (aka 1.0 for float, 255 for int8...).
     */
    class AlphaUnitNode : public Node {
      public:
        virtual void generate(TemplateGenerationContext* _context);
    };
    /**
     * @internal
     * @ingroup OpenCTL
     * 
     * Represent an input or output node.
     */
    class InOutNode : public Node {
      public:
        enum Which {
          In,
          Out
        };
        enum Quantity {
          One,
          All
        };
      public:
        InOutNode(Which _which, Quantity _quantity, unsigned int idx = 0);
      public:
        virtual void generate(TemplateGenerationContext* _context);
      private:
        GTLCore::String m_string;
        GTLCore::String m_idx;
        Quantity m_quantity;
    };
    /**
     * @internal
     * @ingroup OpenCTL
     *
     * represent a variable in the AST.
     */
    class VarNode : public Node {
      public:
        VarNode(const GTLCore::String& _name );
      public:
        virtual void generate(TemplateGenerationContext* _context);
      private:
        GTLCore::String m_name;
    };
    /**
     * @internal
     * @ingroup OpenCTL
     *
     * Maximum value of the current channel.
     */
    class ChannelMaxNode : public Node {
      public:
        virtual void generate(TemplateGenerationContext* _context);
    };
    /**
     * @internal
     * @ingroup OpenCTL
     *
     * Minimum value of the current channel.
     */
    class ChannelMinNode : public Node {
      public:
        virtual void generate(TemplateGenerationContext* _context);
    };
    /**
     * @internal
     * @ingroup OpenCTL
     *
     * Unit value of the current channel.
     */
    class ChannelUnitNode : public Node {
      public:
        virtual void generate(TemplateGenerationContext* _context);
    };
    /**
     * @internal
     * @ingroup OpenCTL
     *
     * Type of the current channel.
     */
    class ChannelTypeNode : public Node {
      public:
        virtual void generate(TemplateGenerationContext* _context);
    };
    /**
     * @internal
     * @ingroup OpenCTL
     *
     * Maximum value of the named type.
     */
    class NamedMaxNode : public Node {
      public:
        NamedMaxNode(const GTLCore::String& _name);
        virtual void generate(TemplateGenerationContext* _context);
      private:
        GTLCore::String m_name;
    };
    /**
     * @internal
     * @ingroup OpenCTL
     *
     * Minimum value of the named type.
     */
    class NamedMinNode : public Node {
      public:
        NamedMinNode(const GTLCore::String& _name);
        virtual void generate(TemplateGenerationContext* _context);
      private:
        GTLCore::String m_name;
    };
    /**
     * @internal
     * @ingroup OpenCTL
     *
     * Unit value of the named type.
     */
    class NamedUnitNode : public Node {
      public:
        NamedUnitNode(const GTLCore::String& _name);
        virtual void generate(TemplateGenerationContext* _context);
      private:
        GTLCore::String m_name;
    };
    /**
     * @internal
     * @ingroup OpenCTL
     *
     * Type of the named type.
     */
    class NamedTypeNode : public Node {
      public:
        NamedTypeNode(const GTLCore::String& _name);
        virtual void generate(TemplateGenerationContext* _context);
      private:
        GTLCore::String m_name;
    };
  }
}

#endif
